package de.fhdw.wtf.context.core;

import de.fhdw.wtf.context.exception.FrameworkException;
import de.fhdw.wtf.persistence.facade.DatabaseManager;
import de.fhdw.wtf.persistence.facade.OracleAuthorizationFacadeImplementation;
import de.fhdw.wtf.persistence.facade.OracleDatabaseManager;

/**
 * A class to represent a container for central important application facades, which interact with each other.
 */
public final class ApplicationContainer {
	
	/**
	 * The static instance variable to realize a singleton.
	 */
	private static ApplicationContainer instance = null;
	
	/**
	 * Provides the instance of the Application Container in the application.
	 * 
	 * @return The only instance of the Application Container in the application.
	 */
	public static synchronized ApplicationContainer getInstance() {
		if (instance == null) {
			instance = new ApplicationContainer();
		}
		return instance;
	}
	
	/**
	 * Private default constructor.
	 */
	private ApplicationContainer() {
	}
	
	/**
	 * A string with the name of the DatabaseManagerFactory that has been used during initialization. It thus implicates
	 * the currently used database Provider.
	 */
	private String usedDatabaseManagerFactoryName = null;
	
	/**
	 * A String representing the name of the application.
	 */
	private String appName;
	
	/**
	 * Provides the Application name.
	 * 
	 * @return A String representing the application name
	 */
	public String getAppName() {
		return this.appName;
	}
	
	/**
	 * Sets the application name, should only be called by the Application Starter.
	 * 
	 * @param appName
	 *            The name of the application.
	 */
	void setAppName(final String appName) {
		this.appName = appName;
	}
	
	/**
	 * 
	 * @return a string with the name of the DatabaseManagerFactory that has been used during initialization. It thus
	 *         implicates the currently used database Provider.
	 */
	public String getUsedDatabaseManagerFactoryName() {
		return this.usedDatabaseManagerFactoryName;
	}
	
	/**
	 * Sets the name of the DatabaseManagerFactory used during initialization. Should only be called from the
	 * Application Starter.
	 * 
	 * @param usedDatabaseManagerFactoryName
	 *            a string with the name of the DatabaseManagerFactory that has been used during initialization. It thus
	 *            implicates the currently used database Provider.
	 */
	void setUsedDatabaseManagerFactoryName(final String usedDatabaseManagerFactoryName) {
		this.usedDatabaseManagerFactoryName = usedDatabaseManagerFactoryName;
	}
	
	/**
	 * The variable to hold the Database Manager, which holds the class and object facade.
	 */
	private DatabaseManager databaseManager;
	
	/**
	 * The central logging instance, which protocols the application behavior.
	 */
	private Logger logger;
	
	/**
	 * The Provider for the object factory Provider, which holds all the object factories which instantiate the
	 * different user defined types.
	 */
	private ObjectFactoryProvider factoryProvider;
	
	/**
	 * Provides the Object Factory Provider, which holds all Object Factories, which are used to provide an instance for
	 * a given Type Id.
	 * 
	 * @return Provides the instance of the Object Factory Provider.
	 */
	public ObjectFactoryProvider getFactoryProvider() {
		return this.factoryProvider;
	}
	
	/**
	 * Sets the Object Factory Provider, should only be called from the Application Starter.
	 * 
	 * @param factoryProvider
	 *            The Object Factory Provider.
	 */
	void setFactoryProvider(final ObjectFactoryProvider factoryProvider) {
		this.factoryProvider = factoryProvider;
	}
	
	/**
	 * Provides the Database Manager which controls the connection with the Database and holds the class and
	 * objectfacade for the persistence layer.
	 * 
	 * @return The Database
	 */
	public DatabaseManager getDatabaseManager() {
		return this.databaseManager;
	}
	
	/**
	 * Sets the Database Manger, should only be called from the Application Starter.
	 * 
	 * @param databaseManager
	 *            The new database manager.
	 */
	void setDatabaseManager(final DatabaseManager databaseManager) {
		this.databaseManager = databaseManager;
	}
	
	/**
	 * Provides the central logging instance, which protocols the application behavior.
	 * 
	 * @return The Logging component in the application.
	 */
	public Logger getLogger() {
		return this.logger;
	}
	
	/**
	 * Sets the central logger of the application, should only be called from the Application Starter.
	 * 
	 * @param logger
	 *            The logger instance.
	 */
	void setLogger(final Logger logger) {
		this.logger = logger;
	}
	
	/**
	 * Provides a a new instance of the System Login Service, which can be used to create users and roles.
	 * 
	 * @return A new instance of the Login Service.
	 */
	public LoginService provideLoginService() {
		if (this.usedDatabaseManagerFactoryName.equals(ApplicationStarter.ORACLE_DATABASE_MANAGER_FACTORY_NAME)) {
			return new LoginService(new OracleAuthorizationFacadeImplementation(this.getAppName(),
					(OracleDatabaseManager) this.getDatabaseManager()));
		} else {
			throw new FrameworkException("Database " + this.usedDatabaseManagerFactoryName + " is not supported");
		}
	}
	
}
