package de.fhdw.wtf.context.model;

import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import de.fhdw.wtf.context.core.ApplicationContainer;
import de.fhdw.wtf.context.core.Logger;
import de.fhdw.wtf.context.core.Observer;
import de.fhdw.wtf.context.core.ObserverEvent;
import de.fhdw.wtf.context.core.PersistenceContext;
import de.fhdw.wtf.context.core.TransactionManager;
import de.fhdw.wtf.context.core.UserAccess;
import de.fhdw.wtf.persistence.exception.ClassFacadeUninitializedException;
import de.fhdw.wtf.persistence.exception.RuntimePersistenceException;
import de.fhdw.wtf.persistence.meta.UserObject;
import de.fhdw.wtf.persistence.meta.UserTransaction;

/**
 * A superclass to all service classes.
 */
public abstract class Service extends UserAccess {
	
	/**
	 * Observers registered with this service.
	 */
	private final Collection<Observer> observers = new Vector<>();
	
	/**
	 * The transaction this Service object lives in.
	 */
	private UserTransaction transaction;
	
	/**
	 * Returns the underlying transaction.
	 * 
	 * @return The underlying transaction.
	 */
	UserObject getObject() {
		return this.transaction;
	}
	
	/**
	 * Sets the underlying transaction.
	 * 
	 * @param transaction
	 *            The underlying transaction.
	 */
	void setTransaction(final UserTransaction transaction) {
		this.transaction = transaction;
	}
	
	/**
	 * This operation signals the end of all operations in the service. This operation closes the running long
	 * transaction of this service. All modifications will be written to the database and published, so they are visible
	 * to other users.
	 */
	public void commit() {
		TransactionManager.commit();
	}
	
	/**
	 * This operation signals the end of all operations in the service. It will perform a rollback in the persistence
	 * layer, so all modifications and changes will be lost, also this transation will be closed.
	 */
	public void rollback() {
		TransactionManager.rollback();
	}
	
	@Override
	public void publish() {
		try {
			TransactionManager.setContext(new PersistenceContext(ApplicationContainer.getInstance()
					.getDatabaseManager().getObjectFacade(), this.transaction));
		} catch (final ClassFacadeUninitializedException e) {
			Logger.getInstance().log(e);
			throw new RuntimePersistenceException(e);
		}
	}
	
	/**
	 * Returns the ID of this service.
	 * 
	 * @return The ID of this service.
	 */
	public String getId() {
		return String.valueOf(this.getObject().getId());
	}
	
	/**
	 * Registers an observer.
	 * 
	 * @param o
	 *            The observer to register.
	 */
	public void registerObserver(final Observer o) {
		this.observers.add(o);
	}
	
	/**
	 * Unregisters an observer.
	 * 
	 * @param o
	 *            The observer to unregister.
	 */
	public void deregisterObserver(final Observer o) {
		this.observers.remove(o);
	}
	
	/**
	 * Notifies all observers about an event.
	 * 
	 * @param event
	 *            The event.
	 */
	public void notifyObservers(final ObserverEvent event) {
		final Iterator<Observer> i = this.observers.iterator();
		while (i.hasNext()) {
			final Observer current = i.next();
			current.update(event);
		}
	}
	
}
