package de.fhdw.wtf.context.model.collections;

import java.util.Iterator;
import java.util.ListIterator;
import java.util.Vector;

import de.fhdw.wtf.context.exception.FrameworkException;
import de.fhdw.wtf.context.model.Anything;
import de.fhdw.wtf.context.model.collections.functors.EquivalenceRelation;
import de.fhdw.wtf.context.model.collections.functors.Function;
import de.fhdw.wtf.context.model.collections.functors.Operator;
import de.fhdw.wtf.context.model.collections.functors.Predicate;

/**
 * An ImmutableList represents a List with an immutable list as backend.
 * 
 * @param <T>
 *            The underlying element type.
 */
public class ImmutableList<T extends Anything> implements ImmutableCollection<T> {
	
	/**
	 * The underlying list.
	 */
	private final java.util.List<T> elements;
	
	/**
	 * Creates an empty ImmutableList.
	 */
	public ImmutableList() {
		this.elements = new Vector<>();
	}
	
	@Override
	public Collection<T> union(final Collection<? extends T> otherCollection) {
		final ImmutableList<T> result = new ImmutableList<>();
		result.elements.addAll(this.elements);
		if (otherCollection instanceof ImmutableCollection) {
			result.addImmutableCollection((ImmutableCollection<? extends T>) otherCollection);
		} else if (otherCollection instanceof MutableCollection) {
			result.addMutableCollection(otherCollection);
		} else {
			throw new FrameworkException("collection not supported");
		}
		return result;
	}
	
	/**
	 * Adds the elements of a MutableCollection to this list.
	 * 
	 * @param otherCollection
	 *            The MutableCollection.
	 */
	private void addMutableCollection(final Collection<? extends T> otherCollection) {
		final Iterator<? extends T> iterator = ((MutableCollection<? extends T>) otherCollection).iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			this.elements.add(current);
		}
	}
	
	/**
	 * Adds the elements of an ImmutableCollection to this list.
	 * 
	 * @param otherCollection
	 *            The ImmutableCollection.
	 */
	private void addImmutableCollection(final ImmutableCollection<? extends T> otherCollection) {
		if (!otherCollection.isEmpty()) {
			this.elements.add(otherCollection.front());
			this.addImmutableCollection(otherCollection.tail());
		}
	}
	
	@Override
	public boolean contains(final T element) {
		return this.elements.contains(element);
	}
	
	@Override
	public ImmutableCollection<T> add(final T element) {
		final ImmutableList<T> result = new ImmutableList<>();
		result.elements.addAll(this.elements);
		result.elements.add(element);
		return result;
	}
	
	@Override
	public T front() {
		return this.elements.get(0);
	}
	
	@Override
	public ImmutableCollection<T> tail() {
		final ImmutableList<T> result = new ImmutableList<>();
		result.elements.addAll(this.elements);
		result.elements.remove(0);
		return result;
	}
	
	@Override
	public ImmutableCollection<T> reverse() {
		final ImmutableList<T> result = new ImmutableList<>();
		final ListIterator<T> iterator = this.elements.listIterator(this.elements.size());
		while (iterator.hasPrevious()) {
			final T current = iterator.previous();
			result.elements.add(current);
		}
		return result;
	}
	
	@Override
	public <U extends Anything> ImmutableCollection<U> map(final Function<T, U> function) {
		final ImmutableList<U> result = new ImmutableList<>();
		final Iterator<T> iterator = this.elements.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			result.elements.add(function.f(current));
		}
		return result;
	}
	
	@Override
	public <U extends Anything> U reduce(final Operator<T, U> aggregation) {
		U result = aggregation.n();
		final Iterator<T> iterator = this.elements.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			result = aggregation.add(result, current);
		}
		return result;
	}
	
	@Override
	public <U extends Anything> ImmutableCollection<U> reduceEquivalent(final EquivalenceRelation<T> equi,
			final Operator<T, U> aggregation) {
		final Vector<ImmutableList<T>> between = new Vector<>();
		final Iterator<T> outerIterator = this.elements.iterator();
		while (outerIterator.hasNext()) {
			final T current = outerIterator.next();
			final ImmutableList<T> currentList = new ImmutableList<>();
			final Iterator<T> innerIterator = this.elements.iterator();
			while (innerIterator.hasNext()) {
				final T innerCurrent = innerIterator.next();
				if (equi.equivalent(current, innerCurrent)) {
					currentList.add(innerCurrent);
				}
			}
			between.add(currentList);
		}
		final ImmutableList<U> result = new ImmutableList<>();
		while (!between.isEmpty()) {
			final ImmutableList<T> current = between.get(0);
			between.remove(0);
			final Iterator<ImmutableList<T>> iterator = between.iterator();
			while (iterator.hasNext()) {
				final ImmutableList<T> innerCurrent = iterator.next();
				if (innerCurrent.equals(current)) {
					iterator.remove();
				}
			}
			result.elements.add(current.reduce(aggregation));
		}
		return result;
	}
	
	@Override
	public ImmutableCollection<T> filter(final Predicate<T> predicate) {
		final ImmutableList<T> result = new ImmutableList<>();
		final Iterator<T> iterator = this.elements.iterator();
		while (iterator.hasNext()) {
			final T current = iterator.next();
			if (predicate.p(current)) {
				result.elements.add(current);
			}
		}
		return result;
	}
	
	@Override
	public T find(final Predicate<T> predicate) {
		final ImmutableCollection<T> preResult = this.filter(predicate);
		if (preResult.isEmpty()) {
			return null;
		}
		return preResult.front();
	}
	
	@Override
	public String toString() {
		return this.elements.toString();
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (!(obj instanceof ImmutableList<?>)) {
			return false;
		}
		
		final ImmutableList<?> other = (ImmutableList<?>) obj;
		
		return this.elements.equals(other.elements);
	}
	
	@Override
	public int hashCode() {
		return this.elements.hashCode();
	}
	
	@Override
	public boolean isEmpty() {
		return this.elements.isEmpty();
	}
	
}
