package de.fhdw.wtf.facade;

import java.io.IOException;
import java.util.Collection;
import java.util.concurrent.ExecutionException;

import de.fhdw.wtf.common.ast.Model;
import de.fhdw.wtf.common.exception.editor.GeneralCheckException;
import de.fhdw.wtf.common.exception.editor.MultipleCheckExceptions;
import de.fhdw.wtf.common.exception.walker.CyclicDependencyException;
import de.fhdw.wtf.common.exception.walker.CyclicPartDefinitionException;
import de.fhdw.wtf.common.task.DependencyTask;
import de.fhdw.wtf.common.task.GroupDependencyTask;
import de.fhdw.wtf.common.task.TaskExecutorFixed;
import de.fhdw.wtf.common.task.result.ExceptionalTaskResult;
import de.fhdw.wtf.common.task.result.OKTaskResult;
import de.fhdw.wtf.common.task.result.TaskResult;
import de.fhdw.wtf.common.task.result.visitor.TaskResultVisitor;
import de.fhdw.wtf.generator.database.generation.InitialGenerator;
import de.fhdw.wtf.generator.database.tasks.AttributeGenerationTask;
import de.fhdw.wtf.generator.database.tasks.SpecializationGenerationTask;
import de.fhdw.wtf.generator.database.tasks.TypeGenerationTask;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;
import de.fhdw.wtf.generator.transformer.clipper.ClipToFileTask;
import de.fhdw.wtf.generator.transformer.clipper.ClipperConfiguration;
import de.fhdw.wtf.generator.transformer.clipper.LinkToGenClassTask;
import de.fhdw.wtf.generator.transformer.transformers.classTransformer.AppStarterTransformer;
import de.fhdw.wtf.generator.transformer.transformers.classTransformer.ConstructorCallGenerationTask;
import de.fhdw.wtf.generator.transformer.transformers.classTransformer.ConstructorSymmetricAttribute;
import de.fhdw.wtf.generator.transformer.transformers.classTransformer.FactoryTransformer;
import de.fhdw.wtf.generator.transformer.transformers.classTransformer.InheritanceTransformer;
import de.fhdw.wtf.generator.transformer.transformers.classTransformer.OperationAttributeTransformer;
import de.fhdw.wtf.generator.transformer.transformers.classTransformer.TypeTransformer;
import de.fhdw.wtf.generator.transformer.visitorTransformation.VisitorTypeTransformer;
import de.fhdw.wtf.generator.writer.tasks.FileWriterTask;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.facade.OracleDataBasePreparator;
import de.fhdw.wtf.persistence.facade.OracleDatabaseManager;
import de.fhdw.wtf.persistence.facade.OracleObjectFacadeImplementation;
import de.fhdw.wtf.persistence.utils.PropertiesReaderFile;
import de.fhdw.wtf.tooling.SyntaxCheckAbstract;

/**
 * This class provides operations for generating Java-sources and generating the needed database for an AST-Model.
 */
public final class ModelManager extends SyntaxCheckAbstract {
	
	/**
	 * The one existing instance of ModelManager.
	 */
	private static ModelManager instance;
	
	/**
	 * Provides the one Instance for ModelManager.
	 *
	 * @return ModelManager
	 */
	public static synchronized ModelManager getInstance() {
		if (instance == null) {
			instance = new ModelManager();
		}
		return instance;
	}
	
	/**
	 * Constructs a ModelManager.
	 */
	private ModelManager() {
		
	}
	
	/**
	 * Generates Java and aspectJ files to the given directory.
	 *
	 * @param referencedModel
	 *            model must be referenced successfully before
	 * @param modelRootDirectory
	 *            Root directory to generate Java sources in.
	 * @throws MultipleCheckExceptions
	 *             An exception that represents many exceptions that occured while generating.
	 * @return The produced GeneratorModel.
	 */
	public GeneratorModel generateJava(final Model referencedModel, final String modelRootDirectory)
			throws MultipleCheckExceptions {
		final ClipperConfiguration configurationClip =
				new ClipperConfiguration(modelRootDirectory + "clipper/generated/model", modelRootDirectory
						+ "/generated/model");
		final ClipperConfiguration configurationLink =
				new ClipperConfiguration(modelRootDirectory + "clipper/", modelRootDirectory);
		return this.generateJava(referencedModel, modelRootDirectory, configurationClip, configurationLink, true);
	}
	
	/**
	 * Generates Java and aspectJ files to the given directory.
	 *
	 * @param referencedModel
	 *            model must be referenced successfully before
	 * @param modelRootDirectory
	 *            Root directory to generate Java sources in. If null, the file writer task is not started.
	 * @param configurationClip
	 *            The configuration for the ClipToFileTask.
	 * @param configurationLink
	 *            The configuration for the LinkToGenClassTask.
	 * @param runVisitorAndFactoryGenerators
	 *            If true, the visitor and factory generators are run.
	 * @throws MultipleCheckExceptions
	 *             An exception that represents many exceptions that occured while generating.
	 * @return The produced GeneratorModel.
	 */
	public GeneratorModel generateJava(final Model referencedModel,
			final String modelRootDirectory,
			final ClipperConfiguration configurationClip,
			final ClipperConfiguration configurationLink,
			final boolean runVisitorAndFactoryGenerators) throws MultipleCheckExceptions {
		final MultipleCheckExceptions results = new MultipleCheckExceptions();
		final TaskExecutorFixed exec = TaskExecutorFixed.create();
		final GeneratorModel javaGeneratorModel = GeneratorModel.create();
		
		final GroupDependencyTask transformers = new GroupDependencyTask(exec);
		/* WalkerTasks and WalkerTasks for types */
		try {
			final TypeTransformer typeTransformer = new TypeTransformer(referencedModel, exec, javaGeneratorModel);
			
			final OperationAttributeTransformer operationAttributeTransformer =
					OperationAttributeTransformer.create(referencedModel, exec, javaGeneratorModel, typeTransformer);
			
			final ConstructorCallGenerationTask constructorCallGenerationTask =
					new ConstructorCallGenerationTask(referencedModel, exec, javaGeneratorModel);
			
			constructorCallGenerationTask.addDependency(typeTransformer);
			
			final InheritanceTransformer inheritanceTransformer =
					new InheritanceTransformer(referencedModel, exec, javaGeneratorModel, operationAttributeTransformer);
			
			inheritanceTransformer.addDependency(constructorCallGenerationTask);
			
			final ConstructorSymmetricAttribute constructorSymmetricAttribute =
					new ConstructorSymmetricAttribute(referencedModel, exec, javaGeneratorModel);
			
			constructorSymmetricAttribute.addDependency(constructorCallGenerationTask);
			constructorSymmetricAttribute.addDependency(operationAttributeTransformer);
			
			transformers.addMembers(
					typeTransformer,
					operationAttributeTransformer,
					constructorCallGenerationTask,
					inheritanceTransformer);
			
			if (runVisitorAndFactoryGenerators) {
				final VisitorTypeTransformer typeVisitor =
						VisitorTypeTransformer
								.create(referencedModel, exec, javaGeneratorModel, inheritanceTransformer);
				final DependencyTask appStarterTransformer =
						new AppStarterTransformer(exec, javaGeneratorModel, inheritanceTransformer);
				appStarterTransformer.addDependency(typeVisitor);
				final DependencyTask factoryTransformer =
						new FactoryTransformer(exec, javaGeneratorModel, inheritanceTransformer);
				factoryTransformer.addDependency(appStarterTransformer);
				
				// TODO der FinderTransformer muss eventuell eingebunden werden (und
				// Dependencies hinzugefügt werden)
				transformers.addMembers(typeVisitor, appStarterTransformer, factoryTransformer);
			}
		} catch (final CyclicPartDefinitionException | CyclicDependencyException e) {
			throw new SystemException(e);
		}
		
		/* Clipper */
		final GroupDependencyTask clipper = new GroupDependencyTask(exec);
		final ClipToFileTask clipperImport = new ClipToFileTask(exec, configurationClip);
		final LinkToGenClassTask methodInsertion =
				new LinkToGenClassTask(exec, javaGeneratorModel, configurationLink, clipperImport);
		try {
			clipper.addMembers(clipperImport, methodInsertion);
			clipper.addDependency(transformers);
		} catch (final CyclicPartDefinitionException | CyclicDependencyException e) {
			throw new SystemException(e);
		}
		
		if (modelRootDirectory != null) {
			new FileWriterTask(exec, javaGeneratorModel, modelRootDirectory, clipper);
		}
		
		try {
			exec.startAllKnownTasks();
			final Collection<TaskResult> resultTasks = exec.getResultsAndShutdown();
			this.addsExceptionalTasksToMultipleExceptions(results, resultTasks);
			
		} catch (final InterruptedException | ExecutionException e) {
			throw new SystemException(e);
		}
		
		if (!results.isEmpty()) {
			throw results;
		}
		
		return javaGeneratorModel;
	}
	
	/**
	 * Generates initial persistence and deletes all existing data!
	 *
	 * @param model
	 *            The AST-Model to generate persistence for.
	 * @param genModel
	 *            The Generator model.
	 * @param dbPropertiesPath
	 *            The path to connection-constants for accessing the database.
	 * @throws MultipleCheckExceptions
	 *             An exception that represents many exceptions that occured while generating.
	 */
	public void generatePersistence(final Model model, final GeneratorModel genModel, final String dbPropertiesPath)
			throws MultipleCheckExceptions {
		final MultipleCheckExceptions results = new MultipleCheckExceptions();
		try {
			this.prepareDatabase(dbPropertiesPath);
			
			final InitialGenerator generator = new InitialGenerator();
			final TaskExecutorFixed taskManager = TaskExecutorFixed.create();
			
			final TypeGenerationTask typeGenerationTask =
					new TypeGenerationTask(model, genModel, taskManager, generator);
			final AttributeGenerationTask attributeGenerationTask =
					new AttributeGenerationTask(model, taskManager, generator);
			final SpecializationGenerationTask specializationGenerationTask =
					new SpecializationGenerationTask(model, taskManager, generator);
			
			attributeGenerationTask.addDependency(typeGenerationTask);
			specializationGenerationTask.addDependency(typeGenerationTask);
			
			taskManager.startAllKnownTasks();
			
			final Collection<TaskResult> resultTasks = taskManager.getResultsAndShutdown();
			this.addsExceptionalTasksToMultipleExceptions(results, resultTasks);
		} catch (final PersistenceException e) {
			results.add(new GeneralCheckException(e));
		} catch (final InterruptedException | ExecutionException | CyclicDependencyException | IOException e) {
			throw new SystemException(e);
		}
		if (!results.isEmpty()) {
			throw results;
		}
		
	}
	
	/**
	 * Inspects TaskResults and adds corresponding CheckExceptions to list.
	 *
	 * @param list
	 *            A list of exceptions encapsulated as MultipleCheckException.
	 * @param taskresults
	 *            A collection of the results of executed tasks.
	 */
	private void addsExceptionalTasksToMultipleExceptions(final MultipleCheckExceptions list,
			final Collection<TaskResult> taskresults) {
		for (final TaskResult taskResult : taskresults) {
			taskResult.accept(new TaskResultVisitor() {
				
				@Override
				public void handleOkTaskResult(final OKTaskResult result) {
					// nothing to do
				}
				
				@Override
				public void handleExceptionalTaskResult(final ExceptionalTaskResult result) {
					// TODO get Markable Exceptions
					list.add(new GeneralCheckException(result.getError()));
				}
			});
		}
	}
	
	/**
	 * Trys to prepare the database by generating the meta-structure and procedures.
	 *
	 * @param propertiesPath
	 *            Path for a file with connection-constants.
	 * @throws IOException
	 *             Thrown when the file with connection-constants could not be accessed.
	 * @throws PersistenceException
	 *             Thrown when some error in the persistence layer occured.
	 */
	private void prepareDatabase(final String propertiesPath) throws IOException, PersistenceException {
		
		final OracleDatabaseManager oracleDatabaseManager = OracleDatabaseManager.getInstance();
		final PropertiesReaderFile prop = new PropertiesReaderFile();
		prop.initialize(propertiesPath);
		oracleDatabaseManager.setConnectionConstantsFromFile(prop);
		oracleDatabaseManager.connect();
		
		final OracleDataBasePreparator preparator = new OracleDataBasePreparator();
		if (!preparator.isTableStructureValid()) {
			preparator.dropWholeSchema();
			preparator.createWholeSchema();
		}
		if (!preparator.areProceduresCreated()) {
			preparator.createProcedures();
		}
		new OracleObjectFacadeImplementation(oracleDatabaseManager, null).clear();
		oracleDatabaseManager.getClassFacade().clear();
	}
}
