package de.fhdw.wtf.generator.writer.writer;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import org.apache.commons.lang.NotImplementedException;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.context.Context;
import org.apache.velocity.runtime.RuntimeConstants;
import org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader;

import de.fhdw.wtf.file.FileUtils;
import de.fhdw.wtf.generator.java.generatorModel.GenAnyType;
import de.fhdw.wtf.generator.java.generatorModel.GenAspect;
import de.fhdw.wtf.generator.java.generatorModel.GenAspectOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenException;
import de.fhdw.wtf.generator.java.generatorModel.GenExternalClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenFullParsedOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaException;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenOperationModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenParameter;
import de.fhdw.wtf.generator.java.generatorModel.GenPrimitiveClass;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenType;
import de.fhdw.wtf.generator.java.generatorModel.GenTypeReference;
import de.fhdw.wtf.generator.java.generatorModel.GenTypeReferenceByName;
import de.fhdw.wtf.generator.java.generatorModel.GenTypeReferenceByReference;
import de.fhdw.wtf.generator.java.generatorModel.GenUserClass;
import de.fhdw.wtf.generator.java.generatorModel.Generic;
import de.fhdw.wtf.generator.java.visitor.GenClassClassVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitor;
import de.fhdw.wtf.generator.java.visitor.GenOperationStateVisitor;
import de.fhdw.wtf.generator.java.visitor.GenOperationVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenTypeReferenceVisitorReturn;

/**
 * 
 * Abstraction for Java-File writer. Handles all properties that classes and interfaces have in common. Such as:
 * <ul>
 * <li>
 * Name</li>
 * <li>Comments</li>
 * <li>
 * Operations</li>
 * <li>Inheritance</li>
 * <li>Non-Generation-Part</li>
 * <li>imports</li>
 * </ul>
 */
public abstract class ClassFileWriter {
	
	static final String TARGED_CLASS_KEY = "targedClass";
	private static final String PACKAGE_KEY = "package";
	static final String PARAMETER_KEY = "parameter";
	private static final String CLASSNAME_KEY = "className";
	static final String PARSED_OPERATIONS_KEY = "parsedOperations";
	static final String SIMPLE_OPERATIONS_KEY = "simpleOperations";
	static final String VISIBILITY_KEY = "visibility";
	static final String MODIFIERS_KEY = "modifiers";
	static final String NAME_KEY = "name";
	private static final String ABSTRACT_KEY = "abstract";
	static final String TYP_KEY = "typ";
	private static final String RETURN_TYP_KEY = "returnTyp";
	static final String CLASS_MODIFIERS_KEY = "classModifiers";
	static final String METHOD_KEY = "method";
	private static final String NON_GERNATION_PART_KEY = "nonGenerationPart";
	private static final String CLASS_COMMENT_KEY = "classComment";
	private static final String OPERATION_COMMENT_KEY = "operationComment";
	static final String EXTENDS_KEY = "extends";
	private static final String IMPLEMENTS_KEY = "implements";
	static final String EXCEPTIONS_KEY = "exceptions";
	private static final String OPERATION_PARAMETER_KEY = "parameter";
	private static final String IMPORTS_KEY = "imports";
	static final String GENERICS_PARAMETER_KEY = "genericParam";
	static final String GENERICS_CLASS_KEY = "genericsClass";
	static final String GENERICS_OPERATION_KEY = "genericsOperation";
	static final String INNER_CLASSES_KEY = "innerClasses";
	
	static final String EXTENDS = "extends ";
	private static final String IMPLEMENTS = "implements ";
	private static final char PACKAGE_PATH_SEP = '.';
	private static final char JAVA_LINE_END = ';';
	private static final char FILE_PATH_SEP = '/';
	private static final String THROWS = "throws ";
	private static final String BLANK = " ";
	private static final String COMMA_SPARATE = ", ";
	private static final String GENERICS_BEGIN = "<";
	private static final String GENERICS_END = ">";
	
	private final VelocityEngine engine;
	private final Template template;
	
	private final boolean fullyQualified;
	
	private Context currentContext;
	
	public ClassFileWriter(final String templateFileName, final boolean fullyQualified) {
		this.engine = new VelocityEngine();
		this.engine.setProperty(RuntimeConstants.RESOURCE_LOADER, "classpath");
		this.engine.setProperty("classpath.resource.loader.class", ClasspathResourceLoader.class.getName());
		this.engine.init();
		this.template = this.engine.getTemplate(templateFileName);
		this.fullyQualified = fullyQualified;
	}
	
	public Context getCurrentContext() {
		return this.currentContext;
	}
	
	/**
	 * Returns the type name (unqualified if {@link #fullyQualified} is false otherwise fully qualified).
	 * 
	 * @param type
	 */
	protected String typeToString(final GenType type) {
		if (this.fullyQualified) {
			return type.getFullyQualifiedTypeNameWithGenericArguments();
		} else {
			return type.getName();
		}
	}
	
	/**
	 * Returns the type name (unqualified if {@link #fullyQualified} is {@code false} otherwise fully qualified) for
	 * {@link GenTypeReferenceByReference} or the Name Attribute for {@link GenTypeReferenceByName}.
	 * 
	 * @param typeReference
	 */
	protected String typeToString(final GenTypeReference typeReference) {
		return typeReference.accept(new GenTypeReferenceVisitorReturn<String>() {
			@Override
			public String handle(final GenTypeReferenceByName typeReferenceByName) {
				return typeReferenceByName.getFullyQualifiedNameWithGenericArguments();
			}
			
			@Override
			public String handle(final GenTypeReferenceByReference typeReferenceByReference) {
				return ClassFileWriter.this.typeToString(typeReferenceByReference.getTyp());
			}
		});
	}
	
	/**
	 * Writes the file for the given {@link GenClass} c.
	 * 
	 * @param c
	 */
	void writeClass(final GenClass c) {
		this.currentContext = new VelocityContext();
		this.setUpContext(c, this.currentContext);
	}
	
	/**
	 * Writes the file for the given {@link GenClass} <code>c</code>.
	 * 
	 * @param rootdir
	 */
	protected void writeToFile(final GenClass c, final File rootdir) {
		final String result = this.generateFileContent(this.currentContext);
		final File path = this.createPath(c, rootdir);
		final File file = this.createFile(path, c);
		this.writeFile(result, path, file);
	}
	
	public String writeToString() {
		return this.generateFileContent(this.currentContext);
	}
	
	/**
	 * Sets up the {@link VelocityContext} with all necessary values for classes.
	 * 
	 * @param c
	 *            {@link GenClass} class to write
	 * @param context
	 *            the {@link VelocityContext} for the class
	 */
	private void setUpContext(final GenClass c, final Context context) {
		this.setClassComment(c, context);
		this.setClassName(c, context);
		this.setClassGenerics(c, context);
		this.setImplements(c, context);
		this.setNonGenerationPart(c, context);
		this.setInnerClasses(c, context);
	}
	
	/**
	 * Sets all generic of the {@link GenClass} <code>c</code>.
	 * 
	 * @param c
	 * @param context
	 */
	private void setClassGenerics(final GenClass c, final Context context) {
		final StringBuilder gen = new StringBuilder();
		final Iterator<Generic> i = c.getGenerics().listIterator();
		if (i.hasNext()) {
			gen.append(GENERICS_BEGIN);
		}
		while (i.hasNext()) {
			final Generic current = i.next();
			gen.append(current.toString());
			if (i.hasNext()) {
				gen.append(COMMA_SPARATE);
			} else {
				gen.append(GENERICS_END);
			}
		}
		context.put(GENERICS_CLASS_KEY, gen.toString());
	}
	
	/**
	 * Sets all values for the classcomment of {@link GenClass} c to the {@link Context}.
	 * 
	 * @param c
	 * @param context
	 */
	private void setClassComment(final GenClass c, final Context context) {
		context.put(CLASS_COMMENT_KEY, c.getComment().getText());
	}
	
	/**
	 * Sets all values for the classname of {@link GenClass} c to the {@link Context}.
	 * 
	 * @param c
	 * @param context
	 */
	private void setClassName(final GenClass c, final Context context) {
		context.put(CLASSNAME_KEY, c.getName());
	}
	
	Collection<String> getSimpleOperations(final Collection<? extends GenOperation> operations) {
		final Vector<String> simpleOperations = new Vector<>();
		for (final GenOperation operation : operations) {
			operation.getState().accept(new GenOperationStateVisitor() {
				@Override
				public void handle(final GenFullParsedOperationState s) {
					// nothing to do
				}
				
				@Override
				public void handle(final GenSimpleOperationState s) {
					simpleOperations.add(s.getFullOperationWithPossibleImplementation());
				}
			});
		}
		return simpleOperations;
	}
	
	Collection<HashMap<String, Object>> getParsedOperations(final Collection<? extends GenOperation> operations) {
		final Vector<HashMap<String, Object>> parsedOperations = new Vector<>();
		for (final GenOperation operation : operations) {
			operation.getState().accept(new GenOperationStateVisitor() {
				@Override
				public void handle(final GenFullParsedOperationState s) {
					final HashMap<String, Object> current = new HashMap<>();
					current.put(VISIBILITY_KEY, s.getVisibility().toString());
					current.put(NAME_KEY, operation.getName());
					current.put(RETURN_TYP_KEY, ClassFileWriter.this.typeToString(s.getReturntyp()));
					s.getModifiers().remove(GenOperationModifier.ABSTRACT);
					final Vector<String> modifiers = new Vector<>();
					for (final GenOperationModifier mod : s.getModifiers()) {
						modifiers.add(mod.toString());
					}
					current.put(GENERICS_OPERATION_KEY, ClassFileWriter.this.getGenericString(operation));
					current.put(MODIFIERS_KEY, modifiers);
					current.put(METHOD_KEY, s.getMethodBody());
					current.put(OPERATION_COMMENT_KEY, s.getComment().getText());
					current.put(OPERATION_PARAMETER_KEY, ClassFileWriter.this.getParamString(operation.getParameters()));
					current.put(EXCEPTIONS_KEY, ClassFileWriter.this.getExceptionString(s.getExceptions()));
					operation.accept(new GenOperationVisitorReturn<GenOperation>() {
						@Override
						public GenOperation handleJavaOperation(final GenJavaOperation operation) {
							return operation;
						}
						
						@Override
						public GenOperation handleAspectOperation(final GenAspectOperation operation) {
							current.put(TARGED_CLASS_KEY, ClassFileWriter.this.typeToString(operation.getOwner()) + ".");
							return operation;
						}
					});
					parsedOperations.add(current);
				}
				
				@Override
				public void handle(final GenSimpleOperationState s) {
					// nothing to do
				}
			});
		}
		return parsedOperations;
	}
	
	/**
	 * Returns the list of generic.
	 * 
	 * @param operation
	 * @return
	 */
	private String getGenericString(final GenOperation operation) {
		final StringBuilder gen = new StringBuilder();
		final Iterator<Generic> i = operation.getGenerics().listIterator();
		if (i.hasNext()) {
			gen.append(GENERICS_BEGIN);
		}
		while (i.hasNext()) {
			final Generic current = i.next();
			String generic = "";
			generic += current.toString();
			gen.append(generic);
			if (i.hasNext()) {
				gen.append(COMMA_SPARATE);
			} else {
				gen.append(GENERICS_END);
			}
		}
		return gen.toString();
	}
	
	/**
	 * Returns the String representing the throws clause. <br>
	 * <code>throws Exc1, Exc2, ...</code>
	 * 
	 * @param exceptions
	 */
	protected String getExceptionString(final Collection<GenException> exceptions) {
		if (exceptions.isEmpty()) {
			return "";
		}
		final StringBuilder result = new StringBuilder();
		result.append(THROWS);
		final Iterator<GenException> i = exceptions.iterator();
		while (i.hasNext()) {
			final GenException current = i.next();
			result.append(this.typeToString(current));
			if (i.hasNext()) {
				result.append(COMMA_SPARATE);
			}
		}
		return result.toString();
	}
	
	/**
	 * Returns the comma separated parameters.<br>
	 * <code>Typ1 name1, Typ1 name2, ...</code>
	 * 
	 * @param parameters
	 */
	protected String getParamString(final Collection<GenParameter> parameters) {
		final StringBuilder params = new StringBuilder();
		final Iterator<GenParameter> i = parameters.iterator();
		while (i.hasNext()) {
			final GenParameter current = i.next();
			params.append(this.typeToString(current.getTyp()));
			final Iterator<Generic> iGen = current.getGenerics().listIterator();
			if (iGen.hasNext()) {
				params.append(GENERICS_BEGIN);
			}
			while (iGen.hasNext()) {
				final Generic currentGen = iGen.next();
				params.append(currentGen.getName());
				if (iGen.hasNext()) {
					params.append(COMMA_SPARATE);
				} else {
					params.append(GENERICS_END);
				}
			}
			params.append(BLANK);
			params.append(current.getName());
			if (i.hasNext()) {
				params.append(COMMA_SPARATE);
			}
		}
		return params.toString();
	}
	
	/**
	 * Sets the nongenerationpart of {@link GenClass} c to the {@link Context}.
	 * 
	 * @param c
	 * @param context
	 */
	private void setNonGenerationPart(final GenClass c, final Context context) {
		context.put(NON_GERNATION_PART_KEY, c.getNonGeneratedPart());
	}
	
	/**
	 * Creates the Path {@link File} for c.
	 * 
	 * @param c
	 */
	private File createPath(final GenClass c, final File rootDir) {
		String path = c.getPackag().toString();
		path = path.replace(PACKAGE_PATH_SEP, FILE_PATH_SEP);
		path = path.replace(JAVA_LINE_END, FILE_PATH_SEP);
		return new File(rootDir.getAbsolutePath() + FILE_PATH_SEP + path);
	}
	
	/**
	 * Creates the {@link File} for class c.
	 * 
	 * @param path
	 * @param c
	 */
	private File createFile(final File path, final GenClass c) {
		final String file = path.getAbsolutePath() + FILE_PATH_SEP + c.getName() + c.getFileEnding();
		return new File(file);
	}
	
	/**
	 * Writes the given result into the file.
	 * 
	 * @param result
	 * @param path
	 * @param file
	 */
	private void writeFile(final String result, final File path, final File file) {
		try {
			System.out.println("  -->  " + file.getAbsolutePath());
			FileUtils.overrideToFile(result, path, file);
		} catch (final IOException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Merges the context into the classtemplate.
	 * 
	 * @param context
	 */
	private String generateFileContent(final Context context) {
		final StringWriter writer = new StringWriter();
		this.template.merge(context, writer);
		return writer.toString();
	}
	
	public abstract String getStringContent(final GenClass c);
	
	/**
	 * Sets the template variable for inheritance.
	 * 
	 * @param context
	 */
	protected void setImplements(final GenClass c, final Context context) {
		final StringBuilder implsS = new StringBuilder();
		final Iterator<GenInterfaceClass> i = c.getImplement().iterator();
		while (i.hasNext()) {
			final GenInterfaceClass current = i.next();
			implsS.append(this.typeToString(current));
			if (i.hasNext()) {
				implsS.append(COMMA_SPARATE);
			}
		}
		if (!c.getImplement().isEmpty()) {
			c.accept(new GenClassVisitor() {
				@Override
				public void handle(final GenInterfaceClass interfaceClass) {
					context.put(IMPLEMENTS_KEY, EXTENDS + implsS.toString());
				}
				
				@Override
				public void handle(final GenClassClass classClass) {
					context.put(IMPLEMENTS_KEY, IMPLEMENTS + implsS.toString());
				}
				
				@Override
				public void handle(final GenPrimitiveClass primitiveClass) {
					// nothing to do
				}
			});
		} else {
			context.put(IMPLEMENTS_KEY, "");
		}
	}
	
	private void setInnerClasses(final GenClass c, final Context context) {
		final Collection<String> innerClassesCollection = new Vector<>();
		final Iterator<GenClassClass> innerClassesIterator = c.getInnerClasses().iterator();
		final boolean subclassesFullyQualified = true;
		while (innerClassesIterator.hasNext()) {
			final GenClassClass current = innerClassesIterator.next();
			final String innerClassAsString = current.accept(new GenClassClassVisitorReturn<String>() {
				
				@Override
				public String handle(final GenException e) {
					final ExceptionFileWriter subExceptionClassFileWriter =
							new ExceptionFileWriter(subclassesFullyQualified);
					return subExceptionClassFileWriter.getStringContent(e);
				}
				
				@Override
				public String handle(final GenAnyType anyType) {
					// TODO handle write inner class for this type
					throw new NotImplementedException();
				}
				
				@Override
				public String handle(final GenJavaException javaE) {
					// TODO handle write inner class for this type
					throw new NotImplementedException();
				}
				
				@Override
				public String handle(final GenUserClass userclass) {
					final JavaClassFileWriter subClassFileWriter = new JavaClassFileWriter(subclassesFullyQualified);
					return subClassFileWriter.getStringContent(userclass);
				}
				
				@Override
				public String handle(final GenAspect aspect) {
					// TODO handle write inner class for this type
					throw new NotImplementedException();
				}
				
				@Override
				public String handle(final GenExternalClassClass externalClass) {
					throw new NotImplementedException();
				}
			});
			innerClassesCollection.add(innerClassAsString);
		}
		context.put(INNER_CLASSES_KEY, innerClassesCollection);
	}
}
