package de.fhdw.wtf.generator.database.generation;

import java.util.HashMap;
import java.util.Map;

import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.exception.TypeOrAssociationNotFoundException;
import de.fhdw.wtf.persistence.facade.ClassFacade;
import de.fhdw.wtf.persistence.facade.OracleClassFacadeImplementation;
import de.fhdw.wtf.persistence.facade.OracleDatabaseManager;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.TypeVisitorReturnException;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.persistence.meta.UserType;

public class InitialGenerator {
	
	/**
	 * The underlying class facade for database access.
	 */
	private final ClassFacade classFacade;
	/**
	 * Maps type names to database Types.
	 */
	private final Map<String, Type> types;
	
	/**
	 * Creates an InitialGenerator.
	 * 
	 * @throws PersistenceException
	 *             if some database error occurs.
	 */
	public InitialGenerator() throws PersistenceException {
		this.classFacade = new OracleClassFacadeImplementation(OracleDatabaseManager.getInstance());
		this.classFacade.initialize();
		this.types = new HashMap<>();
		// this.objectFacade = new
		// OracleObjectFacadeImplementation(OracleDatabaseManager.getInstance(),
		// this.classFacade.getTypeManager());
		
	}
	
	/**
	 * Maps a type name to a database Type object.
	 * 
	 * @param typeName
	 *            The name of the type.
	 * @return The database Type object or null if no such object exists.
	 */
	public Type mapNameToType(final String typeName) {
		return this.types.get(typeName);
	}
	
	/**
	 * creates a new user type.
	 * 
	 * @param className
	 *            The Name of the User Type.
	 * @param abs
	 *            A boolean flag indicating whether this User Type is abstract or not
	 * @param trans
	 *            A boolean flag indicating whether this User Type is also an Transaction or not.
	 * @return the new user type
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	public UserType createClass(final String className, final boolean abs, final boolean trans)
			throws PersistenceException {
		final UserType result = this.classFacade.createUserType(className, abs, false);
		this.types.put(className, result);
		return result;
	}
	
	// /**
	// * Deletes the given association and unsets all links of it.
	// * @param associationName
	// * @throws PersistenceException
	// */
	// private void deleteAssociation(String associationName) throws
	// PersistenceException {
	// UnidirectionalAssociation a =
	// this.classFacade.getTypeManager().getAssociationForName(associationName);
	// // TODO implement!
	// }
	
	/**
	 * created an unidirectional association between source and target.
	 * 
	 * @param name
	 *            name of the association
	 * @param essential
	 *            A boolean flag indicating whether there is an existing object necessary. (when true: 1..)
	 * @param unique
	 *            A boolean flag indicating whether the association is set-valued. (when true: ..1)
	 * @param source
	 *            the id of the owning side of the UnidirectionalAssociation
	 * @param target
	 *            the id of the targeting side of the UnidirectionalAssociation.
	 * @return Provides a new UnidirectionalAssociation with the given attributes and the next possible ID.
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	public UnidirectionalAssociation createUnidirectionalAssociation(final String name,
			final boolean essential,
			final boolean unique,
			final long source,
			final long target) throws PersistenceException {
		UserType s;
		Type t;
		final Type s1 = this.classFacade.getTypeManager().getTypeForId(source);
		s = s1.accept(new TypeVisitorReturnException<UserType, TypeOrAssociationNotFoundException>() {
			@Override
			public UserType handleUserType(final UserType t) {
				return t;
			}
			
			@Override
			public UserType handleBaseType(final Type t) throws TypeOrAssociationNotFoundException {
				throw new TypeOrAssociationNotFoundException(t.getId());
			}
		});
		t = this.classFacade.getTypeManager().getTypeForId(target);
		return this.classFacade.createUnidirectionalAssociation(name, essential, unique, s, t);
	}
	
	/**
	 * Creates a map association in the database.
	 * 
	 * @param name
	 *            The name of the association.
	 * @param essential
	 *            If true, the association is essential.
	 * @param source
	 *            The owner type of the association.
	 * @param target
	 *            The target type of the association.
	 * @param key
	 *            The key type of the map association.
	 * @return A {@link MapAssociation} object.
	 * @throws PersistenceException
	 *             if some database error occurs
	 */
	public MapAssociation createMapAssociation(final String name,
			final boolean essential,
			final long source,
			final long target,
			final long key) throws PersistenceException {
		final UserType s =
				this.classFacade.getTypeManager().getTypeForId(source)
						.accept(new TypeVisitorReturnException<UserType, TypeOrAssociationNotFoundException>() {
							@Override
							public UserType handleUserType(final UserType type) {
								return type;
							}
							
							@Override
							public UserType handleBaseType(final Type type) throws TypeOrAssociationNotFoundException {
								throw new TypeOrAssociationNotFoundException(type.getId());
							}
						});
		final Type t = this.classFacade.getTypeManager().getTypeForId(target);
		final Type k = this.classFacade.getTypeManager().getTypeForId(key);
		return this.classFacade.createMapAssociation(name, essential, s, t, k);
	}
	
	/**
	 * Creates a specialization-relationship between two user types.
	 * 
	 * @param ancestor
	 *            The id of the Ancestor which means the super type in the specialization
	 * @param descendant
	 *            The id of the descendant which means the sub type in the specialization
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	public void createSpecialisation(final long ancestor, final long descendant) throws PersistenceException {
		final Type d = this.classFacade.getTypeManager().getTypeForId(descendant);
		final UserType a =
				this.classFacade.getTypeManager().getTypeForId(ancestor)
						.accept(new TypeVisitorReturnException<UserType, TypeOrAssociationNotFoundException>() {
							@Override
							public UserType handleUserType(final UserType t) throws TypeOrAssociationNotFoundException {
								return t;
							}
							
							@Override
							public UserType handleBaseType(final Type t) throws TypeOrAssociationNotFoundException {
								throw new TypeOrAssociationNotFoundException(t.getId());
							}
						});
		this.classFacade.createSpecializationBetween(a, d);
		
	}
	
	/**
	 * Returns the underlying class facade.
	 * 
	 * @return The underlying class facade.
	 */
	public ClassFacade getClassFacade() {
		return this.classFacade;
	}
	
}
