package de.fhdw.wtf.generator.database.tasks;

import de.fhdw.wtf.common.ast.Attribute;
import de.fhdw.wtf.common.ast.ConstructorOrOperation;
import de.fhdw.wtf.common.ast.DatabaseIDSetState;
import de.fhdw.wtf.common.ast.Group;
import de.fhdw.wtf.common.ast.Model;
import de.fhdw.wtf.common.ast.type.AtomicType;
import de.fhdw.wtf.common.ast.type.ClassType;
import de.fhdw.wtf.common.ast.type.CompositeType;
import de.fhdw.wtf.common.ast.type.ListType;
import de.fhdw.wtf.common.ast.type.MapType;
import de.fhdw.wtf.common.ast.type.ProductType;
import de.fhdw.wtf.common.ast.type.SumType;
import de.fhdw.wtf.common.ast.type.ThrownType;
import de.fhdw.wtf.common.ast.type.TypeProxy;
import de.fhdw.wtf.common.ast.visitor.CompositeTypeVisitorException;
import de.fhdw.wtf.common.ast.visitor.TypeVisitorException;
import de.fhdw.wtf.common.constants.referencer.BaseTypeConstants;
import de.fhdw.wtf.common.exception.generation.NoTypeIdSetException;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.context.model.collections.PersistentList;
import de.fhdw.wtf.context.model.collections.PersistentMap;
import de.fhdw.wtf.generator.database.generation.InitialGenerator;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.facade.TypeManager;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.walker.walker.HelperUtils;
import de.fhdw.wtf.walker.walker.SimpleWalkerTask;

/**
 * The {@link AttributeGenerationTask} puts all classes of the Model (AST) into the database. It also adds Type-Ids to
 * the AST.
 * 
 */
public class AttributeGenerationTask extends SimpleWalkerTask {
	
	/**
	 * The name of the default attribute 'This'.
	 */
	private static final String THIS_ATTRIBUTE_NAME = "This";
	/**
	 * The name of the default attribute '$generatedObjects'.
	 */
	private static final String GENERATED_OBJECTS_ATTRIBUTE_NAME = "$generatedObjects";
	
	/**
	 * Creates an {@link AttributeGenerationTask}.
	 * 
	 * @param m
	 *            The associated model.
	 * @param man
	 *            The task manager to use.
	 * @param generator
	 *            The generator to use.
	 */
	public AttributeGenerationTask(final Model m, final TaskExecutor man, final InitialGenerator generator) {
		super(m, man);
		this.generator = generator;
		this.persistentList = null;
		this.persistentMap = null;
	}
	
	/**
	 * The generator to use.
	 */
	private final InitialGenerator generator;
	/**
	 * The type used for persistent lists.
	 */
	private Type persistentList;
	/**
	 * The type used for persistent maps.
	 */
	private Type persistentMap;
	
	@Override
	public void handleAttribute(final Attribute a, final ClassType owner) throws TaskException {
		try {
			a.getAttrType().accept(new AttributeTypeVisitor(a, owner));
		} catch (final TaskException e) {
			throw e;
		} catch (final Exception e) {
			throw new TaskException(e);
		}
		
	}
	
	@Override
	public void finalizeTask() {
		// Not involved in this task
	}
	
	@Override
	public void handleConstructorOrOperation(final ConstructorOrOperation coo, final ClassType owner)
			throws TaskException {
		// Not involved in this task
	}
	
	@Override
	public void handleGroup(final Group g) throws TaskException {
		// Not involved in this task
	}
	
	/**
	 * Creates the fields 'This' and '$generatedObjects' as attributes of the class for classId in the persistence.
	 * 
	 * @param c
	 *            The class to create the fields for.
	 */
	@Override
	public void handleClass(final ClassType c) throws TaskException {
		long classId;
		try {
			classId = c.getTypeId().getId();
			final TypeManager typeManager = this.generator.getClassFacade().getTypeManager();
			if (!typeManager.existsType("de.fhdw.wtf.context.model.collections.MutableMap")) {
				this.generator.createClass("de.fhdw.wtf.context.model.collections.MutableMap", false, false);
			}
			this.generator.createUnidirectionalAssociation(THIS_ATTRIBUTE_NAME, false, false, classId, classId);
			this.generator.createUnidirectionalAssociation(GENERATED_OBJECTS_ATTRIBUTE_NAME, false, false, classId, this.generator
					.mapNameToType(PersistentMap.class.getName()).getId());
		} catch (final NoTypeIdSetException | PersistenceException e) {
			throw new TaskException(e);
		}
	}
	
	@Override
	public void beginTask() throws TaskException {
		this.persistentList = this.generator.mapNameToType(PersistentList.class.getName());
		this.persistentMap = this.generator.mapNameToType(PersistentMap.class.getName());
	}
	
	/**
	 * Handles an attribute's type.
	 */
	private class AttributeTypeVisitor implements TypeVisitorException<Exception> {
		
		/**
		 * The attribute to operate on.
		 */
		private final Attribute a;
		/**
		 * The owner type of the attribute.
		 */
		private final ClassType owner;
		
		/**
		 * Creates an {@link AttributeTypeVisitor}.
		 * 
		 * @param a
		 *            The attribute to operate on.
		 * @param owner
		 *            The owner type of the attribute.
		 */
		AttributeTypeVisitor(final Attribute a, final ClassType owner) {
			this.a = a;
			this.owner = owner;
		}
		
		@Override
		public void handle(final AtomicType s) throws Exception {
			final UnidirectionalAssociation newAsso =
					AttributeGenerationTask.this.generator.createUnidirectionalAssociation(
							this.a.getName(),
							true,
							true,
							this.owner.getTypeId().getId(),
							s.getTypeId().getId());
			this.a.setAttributeId(new DatabaseIDSetState(newAsso.getId()));
		}
		
		@Override
		public void handle(final CompositeType c) throws Exception {
			c.accept(new CompositeAttributeTypeVisitor(this.a, this.owner));
		}
		
		@Override
		public void handle(final TypeProxy s) throws Exception {
			HelperUtils.getReferencedType(s).accept(this);
		}
	}
	
	/**
	 * Handles an attribute's composite type.
	 */
	private class CompositeAttributeTypeVisitor implements CompositeTypeVisitorException<Exception> {
		/**
		 * The attribute to operate on.
		 */
		private final Attribute a;
		/**
		 * The owner type of the attribute.
		 */
		private final ClassType owner;
		
		/**
		 * Creates an {@link AttributeTypeVisitor}.
		 * 
		 * @param a
		 *            The attribute to operate on.
		 * @param owner
		 *            The owner type of the attribute.
		 */
		CompositeAttributeTypeVisitor(final Attribute a, final ClassType owner) {
			this.a = a;
			this.owner = owner;
		}
		
		@Override
		public void handle(final ListType list) throws Exception {
			final UnidirectionalAssociation newAsso =
					AttributeGenerationTask.this.generator.createUnidirectionalAssociation(
							CompositeAttributeTypeVisitor.this.a.getName(),
							true,
							true,
							CompositeAttributeTypeVisitor.this.owner.getTypeId().getId(),
							AttributeGenerationTask.this.persistentList.getId());
			CompositeAttributeTypeVisitor.this.a.setAttributeId(new DatabaseIDSetState(newAsso.getId()));
		}
		
		@Override
		public void handle(final MapType map) throws Exception {
			final UnidirectionalAssociation newAsso =
					AttributeGenerationTask.this.generator.createUnidirectionalAssociation(
							CompositeAttributeTypeVisitor.this.a.getName(),
							true,
							true,
							CompositeAttributeTypeVisitor.this.owner.getTypeId().getId(),
							AttributeGenerationTask.this.persistentMap.getId());
			CompositeAttributeTypeVisitor.this.a.setAttributeId(new DatabaseIDSetState(newAsso.getId()));
		}
		
		@Override
		public void handle(final ProductType product) throws Exception {
			final UnidirectionalAssociation newAsso =
					AttributeGenerationTask.this.generator.createUnidirectionalAssociation(
							CompositeAttributeTypeVisitor.this.a.getName(),
							true,
							true,
							CompositeAttributeTypeVisitor.this.owner.getTypeId().getId(),
							product.getTypeId().getId());
			CompositeAttributeTypeVisitor.this.a.setAttributeId(new DatabaseIDSetState(newAsso.getId()));
		}
		
		@Override
		public void handle(final SumType sum) throws Exception {
			final UnidirectionalAssociation newAsso =
					AttributeGenerationTask.this.generator.createUnidirectionalAssociation(
							CompositeAttributeTypeVisitor.this.a.getName(),
							true,
							true,
							CompositeAttributeTypeVisitor.this.owner.getTypeId().getId(),
							sum.getTypeId().getId());
			CompositeAttributeTypeVisitor.this.a.setAttributeId(new DatabaseIDSetState(newAsso.getId()));
		}
		
		@Override
		public void handle(final ThrownType thrownType) throws Exception {
			throw new TaskException("A type that should be thrown is not possible in an attribute.");
		}
	}
}
