package de.fhdw.wtf.generator.transformer.transformers.classTransformer;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import de.fhdw.wtf.common.ast.Attribute;
import de.fhdw.wtf.common.ast.Constructor;
import de.fhdw.wtf.common.ast.ConstructorByReferenceState;
import de.fhdw.wtf.common.ast.ConstructorByTypeAndSignatureState;
import de.fhdw.wtf.common.ast.ConstructorInvalidState;
import de.fhdw.wtf.common.ast.ConstructorOrOperation;
import de.fhdw.wtf.common.ast.ConstructorReference;
import de.fhdw.wtf.common.ast.Group;
import de.fhdw.wtf.common.ast.Model;
import de.fhdw.wtf.common.ast.type.AtomicType;
import de.fhdw.wtf.common.ast.type.BaseType;
import de.fhdw.wtf.common.ast.type.ClassType;
import de.fhdw.wtf.common.ast.type.CompositeType;
import de.fhdw.wtf.common.ast.type.ProductElementType;
import de.fhdw.wtf.common.ast.type.ProductType;
import de.fhdw.wtf.common.ast.type.Type;
import de.fhdw.wtf.common.ast.type.TypeProxy;
import de.fhdw.wtf.common.ast.visitor.AtomicTypeVisitorReturnException;
import de.fhdw.wtf.common.ast.visitor.ConstructorReferenceStateVisitorReturnException;
import de.fhdw.wtf.common.ast.visitor.TypeVisitorReturnException;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.common.token.DummyToken;
import de.fhdw.wtf.generator.java.generatorModel.GenAnyType;
import de.fhdw.wtf.generator.java.generatorModel.GenAttributeModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenComment;
import de.fhdw.wtf.generator.java.generatorModel.GenException;
import de.fhdw.wtf.generator.java.generatorModel.GenExternalInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenFullParsedOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenHasNoGenericType;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceWithClassImplClass;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaAttribute;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenMutableMap;
import de.fhdw.wtf.generator.java.generatorModel.GenOperationModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenParameter;
import de.fhdw.wtf.generator.java.generatorModel.GenPrimitiveClass;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenStringType;
import de.fhdw.wtf.generator.java.generatorModel.GenType;
import de.fhdw.wtf.generator.java.generatorModel.GenTypeReferenceByName;
import de.fhdw.wtf.generator.java.generatorModel.GenTypeReferenceByReference;
import de.fhdw.wtf.generator.java.generatorModel.GenUnqualifiedPackage;
import de.fhdw.wtf.generator.java.generatorModel.GenUserClass;
import de.fhdw.wtf.generator.java.generatorModel.GenVisibility;
import de.fhdw.wtf.generator.java.generatorModel.GenVoidType;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;
import de.fhdw.wtf.generator.java.generatorModel.Generic;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorException;
import de.fhdw.wtf.generator.transformer.exception.GenTypeNotReferencedException;
import de.fhdw.wtf.walker.walker.SimpleWalkerTask;

/**
 * This Task generates the $super-methods and methods that are being called within $super (such as the getter for the
 * tuples) in every class. The $super-methods main purpose is to call the right constructors of super classes in the
 * right order.
 */
public class ConstructorCallGenerationTask extends SimpleWalkerTask {
	
	/**
	 * The generator model to add constructor calls to.
	 */
	private final GeneratorModel generatorModel;
	
	/**
	 * Contains the information which type is responsible to instantiate which other types.
	 */
	private final Map<Type, List<Type>> callOrder;
	
	/**
	 * This map contains the numbers of generic tuples to generate for the getter-Methods for super constructor
	 * parameters. The number stands for the amount of generic fields in the tuple.
	 */
	private final Map<Integer, GenUserClass> genericTuplesToGenerate;
	
	/**
	 * Constructor of {@link ConstructorCallGenerationTask}.
	 * 
	 * @param m
	 *            The model to traverse.
	 * @param taskmanager
	 *            The task manager to register with.
	 * @param generatorModel
	 *            the model, which will be checked.
	 */
	public ConstructorCallGenerationTask(final Model m,
			final TaskExecutor taskmanager,
			final GeneratorModel generatorModel) {
		super(m, taskmanager);
		this.callOrder = m.getConstructorCallDependencies();
		this.generatorModel = generatorModel;
		this.genericTuplesToGenerate = new HashMap<>();
	}
	
	@Override
	public void handleClass(final ClassType c) throws TaskException {
		final GenClass genClass = this.generatorModel.getClassMapping().get(c);
		genClass.accept(new GenClassVisitorException<TaskException>() {
			
			@Override
			public void handle(final GenClassClass classClass) throws TaskException {
				// ConstructorCallGenerationTask.this.handleGenClassClass(c, classClass);
			}
			
			@Override
			public void handle(final GenInterfaceClass interfaceClass) throws TaskException {
				interfaceClass.accept(new GenInterfaceClassVisitorException<TaskException>() {
					
					@Override
					public void handle(final GenSimpleInterfaceClass simpleInterface) throws TaskException {
						// Nothing to do.
					}
					
					@Override
					public void handle(final GenInterfaceWithClassImplClass interfaceWithImplClass)
							throws TaskException {
						ConstructorCallGenerationTask.this.handleGenInterfaceWithClassImplClass(
								c,
								interfaceWithImplClass);
					}
					
					@Override
					public void handle(final GenExternalInterfaceClass iface) throws TaskException {
						// Nothing to do.
					}
					
				});
			}
			
			@Override
			public void handle(final GenPrimitiveClass primitiveClass) {
				// Nothing to do.
			}
			
		});
	}
	
	/**
	 * Generates the constructors and the corresponding $super and setThis methods for the classType c and its
	 * GenInterfaceWithClassImplClass.
	 * 
	 * @param c
	 *            the ClassType.
	 * @param interfaceClass
	 *            The interface class for c.
	 * @throws TaskException
	 *             if the types don't match each other.
	 */
	private void handleGenInterfaceWithClassImplClass(final ClassType c,
			final GenInterfaceWithClassImplClass interfaceClass) throws TaskException {
		final GenClassClass genClassClass = interfaceClass.getClassRepresentation();
		final List<Type> typesToInstantiate =
				this.callOrder.containsKey(c) ? this.callOrder.get(c) : new LinkedList<Type>();
		int superMethodNumber = 0;
		final Collection<Constructor> constructors = c.getConstructors();
		final Iterator<Constructor> constructorIterator = constructors.iterator();
		while (constructorIterator.hasNext()) {
			superMethodNumber++;
			final Constructor constructor = constructorIterator.next();
			this.generateConstructorAndSuper(
					genClassClass,
					interfaceClass,
					constructor.getParameters(),
					constructor.getSuperConstructors(),
					typesToInstantiate,
					superMethodNumber);
		}
		this.generateSetThis(genClassClass, interfaceClass, typesToInstantiate);
	}
	
	/**
	 * generates the setThis-Method.
	 * 
	 * @param genClassClass
	 *            the GenClassClass.
	 * @param interfaceClass
	 *            the InterfaceClass for genClassClass.
	 * @param typesToInstantiate
	 *            the types that have to be instantiated.
	 * @throws GenTypeNotReferencedException
	 *             if no GenType for a super type can be found.
	 */
	private void generateSetThis(final GenClassClass genClassClass,
			final GenInterfaceWithClassImplClass interfaceClass,
			final List<Type> typesToInstantiate) throws GenTypeNotReferencedException {
		final StringBuilder setThisMethodBuilder = new StringBuilder("this.This=This;");
		final Iterator<Type> superTypes = typesToInstantiate.iterator();
		while (superTypes.hasNext()) {
			final Type superType = superTypes.next();
			final GenType genType = this.generatorModel.getGenTypeForType(superType);
			final String typeName = genType.getFullyQualifiedTypeName();
			final String typeNameImpl = typeName + "$Impl";
			final String typeNameImplShort = typeNameImpl.substring(typeNameImpl.lastIndexOf('.') + 1);
			final String innerPackage = typeName + "_InnerPackage." + typeNameImplShort;
			
			setThisMethodBuilder.append("\n\t\t");
			setThisMethodBuilder.append("((");
			setThisMethodBuilder.append(innerPackage);
			setThisMethodBuilder.append(")");
			setThisMethodBuilder.append("$generatedObjects.get(new de.fhdw.wtf.context.model.Str(\"");
			setThisMethodBuilder.append(innerPackage);
			setThisMethodBuilder.append("\"))).setThis(this);");
		}
		
		final List<GenParameter> setThisParams = new Vector<>();
		setThisParams.add(GenParameter.create(
				"This",
				GenTypeReferenceByName.create(interfaceClass.getFullyQualifiedTypeName())));
		final GenJavaOperation setThisOperation =
				GenJavaOperation.create("setThis", setThisParams, GenFullParsedOperationState.create(
						GenComment.create("/**\n\t * Sets This.\n\t **/"),
						new Vector<GenException>(),
						GenTypeReferenceByName.create(GenVoidType.getInstance().getFullyQualifiedTypeName()),
						new Vector<GenOperationModifier>(),
						GenVisibility.PUBLIC,
						setThisMethodBuilder.toString()));
		genClassClass.addOperation(setThisOperation);
	}
	
	/**
	 * Generates a constructor for genClassClass.
	 * 
	 * @param genClassClass
	 *            the GenClassClass.
	 * @param interfaceClass
	 *            the InterfaceClass for genClassClass.
	 * @param parameters
	 *            the parameters of the constructor to create.
	 * @param superConstructors
	 *            the super constructors to call.
	 * @param typesToInstantiate
	 *            the types that have to be instantiated.
	 * @param superMethodNumber
	 *            the number of the currently generated $super-method
	 * @throws TaskException
	 *             if the types don't match each other.
	 */
	private void generateConstructorAndSuper(final GenClassClass genClassClass,
			final GenInterfaceClass interfaceClass,
			final ProductType parameters,
			final Collection<ConstructorReference> superConstructors,
			final List<Type> typesToInstantiate,
			final int superMethodNumber) throws TaskException {
		final GenParameter generatedObjects =
				GenParameter.create(
						"$generatedObjects",
						GenMutableMap.create(GenStringType.getInstance(), GenAnyType.getInstance()));
		final List<GenParameter> genParameters = this.createGenParameters(parameters);
		final List<GenParameter> genParametersWithMap = new LinkedList<>();
		genParametersWithMap.add(generatedObjects);
		genParametersWithMap.addAll(genParameters);
		final String prefix = genParameters.isEmpty() ? "" : ", ";
		this.generateConstructor(
				genClassClass,
				genParameters,
				"this("
						+ "new de.fhdw.wtf.context.model.collections.MutableMap<de.fhdw.wtf.context.model.Str,de.fhdw.wtf.context.model.AnyType>()"
						+ this.generateParameterString(prefix, genParameters) + ";",
				GenVisibility.PUBLIC);
		this.generateConstructor(
				genClassClass,
				genParametersWithMap,
				this.generateSuperCall(genParametersWithMap, genParameters),
				GenVisibility.PUBLIC);
		this.generateSuper(
				genClassClass,
				interfaceClass,
				genParametersWithMap,
				genParameters,
				superConstructors,
				typesToInstantiate,
				superMethodNumber);
		this.generateInitializeOnCreation(genClassClass, genParameters);
	}
	
	private void generateInitializeOnCreation(final GenClassClass genClassClass, final List<GenParameter> genParameters) {
		genClassClass.addOperation(GenJavaOperation.create(
				"initializeOnCreation",
				genParameters,
				GenFullParsedOperationState.create(
						GenComment.create("/**\n\t * Initializes the objects fields.\n\t */"),
						new Vector<GenException>(),
						GenTypeReferenceByName.create("void"),
						new Vector<GenOperationModifier>(),
						GenVisibility.PRIVATE,
						"// TODO implement initializeOnCreation")));
		
	}
	
	/**
	 * Generates the $super-Method for a list of super constructors.
	 * 
	 * @param genClassClass
	 *            the GenClassClass.
	 * @param interfaceClass
	 *            the InterfaceClass for genClassClass.
	 * @param genParametersWithMap
	 *            The parameters of $super.
	 * @param genParametersWithoutMap
	 *            The parameters to pass to the tuple getters.
	 * @param superConstructors
	 *            the super constructors to call.
	 * @param typesToInstantiate
	 *            the types that have to be instantiated.
	 * @param superMethodNumber
	 *            the number of the currently generated $super-method
	 * @throws TaskException
	 *             if the types don't match each other.
	 */
	private void generateSuper(final GenClassClass genClassClass,
			final GenInterfaceClass interfaceClass,
			final List<GenParameter> genParametersWithMap,
			final List<GenParameter> genParametersWithoutMap,
			final Collection<ConstructorReference> superConstructors,
			final List<Type> typesToInstantiate,
			final int superMethodNumber) throws TaskException {
		final StringBuilder superMethodBuilder = new StringBuilder();
		int superTypeNumber = 0;
		final Iterator<Type> superTypes = typesToInstantiate.iterator();
		while (superTypes.hasNext()) {
			superTypeNumber++;
			final Type superType = superTypes.next();
			final GenType genType = this.generatorModel.getGenTypeForType(superType);
			final Constructor constructorToCall = this.getConstructorForType(superType, superConstructors);
			final List<GenParameter> constructorParametersWithoutMap =
					this.createGenParameters(constructorToCall.getParameters());
			final String typeName = genType.getFullyQualifiedTypeName();
			final String tupleName = "tuple_" + superMethodNumber + "_" + superTypeNumber;
			final String typeNameImpl = typeName + "$Impl";
			final String typeNameImplShort = typeNameImpl.substring(typeNameImpl.lastIndexOf('.') + 1);
			final String innerPackage = typeName + "_InnerPackage." + typeNameImplShort;
			superMethodBuilder.append("if ($generatedObjects.get(new de.fhdw.wtf.context.model.Str(\"");
			superMethodBuilder.append(innerPackage);
			superMethodBuilder.append("\"))==null){\n\t\t\t\t");
			if (!constructorParametersWithoutMap.isEmpty()) {
				superMethodBuilder.append(this.generateTupleInstantiationStringAndTupleClassAndTupleGetter(
						genClassClass,
						tupleName,
						typeName,
						constructorParametersWithoutMap,
						genParametersWithoutMap,
						superMethodNumber,
						superTypeNumber));
				superMethodBuilder.append("\n\t\t\t\t");
			}
			superMethodBuilder.append("this.$generatedObjects.put(new de.fhdw.wtf.context.model.Str(\"");
			superMethodBuilder.append(innerPackage);
			superMethodBuilder.append("\")");
			superMethodBuilder.append(", ");
			superMethodBuilder.append("new ");
			superMethodBuilder.append(innerPackage);
			superMethodBuilder.append("(");
			superMethodBuilder.append("$generatedObjects");
			if (!constructorParametersWithoutMap.isEmpty()) {
				superMethodBuilder.append(", ");
				superMethodBuilder.append(this.generateTupleGetterAndCallString(
						tupleName,
						constructorParametersWithoutMap));
			} else {
				superMethodBuilder.append(")");
			}
			superMethodBuilder.append(");");
			superMethodBuilder.append("\n\t\t\t}\n\t\t");
			superMethodBuilder.append("else{\n\t\t\t\t");
			superMethodBuilder.append("this.$generatedObjects.put(new de.fhdw.wtf.context.model.Str(\"");
			superMethodBuilder.append(innerPackage);
			superMethodBuilder.append("\")");
			superMethodBuilder.append(", ");
			superMethodBuilder.append("$generatedObjects.get(new de.fhdw.wtf.context.model.Str(\"");
			superMethodBuilder.append(innerPackage);
			superMethodBuilder.append("\")));\n\t\t\t}\n\t\t");
		}
		superMethodBuilder.append("this.$generatedObjects.put(new de.fhdw.wtf.context.model.Str(\"");
		superMethodBuilder.append(genClassClass.getFullyQualifiedTypeName());
		superMethodBuilder.append("\"),this);");
		superMethodBuilder.append("\n\t\t");
		superMethodBuilder.append("$generatedObjects.put(new de.fhdw.wtf.context.model.Str(\"");
		superMethodBuilder.append(genClassClass.getFullyQualifiedTypeName());
		superMethodBuilder.append("\"),this);");
		superMethodBuilder.append("\n\t\t");
		superMethodBuilder.append("this.setThis(this);");
		
		final GenJavaOperation superOperation =
				GenJavaOperation.create(
						"$super",
						GenVisibility.PRIVATE,
						genParametersWithMap,
						new LinkedList<GenException>(),
						superMethodBuilder.toString(),
						GenVoidType.getInstance(),
						new LinkedList<GenOperationModifier>(),
						GenComment.create("/**\n\t * Instantiates objects for the super types of this class.\n\t **/"));
		genClassClass.addOperation(superOperation);
	}
	
	/**
	 * Generates a tuple with the same amount of generic fields as the size of constructorParametersWithoutMap (if it
	 * does not already exist), a matching getter method in genClassClass that returns a tuple instantiation filled with
	 * values matching the types of constructorParametersWithoutMap and a call-string to that getter-method including a
	 * tuple reference with the name tuple_superMethodNumber_superTypeNumber.
	 * 
	 * @param genClassClass
	 *            the class to generate $super for.
	 * @param tupleName
	 *            the name of the tuple reference to generate
	 * @param superTypeName
	 *            the name of the super type for the todo of the getter.
	 * @param constructorParametersWithoutMap
	 *            the parameters of the super type constructor without $generatedObjects.
	 * @param genParametersWithoutMap
	 *            the parameters of $super without $generatedObjects.
	 * @param superMethodNumber
	 *            the number of the super method.
	 * @param superTypeNumber
	 *            the number of the super type.
	 * @return a call-string to that getter-method including a tuple reference with the name
	 *         tuple_superMethodNumber_superTypeNumber.
	 */
	private String generateTupleInstantiationStringAndTupleClassAndTupleGetter(final GenClassClass genClassClass,
			final String tupleName,
			final String superTypeName,
			final List<GenParameter> constructorParametersWithoutMap,
			final List<GenParameter> genParametersWithoutMap,
			final int superMethodNumber,
			final int superTypeNumber) {
		final StringBuilder resultStringBuilder = new StringBuilder();
		final StringBuilder genericListStringBuilder = new StringBuilder();
		final Iterator<GenParameter> parameters = constructorParametersWithoutMap.iterator();
		while (parameters.hasNext()) {
			final GenParameter current = parameters.next();
			genericListStringBuilder.append(current.getTyp().getFullyQualifiedNameWithGenericArguments());
			if (parameters.hasNext()) {
				genericListStringBuilder.append(", ");
			}
		}
		final int genericCount = constructorParametersWithoutMap.size();
		final String superParameterString = this.generateParameterString("", genParametersWithoutMap);
		final String genericString = "<" + genericListStringBuilder.toString() + "> ";
		final String tupleTypePrefix = "generated.Tuple" + genericCount;
		final String tupleTypeString = tupleTypePrefix + genericString;
		final String getterName = "getTuple_" + superMethodNumber + "_" + superTypeNumber;
		resultStringBuilder.append(tupleTypeString);
		resultStringBuilder.append(tupleName);
		resultStringBuilder.append(" = ");
		resultStringBuilder.append("this.");
		resultStringBuilder.append(getterName);
		resultStringBuilder.append("(");
		resultStringBuilder.append(superParameterString);
		resultStringBuilder.append(";");
		final GenUserClass tuple = this.generateTupleClass(genericCount);
		this.generateTupleGetterMethod(
				genClassClass,
				getterName,
				genParametersWithoutMap,
				constructorParametersWithoutMap,
				tupleTypeString,
				tupleTypePrefix,
				genericString,
				tuple,
				superTypeName);
		return resultStringBuilder.toString();
	}
	
	/**
	 * Generates a tuple class with name Tuple<genericCount> and genericCount generic fields if it does not exist
	 * already.
	 * 
	 * @param genericCount
	 *            the amount of generic types.
	 * @return a tuple class with name Tuple<genericCount> and genericCount generic fields.
	 */
	private GenUserClass generateTupleClass(final int genericCount) {
		if (this.genericTuplesToGenerate.containsKey(genericCount)) {
			return this.genericTuplesToGenerate.get(genericCount);
		}
		final List<Generic> generics = new Vector<>();
		final List<GenJavaOperation> operations = new Vector<>();
		final List<GenJavaAttribute> attributes = new Vector<>();
		final List<GenParameter> constructorParameters = new Vector<>();
		final StringBuilder constructorMethodBuilder = new StringBuilder();
		for (int i = 1; i <= genericCount; i++) {
			final String genericTypeName = "ValueType" + i;
			final String genericAttributeNameSuffix = "alue" + i;
			final String genericAttributeName = "v" + genericAttributeNameSuffix;
			final String genericOperationNameSuffix = "V" + genericAttributeNameSuffix;
			final Generic generic = Generic.create(genericTypeName, GenHasNoGenericType.create());
			generics.add(generic);
			final GenJavaOperation operation =
					GenJavaOperation.create(
							"get" + genericOperationNameSuffix,
							new Vector<GenParameter>(),
							GenFullParsedOperationState.create(
									GenComment.create("/**\n\t * Returns the value of type " + genericTypeName
											+ ".\n\t **/"),
									new Vector<GenException>(),
									GenTypeReferenceByReference.create(generic),
									new Vector<GenOperationModifier>(),
									GenVisibility.PUBLIC,
									"return this." + genericAttributeName + ";"));
			operations.add(operation);
			final GenJavaAttribute attribute =
					GenJavaAttribute.create(
							genericAttributeName,
							GenVisibility.PRIVATE,
							generic,
							new Vector<GenAttributeModifier>());
			attributes.add(attribute);
			final GenParameter parameter = GenParameter.create(genericAttributeName, generic);
			constructorParameters.add(parameter);
			constructorMethodBuilder
					.append("this." + genericAttributeName + " = " + genericAttributeName + ";\n\t\t\t");
		}
		
		final GenUserClass tuple =
				GenUserClass.create(
						"Tuple" + genericCount,
						operations,
						new Vector<GenInterfaceClass>(),
						attributes,
						new Vector<GenClassModifier>(),
						new Vector<GenJavaOperation>(),
						null,
						GenUnqualifiedPackage.create("generated"),
						GenComment.create("/**\n* A tuple with " + genericCount + " generic fields.\n**/"),
						"");
		
		tuple.getGenerics().addAll(generics);
		
		final GenJavaOperation constructor =
				GenJavaOperation.create(
						"",
						constructorParameters,
						GenFullParsedOperationState.create(
								GenComment.create("/**\n\t * Creates a new Tuple" + genericCount + ".\n\t **/"),
								new Vector<GenException>(),
								GenTypeReferenceByReference.create(tuple),
								new Vector<GenOperationModifier>(),
								GenVisibility.PUBLIC,
								constructorMethodBuilder.toString()));
		tuple.getConstructors().add(constructor);
		
		this.generatorModel.addNonAstClass(tuple);
		this.genericTuplesToGenerate.put(genericCount, tuple);
		return tuple;
	}
	
	/**
	 * Generates the getter method for the tuple of a concrete super constructor call. The generated method enables the
	 * user to provide the parameters for the super constructor call in a tuple.
	 * 
	 * @param genClassClass
	 *            the class with $super.
	 * @param getterName
	 *            the name of the getter.
	 * @param genParametersWithoutMap
	 *            the parameters of $super without $generatedObjects.
	 * @param constructorParametersWithoutMap
	 *            the parameters of the super constructor without $generatedObjects.
	 * @param tupleTypeString
	 *            the concrete type of the tuple as string.
	 * @param tupleTypePrefix
	 *            the name of the tuple type as string.
	 * @param genericString
	 *            the string with the generic types of the tuple.
	 * @param returnType
	 *            the return type of the getter (the tuple itself).
	 * @param superTypeName
	 *            the name of the super type for the todo comment.
	 */
	private void generateTupleGetterMethod(final GenClassClass genClassClass,
			final String getterName,
			final List<GenParameter> genParametersWithoutMap,
			final List<GenParameter> constructorParametersWithoutMap,
			final String tupleTypeString,
			final String tupleTypePrefix,
			final String genericString,
			final GenType returnType,
			final String superTypeName) {
		final StringBuilder methodStringBuilder =
				new StringBuilder(
						"//TODO replace the following null values with your desired values for the constructor of the super type "
								+ superTypeName);
		methodStringBuilder.append("\n\t\t");
		methodStringBuilder.append(tupleTypeString);
		methodStringBuilder.append(" tuple ");
		methodStringBuilder.append("= new ");
		methodStringBuilder.append(tupleTypePrefix);
		methodStringBuilder.append("<>(" + this.generateNullStringWithCasts(constructorParametersWithoutMap) + ");");
		methodStringBuilder.append("\n\t\treturn tuple;");
		
		final GenJavaOperation tupleGetterMethod =
				GenJavaOperation.create(
						genericString + " " + getterName,
						genParametersWithoutMap,
						GenFullParsedOperationState.create(
								GenComment.create("/**\n\t * Returns a tuple with the parameters for a "
										+ "super constructor." + "\n\t * "
										+ "The name of this operation describes the signature of "
										+ "said constructor." + "\n\t **/"),
								new Vector<GenException>(),
								GenTypeReferenceByReference.create(returnType),
								new Vector<GenOperationModifier>(),
								GenVisibility.PRIVATE,
								methodStringBuilder.toString()));
		genClassClass.addOperation(tupleGetterMethod);
		
	}
	
	/**
	 * Generates a comma separated parameter string starting with '(' and ending with ')' with as many null values as
	 * constructorParametersWithoutMap has entries. Adds a cast to the matching parameter type from
	 * constructorParametersWithoutMap before each null value.
	 * 
	 * @param constructorParametersWithoutMap
	 *            the parameters to use.
	 * @return a comma separated parameter string starting with '(' and ending with ')' with as many null values as
	 *         constructorParametersWithoutMap has entries.
	 */
	private String generateNullStringWithCasts(final List<GenParameter> constructorParametersWithoutMap) {
		final StringBuilder resultBuilder = new StringBuilder();
		final Iterator<GenParameter> parameters = constructorParametersWithoutMap.iterator();
		while (parameters.hasNext()) {
			resultBuilder.append("(");
			resultBuilder.append(parameters.next().getTyp().getFullyQualifiedNameWithGenericArguments());
			resultBuilder.append(") null");
			if (parameters.hasNext()) {
				resultBuilder.append(",");
			}
		}
		return resultBuilder.toString();
	}
	
	/**
	 * Searches for a constructor in superConstructors that matches superType.
	 * 
	 * @param superType
	 *            the superType.
	 * @param superConstructors
	 *            a list of super constructors to call.
	 * @return the matching constructor.
	 * @throws TaskException
	 *             is being thrown when no matching constructor can be found in superConstructors.
	 */
	private Constructor getConstructorForType(final Type superType,
			final Collection<ConstructorReference> superConstructors) throws TaskException {
		if (superConstructors.isEmpty()) {
			return Constructor.create(
					ProductType.create(DummyToken.getInstance()),
					this.getTypeAsClassType(superType),
					new LinkedList<ConstructorReference>(),
					DummyToken.getInstance(),
					DummyToken.getInstance());
		}
		final Iterator<ConstructorReference> constructorReferences = superConstructors.iterator();
		boolean found = false;
		Constructor constructor = null;
		while (!found && constructorReferences.hasNext()) {
			constructor = this.getConstructorFromConstructorReference(constructorReferences.next());
			found = constructor.getReturnType().equals(superType);
		}
		if (!found) {
			throw new TaskException(
					"ConstructorCallGenerationTask: No constructor found while trying to generate code snipped for: "
							+ superType + " in $super!");
		}
		return constructor;
	}
	
	/**
	 * Tries to find a ClassType for superType.
	 * 
	 * @param superType
	 *            the type.
	 * @return a ClassType that is representative for superType.
	 * @throws TaskException
	 *             if no ClassType for the super type can be found.
	 */
	private ClassType getTypeAsClassType(final Type superType) throws TaskException {
		return superType.accept(new ClassTypeFinder());
	}
	
	/**
	 * Tries to find a ClassType for superType.
	 */
	private static class ClassTypeFinder implements TypeVisitorReturnException<ClassType, TaskException> {
		
		@Override
		public ClassType handle(final AtomicType atomicType) throws TaskException {
			return atomicType.accept(new AtomicTypeVisitorReturnException<ClassType, TaskException>() {
				
				@Override
				public ClassType handle(final BaseType baseType) throws TaskException {
					throw new TaskException("ConstructorCallGenerationTask: No BaseType allowed here!");
				}
				
				@Override
				public ClassType handle(final ClassType clazz) throws TaskException {
					return clazz;
				}
				
			});
		}
		
		@Override
		public ClassType handle(final CompositeType compositeType) throws TaskException {
			throw new TaskException("ConstructorCallGenerationTask: No SumType allowed here!");
		}
		
		@Override
		public ClassType handle(final TypeProxy typeProxy) throws TaskException {
			return typeProxy.getTarget().accept(this);
		}
		
	}
	
	/**
	 * Gets the constructor for reference.
	 * 
	 * @param reference
	 *            the reference.
	 * @return a constructor that is referenced via reference.
	 * @throws TaskException
	 *             if no constructor can be found (Wrong reference state).
	 */
	private Constructor getConstructorFromConstructorReference(final ConstructorReference reference)
			throws TaskException {
		/**
		 * Message if reference is in wrong state.
		 */
		final String exceptionMessage =
				"ConstructorCallGenerationTask: ConstructorReference"
						+ " needs to be in ConstructorByReferenceState at this point. Current state: ";
		
		return reference.getState().accept(
				new ConstructorReferenceStateVisitorReturnException<Constructor, TaskException>() {
					
					@Override
					public Constructor handle(final ConstructorByTypeAndSignatureState byName) throws TaskException {
						throw new TaskException(exceptionMessage + byName);
					}
					
					@Override
					public Constructor handle(final ConstructorInvalidState invalid) throws TaskException {
						throw new TaskException(exceptionMessage + invalid);
					}
					
					@Override
					public Constructor handle(final ConstructorByReferenceState byReference) {
						return byReference.getTarget();
					}
					
				});
	}
	
	/**
	 * Creates GenParameters for parameters.
	 * 
	 * @param parameters
	 *            the parameters to generate.
	 * @return GenParameters that match parameters.
	 * @throws GenTypeNotReferencedException
	 *             if no genType for a parameter type can be found.
	 */
	private List<GenParameter> createGenParameters(final ProductType parameters) throws GenTypeNotReferencedException {
		final List<ProductElementType> elements = parameters.getElements();
		final List<GenParameter> result = new LinkedList<>();
		for (final ProductElementType productElement : elements) {
			final GenType paramType = this.generatorModel.getGenTypeForType(productElement.getType());
			result.add(GenParameter.create(productElement.getName(), paramType));
		}
		return result;
	}
	
	/**
	 * D Generates a constructor.
	 * 
	 * @param genClassClass
	 *            for class.
	 * @param genParameters
	 *            with parameters.
	 * @param methodBody
	 *            with implementation.
	 * @param visibility
	 */
	private void generateConstructor(final GenClassClass genClassClass,
			final List<GenParameter> genParameters,
			final String methodBody,
			final GenVisibility visibility) {
		final GenComment comment =
				GenComment
						.create("/** \n\t * " + "Creates an instance of type " + genClassClass.getName() + "\n\t **/");
		final GenJavaOperation constructorWithMap =
				GenJavaOperation.create(
						"",
						visibility,
						genParameters,
						new Vector<GenException>(),
						methodBody,
						genClassClass,
						new Vector<GenOperationModifier>(),
						comment);
		genClassClass.addConstructor(constructorWithMap);
	}
	
	/**
	 * Generates a call to a $super-Method with genParameters.
	 * 
	 * @param genParametersWithMap
	 *            will be added (as names) to the call.
	 * @param genParameters
	 * @return a $super-call.
	 */
	private String generateSuperCall(final List<GenParameter> genParametersWithMap,
			final List<GenParameter> genParameters) {
		return "this.$super(" + this.generateParameterString("", genParametersWithMap) + ";"
				+ "\n\t\tthis.initializeOnCreation(" + this.generateParameterString("", genParameters) + ";";
	}
	
	/**
	 * Generates a String with the names of the parameters (comma separated and with a ')' at the end).
	 * 
	 * @param prefix
	 *            THe Prefix to add to the list.
	 * 
	 * @param genParameters
	 *            parameters to include.
	 * @return the parameter-string.
	 */
	private String generateParameterString(final String prefix, final List<GenParameter> genParameters) {
		final StringBuilder superCall = new StringBuilder(prefix);
		final Iterator<GenParameter> parameterIterator = genParameters.iterator();
		while (parameterIterator.hasNext()) {
			final GenParameter parameter = parameterIterator.next();
			superCall.append(parameter.getName());
			if (parameterIterator.hasNext()) {
				superCall.append(", ");
			}
		}
		superCall.append(")");
		return superCall.toString();
	}
	
	private String generateTupleGetterAndCallString(final String tupleName,
			final List<GenParameter> constructorParametersWithoutMap) {
		final StringBuilder call = new StringBuilder();
		int genericFieldCount = 0;
		final Iterator<GenParameter> parameters = constructorParametersWithoutMap.iterator();
		while (parameters.hasNext()) {
			genericFieldCount++;
			final GenParameter current = parameters.next();
			call.append(tupleName);
			call.append(".<");
			call.append(current.getTyp().getFullyQualifiedNameWithGenericArguments());
			call.append(">getValue");
			call.append(genericFieldCount);
			call.append("(");
			call.append(")");
			if (parameters.hasNext()) {
				call.append(", ");
			}
		}
		call.append(")");
		return call.toString();
	}
	
	@Override
	public void handleGroup(final Group g) throws TaskException {
		// Nothing to do.
		
	}
	
	@Override
	public void handleAttribute(final Attribute a, final ClassType owner) throws TaskException {
		// Nothing to do.
	}
	
	@Override
	public void handleConstructorOrOperation(final ConstructorOrOperation coo, final ClassType owner)
			throws TaskException {
		// Nothing to do.
	}
	
	@Override
	public void finalizeTask() throws TaskException {
		// Nothing to do.
	}
	
	@Override
	public void beginTask() throws TaskException {
		// Nothing to do.
	}
	
}
