package de.fhdw.wtf.generator.transformer.transformers.classTransformer;

import java.util.Collection;

import de.fhdw.wtf.common.ast.Model;
import de.fhdw.wtf.common.ast.type.AtomicType;
import de.fhdw.wtf.common.ast.type.CompositeType;
import de.fhdw.wtf.common.ast.type.ListType;
import de.fhdw.wtf.common.ast.type.MapType;
import de.fhdw.wtf.common.ast.type.ProductType;
import de.fhdw.wtf.common.ast.type.SumType;
import de.fhdw.wtf.common.ast.type.ThrownType;
import de.fhdw.wtf.common.ast.type.Type;
import de.fhdw.wtf.common.ast.type.TypeProxy;
import de.fhdw.wtf.common.ast.visitor.CompositeTypeVisitorReturn;
import de.fhdw.wtf.common.ast.visitor.TypeVisitorReturn;
import de.fhdw.wtf.common.exception.walker.CyclicDependencyException;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.generator.java.generatorModel.GenAspect;
import de.fhdw.wtf.generator.java.generatorModel.GenClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenCollectionType;
import de.fhdw.wtf.generator.java.generatorModel.GenDeclareInheritance;
import de.fhdw.wtf.generator.java.generatorModel.GenDummyType;
import de.fhdw.wtf.generator.java.generatorModel.GenExternalInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenImportType;
import de.fhdw.wtf.generator.java.generatorModel.GenIntegerType;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceWithClassImplClass;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaUtilCollection;
import de.fhdw.wtf.generator.java.generatorModel.GenMapType;
import de.fhdw.wtf.generator.java.generatorModel.GenPrimitiveClass;
import de.fhdw.wtf.generator.java.generatorModel.GenPrimitiveType;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenStringType;
import de.fhdw.wtf.generator.java.generatorModel.GenType;
import de.fhdw.wtf.generator.java.generatorModel.GenVoidType;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;
import de.fhdw.wtf.generator.java.generatorModel.Generic;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitor;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitor;
import de.fhdw.wtf.generator.java.visitor.GenPrimitiveTypeVisitor;
import de.fhdw.wtf.generator.java.visitor.GenTypeVisitor;
import de.fhdw.wtf.generator.transformer.exception.GenTypeNotReferencedException;
import de.fhdw.wtf.walker.walker.HelperUtils;
import de.fhdw.wtf.walker.walker.SimpleWalkerTaskForTypes;

/**
 * This Transformer for inheritance adds all implements or extends relations to the {@link GeneratorModel}. If a
 * {@link Class} has more than one superclass declared in wtf only the last one will be recognized.
 */
public class InheritanceTransformer extends SimpleWalkerTaskForTypes {
	
	private final GeneratorModel generatorModel;
	
	/**
	 * Constructor of {@link InheritanceTransformer}.
	 * 
	 * @param m
	 *            The model to traverse.
	 * @param taskmanager
	 *            The task manager to register with.
	 * @param generatorModel
	 *            the model, which will be checked.
	 */
	public InheritanceTransformer(final Model m,
			final TaskExecutor taskmanager,
			final GeneratorModel generatorModel,
			final OperationAttributeTransformer operationAttributeTransformer) {
		super(m, taskmanager, true);
		this.generatorModel = generatorModel;
		
		try {
			this.addDependency(operationAttributeTransformer);
		} catch (final CyclicDependencyException e) {
			// should not happen
			e.printStackTrace();
			throw new Error("Dependency tasks are cyclic in InheritanceTransformer.");
		}
	}
	
	/**
	 * Handles inheritance for products, classes and interfaces.
	 */
	@Override
	public void handleType(final Type c) throws TaskException {
		// new realization
		
		if (this.isEmptySum(c)) {
			// The EmptySum will not be generated, because "void" represents it
			return;
		}
		
		final Collection<Type> superTypes = c.getSuperTypes();
		
		for (final Type superType : superTypes) {
			this.createInheritance(c, superType);
		}
	}
	
	private boolean isEmptySum(final Type c) {
		return c.accept(new TypeVisitorReturn<Boolean>() {
			
			@Override
			public Boolean handle(final AtomicType atomicType) {
				return false;
			}
			
			@Override
			public Boolean handle(final CompositeType compositeType) {
				return compositeType.accept(new CompositeTypeVisitorReturn<Boolean>() {
					
					@Override
					public Boolean handle(final ListType list) {
						return false;
					}
					
					@Override
					public Boolean handle(final MapType map) {
						return false;
					}
					
					@Override
					public Boolean handle(final ProductType product) {
						return false;
					}
					
					@Override
					public Boolean handle(final SumType sum) {
						return Integer.valueOf(0).equals(sum.getElementsSizeWithoutThrownTypes()); // Hier vielleicht
																									// doch normal size.
					}
					
					@Override
					public Boolean handle(final ThrownType thrownType) {
						return false;
					}
				});
			}
			
			@Override
			public Boolean handle(final TypeProxy typeProxy) {
				return InheritanceTransformer.this.isEmptySum(HelperUtils.getTargetType(typeProxy));
			}
		});
	}
	
	/**
	 * Creates the inheritance from the {@link GenType}-representation for <code>subType</code> to the {@link GenType}
	 * -representation for <code>superType</code>.
	 *
	 * @param subType
	 *            The {@link Type} that is a specialization of superType.
	 * @param superType
	 *            The {@link Type} that is a generalization of subType.
	 * @throws GenTypeNotReferencedException
	 *             Thrown when sub- or superType is not referenced in the Type-mapping of the generatorModel.
	 */
	private void createInheritance(final Type subType, final Type superType) throws GenTypeNotReferencedException {
		final GenType genSubType = this.generatorModel.getGenTypeForType(subType);
		final GenType genSuperType = this.generatorModel.getGenTypeForType(superType);
		genSubType.accept(new GenTypeVisitor() {
			
			@Override
			public void handle(final GenClass genSubClass) {
				InheritanceTransformer.this.createInheritanceFromGenClass(genSubClass, genSuperType);
			}
			
			@Override
			public void handle(final GenPrimitiveType primitiveType) {
				primitiveType.accept(new GenPrimitiveTypeVisitor() {
					
					@Override
					public void handle(final GenVoidType voidType) {
						throw new Error("Inheritance shall not be generated for GenVoidType.");
					}
					
					@Override
					public void handle(final GenStringType stringType) {
						InheritanceTransformer.this.createInheritanceFromGenClass(
								GenStringType.getCorrespondingClass(),
								genSuperType);
					}
					
					@Override
					public void handle(final GenIntegerType integerType) {
						InheritanceTransformer.this.createInheritanceFromGenClass(
								GenIntegerType.getCorrespondingClass(),
								genSuperType);
					}
				});
			}
			
			@Override
			public void handle(final GenDummyType dummy) {
				throw new Error("Inheritance shall not be generated for GenDummyType.");
			}
			
			@Override
			public void handle(final GenJavaUtilCollection javaUtilCollection) {
				throw new Error("Inheritance shall not be generated for GenJavaUtilCollection.");
			}
			
			@Override
			public void handle(final Generic generic) {
				throw new Error("Inheritance shall not be generated for Generic.");
			}
			
			@Override
			public void handle(final GenImportType importType) {
				throw new Error("Inheritance shall not be generated for GenImportType.");
			}
			
			@Override
			public void handle(final GenMapType mapType) {
				// map -> ?
				// generate inheritance for maps
				// inheritance will not be generated since maps are not represented by own classes
			}
			
			@Override
			public void handle(final GenCollectionType collectionType) {
				// collection -> ?
				// generate inheritance for collections
				// inheritance will not be generated since maps are not represented by own classes
			}
		});
	}
	
	/**
	 * Creates the inheritance from a {@link GenClass} to a {@link GenType}.
	 *
	 * @param genSubClass
	 *            The {@link GenClass} that shall be a specialization of genSuperType.
	 * @param genSuperType
	 *            The {@link GenType} that shall be a generalization of genSubClass.
	 */
	private void createInheritanceFromGenClass(final GenClass genSubClass, final GenType genSuperType) {
		genSuperType.accept(new GenTypeVisitor() {
			
			@Override
			public void handle(final GenClass genSuperClass) {
				InheritanceTransformer.this.createInheritanceFromGenClassToGenClass(genSubClass, genSuperClass);
			}
			
			@Override
			public void handle(final GenPrimitiveType primitiveType) {
				// The class-representation of primitive Types are GenClasses. Therefore the inheritance needs to be
				// realized on those class-representations.
				// throw new Error("Inheritance shall not be generated if the superType is a GenPrimitiveType.");
			}
			
			@Override
			public void handle(final GenDummyType dummy) {
				throw new Error("Inheritance shall not be generated for GenDummyType.");
			}
			
			@Override
			public void handle(final GenJavaUtilCollection javaUtilCollection) {
				throw new Error("Inheritance shall not be generated for GenJavaUtilCollection.");
			}
			
			@Override
			public void handle(final Generic generic) {
				throw new Error("Inheritance shall not be generated for Generic.");
			}
			
			@Override
			public void handle(final GenImportType importType) {
				throw new Error("Inheritance shall not be generated for GenImportType.");
			}
			
			@Override
			public void handle(final GenMapType mapType) {
				throw new Error("Inheritance shall not be generated for GenMapType.");
			}
			
			@Override
			public void handle(final GenCollectionType collectionType) {
				throw new Error("Inheritance shall not be generated for GenCollectionType.");
			}
		});
	}
	
	/**
	 * Creates the inheritance from a {@link GenClass} to a {@link GenClass}.
	 *
	 * @param genSubClass
	 *            The {@link GenClass} that shall be a specialization of genSuperClass.
	 * @param genSuperClass
	 *            The {@link GenClass} that shall be a generalization of genSubClass.
	 */
	private void createInheritanceFromGenClassToGenClass(final GenClass genSubClass, final GenClass genSuperClass) {
		genSuperClass.accept(new GenClassVisitor() {
			
			@Override
			public void handle(final GenInterfaceClass genSuperInterfaceClass) {
				// class/interface -> interface
				genSubClass.accept(new GenClassVisitor() {
					
					@Override
					public void handle(final GenPrimitiveClass genSubPrimitiveClass) {
						// class (of primitive type) -> interface
						final GenAspect aspectForBaseTypeExpansion =
								InheritanceTransformer.this.generatorModel.getAspectForType(genSubPrimitiveClass);
						
						aspectForBaseTypeExpansion.addDeclareParents(GenDeclareInheritance.create(
								genSuperInterfaceClass,
								genSubPrimitiveClass));
					}
					
					@Override
					public void handle(final GenInterfaceClass genSubInterfaceClass) {
						// interface -> interface
						genSubInterfaceClass.getImplement().add(genSuperInterfaceClass);
						genSubInterfaceClass.accept(new GenInterfaceClassVisitor() {
							
							@Override
							public void handle(final GenInterfaceWithClassImplClass genSubInterfaceWithImplClass) {
								// interface with Impl -> interface
								genSuperInterfaceClass.accept(new GenInterfaceClassVisitor() {
									
									@Override
									public void handle(final GenInterfaceWithClassImplClass genSuperIntWImplCls) {
										// interface with Impl -> interface with Impl
										// also add inheritance from Impl-class to Impl-class
										
									}
									
									@Override
									public void handle(final GenSimpleInterfaceClass genSuperSimpleInterface) {
										// interface with Impl -> real interface
										// nothing more
									}
									
									@Override
									public void handle(final GenExternalInterfaceClass iface) {
										// interface with Impl --> external interface
										// nothing to do
									}
								});
							}
							
							@Override
							public void handle(final GenSimpleInterfaceClass genSubSimpleInterface) {
								// real interface -> interface
								// nothing more
							}
							
							@Override
							public void handle(final GenExternalInterfaceClass iface) {
								throw new Error("Interface " + genSuperInterfaceClass.getFullyQualifiedTypeName()
										+ " cannot be added as superinterface to external interface "
										+ iface.getFullyQualifiedTypeName());
							}
						});
					}
					
					@Override
					public void handle(final GenClassClass genSubClassClass) {
						// class -> interface
						genSuperInterfaceClass.accept(new GenInterfaceClassVisitor() {
							
							@Override
							public void handle(final GenInterfaceWithClassImplClass interfaceWithImplClass) {
								// class -> interface with Impl
								
							}
							
							@Override
							public void handle(final GenSimpleInterfaceClass simpleInterface) {
								// class -> real interface
								genSubClassClass.getImplement().add(genSuperInterfaceClass);
							}
							
							@Override
							public void handle(final GenExternalInterfaceClass iface) {
								// class -> external interface
								genSubClassClass.getImplement().add(iface);
							}
						});
					}
				});
			}
			
			@Override
			public void handle(final GenClassClass genSuperClassClass) {
				// class/interface -> class
				genSubClass.accept(new GenClassVisitor() {
					
					@Override
					public void handle(final GenInterfaceClass genSubInterfaceClass) {
						// interface -> class
						throw new Error("Inheritance transformation not possible! An "
								+ "interface can not extend a class.");
					}
					
					@Override
					public void handle(final GenClassClass genSubClassClass) {
						// class -> class
						genSubClassClass.setExtend(genSuperClassClass);
						// InheritanceTransformer.this.maybeAddSuperConstructorCall(genSubClassClass);
						
					}
					
					@Override
					public void handle(final GenPrimitiveClass genSubPrimitiveClass) {
						// class (of primitive Type) -> class
						throw new Error("Inheritance transformation not possible!"
								+ "An class for an primitive type can not extend a class.");
					}
				});
			}
			
			@Override
			public void handle(final GenPrimitiveClass genSuperPrimitiveClass) {
				// class/interface -> class (of primitive Type)
				throw new Error("Inheritance from classes for primitive types is not allowed!");
			}
		});
	}
	
	@Override
	public String toString() {
		return "Inheritance generation";
	}
	
	@Override
	public void beginTask() throws TaskException {
		// nothing to do
	}
	
	@Override
	public void finalizeTask() throws TaskException {
		// nothing to do
	}
}
