package de.fhdw.wtf.mojo;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Iterator;
import java.util.concurrent.ExecutionException;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;

import de.fhdw.wtf.common.ast.Model;
import de.fhdw.wtf.common.exception.editor.CheckException;
import de.fhdw.wtf.common.exception.editor.MultipleCheckExceptions;
import de.fhdw.wtf.common.exception.walker.CyclicDependencyException;
import de.fhdw.wtf.common.exception.walker.CyclicPartDefinitionException;
import de.fhdw.wtf.facade.ModelManager;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;

/**
 * A Maven Mojo, which generates the full WTF Stack: Java-Classes and Persitstence into the current project.
 * 
 * @goal generate
 * @phase process-sources
 * 
 */
public class FullGenerate extends AbstractMojo {
	
	/**
	 * A file to represent the root of the java sources.
	 * 
	 * @parameter property="basedir"
	 */
	private File root;
	
	@Override
	public void execute() throws MojoExecutionException, MojoFailureException {
		this.getLog().info("Starting generation");
		try {
			if (this.root == null) {
				this.getLog().error("Root Path is null");
				this.root = new File(".");
			}
			this.getLog().info("Execution Path is " + this.root.getPath());
			final URI modelUri =
					new URI("file://" + this.root.getPath().replace('\\', '/') + "/src/main/resources/models/model.wtf");
			final String outputDir = this.root.getPath() + "/src/main/java";
			final String dbProprties = this.root.getPath() + "/src/main/resources/config/oracledb.properties";
			this.getLog().debug("Model File URI: " + modelUri.toString());
			this.getLog().debug("Output Director: " + outputDir);
			this.getLog().debug("Database Properties in: " + dbProprties);
			
			final ModelManager modelManager = ModelManager.getInstance();
			this.getLog().info("Parsing Model");
			try {
				final Model modelFromFile = modelManager.getModelFromFile(modelUri);
				final GeneratorModel genModel = modelManager.generateJava(modelFromFile, outputDir);
				modelManager.generatePersistence(modelFromFile, genModel, dbProprties);
			} catch (final MultipleCheckExceptions e) {
				final Iterator<CheckException> i = e.iterator();
				while (i.hasNext()) {
					final CheckException current = i.next();
					this.getLog().error(current);
				}
			} catch (final CyclicDependencyException | InterruptedException | CyclicPartDefinitionException
					| ExecutionException e1) {
				this.getLog().error(e1);
			}
		} catch (final URISyntaxException e) {
			this.getLog().error(e);
		}
		
	}
}
