package de.fhdw.wtf.generator.java.generatorModel;

import java.util.HashMap;
import java.util.Map;
import java.util.Vector;

import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitor;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorReturn;

/**
 * Represents an external interface. Such an interface may not be changed by e.g. adding operations to it. It is used to
 * link generated classes/interfaces via implements/extends to externally available interfaces. This is used e.g. to
 * make the generated.sums.Anything interface an extension of de.fhdw.wtf.context.model.Anything.
 * 
 * The class features a static getInstance() method which does a bit of bookkeeping in order to create at most one
 * GenExternalInterfaceClass object per external interface.
 */
public class GenExternalInterfaceClass extends GenInterfaceClass {
	
	/**
	 * Maps fully qualified names of external interfaces to GenExternalInterfaceClass objects.
	 */
	private static Map<String, GenExternalInterfaceClass> extIfaceMap = new HashMap<>();
	
	/**
	 * Creates a GenExternalInterfaceClass object.
	 * 
	 * @param packageName
	 *            The package name of the external interface.
	 * @param typeName
	 *            The type name of the external interface.
	 */
	protected GenExternalInterfaceClass(final String packageName, final String typeName) {
		super(typeName, new Vector<GenJavaOperation>(), new Vector<GenInterfaceClass>(), GenUnqualifiedPackage
				.create(packageName), null, null);
	}
	
	/**
	 * Returns a GenExternalInterfaceClass object for a given external interfaces, specified by a fully qualified name.
	 * 
	 * @param fullyQualifiedName
	 *            The fully qualified name of an external interface.
	 * @return The GenExternalInterfaceClass object.
	 */
	public static synchronized GenExternalInterfaceClass getInstance(final String fullyQualifiedName) {
		GenExternalInterfaceClass iface = extIfaceMap.get(fullyQualifiedName);
		if (iface == null) {
			final int pos = fullyQualifiedName.lastIndexOf('.');
			if (pos < 0) {
				iface = new GenExternalInterfaceClass("", fullyQualifiedName);
			} else {
				iface =
						new GenExternalInterfaceClass(fullyQualifiedName.substring(0, pos),
								fullyQualifiedName.substring(pos + 1));
			}
			
			extIfaceMap.put(fullyQualifiedName, iface);
		}
		return iface;
	}
	
	@Override
	public void accept(final GenInterfaceClassVisitor visitor) {
		visitor.handle(this);
	}
	
	@Override
	public <X> X accept(final GenInterfaceClassVisitorReturn<X> visitor) {
		return visitor.handle(this);
	}
	
	@Override
	public <Y extends Exception> void accept(final GenInterfaceClassVisitorException<Y> visitor) throws Y {
		visitor.handle(this);
	}
	
}
