package de.fhdw.wtf.generator.writer.writer;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;

import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.context.Context;
import org.apache.velocity.runtime.RuntimeConstants;
import org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader;

import de.fhdw.wtf.file.FileUtils;
import de.fhdw.wtf.generator.java.generatorModel.GenClass;

public class FileHeaderWriter {
	
	private final ClassFileWriter classFileWriter;
	
	private static final String PACKAGE_KEY = "package";
	private static final String IMPORTS_KEY = "imports";
	private static final String GEN_CLASS_KEY = "genclass";
	private static final char JAVA_LINE_END = ';';
	private static final char PACKAGE_PATH_SEP = '.';
	private static final char FILE_PATH_SEP = '/';
	
	private static final String TEMPLATE_FILE_NAME = "de/fhdw/wtf/generator/templates/fileheader.vm";
	
	private final Context context = new VelocityContext();
	private final Template template;
	private final VelocityEngine engine;
	
	public FileHeaderWriter(final ClassFileWriter classFileWriter) {
		this.classFileWriter = classFileWriter;
		this.engine = new VelocityEngine();
		this.engine.setProperty(RuntimeConstants.RESOURCE_LOADER, "classpath");
		this.engine.setProperty("classpath.resource.loader.class", ClasspathResourceLoader.class.getName());
		this.engine.init();
		this.template = this.engine.getTemplate(TEMPLATE_FILE_NAME);
	}
	
	public void writeGenClass(final GenClass genclass, final File rootDir) {
		this.setUpContext(genclass);
		this.writeToFile(genclass, rootDir);
	}
	
	private void setUpContext(final GenClass c) {
		this.setPackage(c, this.context);
		this.setImports(c, this.context);
		this.setGenClass(c, this.context);
	}
	
	/**
	 * Writes the file for the given {@link GenClass} <code>c</code>.
	 * 
	 * @param rootDir
	 */
	private void writeToFile(final GenClass c, final File rootDir) {
		final String result = this.generateFileContent(this.context);
		final File path = this.createPath(c, rootDir);
		final File file = this.createFile(path, c);
		this.writeFile(result, path, file);
	}
	
	/**
	 * Merges the context into the classtemplate.
	 * 
	 * @param context
	 * @return
	 */
	private String generateFileContent(final Context context) {
		final StringWriter writer = new StringWriter();
		this.template.merge(context, writer);
		return writer.toString();
	}
	
	/**
	 * Creates the Path {@link File} for c.
	 * 
	 * @param c
	 */
	private File createPath(final GenClass c, final File rootDir) {
		String path = c.getPackag().toString();
		path = path.replace(PACKAGE_PATH_SEP, FILE_PATH_SEP);
		path = path.replace(JAVA_LINE_END, FILE_PATH_SEP);
		return new File(rootDir.getAbsolutePath() + FILE_PATH_SEP + path);
	}
	
	/**
	 * Creates the {@link File} for class c.
	 * 
	 * @param path
	 * @param c
	 */
	private File createFile(final File path, final GenClass c) {
		final String file = path.getAbsolutePath() + FILE_PATH_SEP + c.getName() + c.getFileEnding();
		return new File(file);
	}
	
	/**
	 * Writes the given result into the file.
	 * 
	 * @param result
	 * @param path
	 * @param file
	 */
	private void writeFile(final String result, final File path, final File file) {
		try {
			System.out.println("  -->  " + file.getAbsolutePath());
			FileUtils.overrideToFile(result, path, file);
		} catch (final IOException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Sets the imports block for c.
	 * 
	 * @param c
	 * @param context
	 */
	private void setImports(final GenClass c, final Context context) {
		context.put(IMPORTS_KEY, c.getImports());
	}
	
	/**
	 * Sets all values for the package declaration of {@link GenClass} c to the {@link Context}.
	 * 
	 * @param c
	 * @param context
	 */
	private void setPackage(final GenClass c, final Context context) {
		context.put(PACKAGE_KEY, c.getPackag().toString() + JAVA_LINE_END);
	}
	
	/**
	 * Sets the content of the containing type c.
	 * 
	 * @param c
	 * @param context
	 */
	private void setGenClass(final GenClass c, final Context context) {
		context.put(GEN_CLASS_KEY, this.classFileWriter.getStringContent(c));
	}
}
