package de.fhdw.wtf.generator.java.generatorModel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Vector;

import de.fhdw.wtf.generator.java.visitor.GenerationModelItemVisitor;

/**
 * A GenParameter represents a java-like parameter. For example for operations.
 */
public final class GenParameter extends GenerationModelItem {
	
	/**
	 * The referenced type of the parameter.
	 */
	private final GenTypeReference typ;
	
	/**
	 * The modifiers of the parameter.
	 */
	private final Collection<GenParameterModifier> modifiers;
	
	/**
	 * Generics in addition to the type of the parameter.
	 */
	private final ArrayList<Generic> generics;
	
	/**
	 * Instantiates a new {@link GenParameter}.
	 * 
	 * @param name
	 *            The name to identify the parameter.
	 * @param typ
	 *            The referenced type of the parameter.
	 */
	private GenParameter(final String name, final GenTypeReference typ) {
		super(name);
		this.typ = typ;
		this.modifiers = new Vector<>();
		this.generics = new ArrayList<>();
	}
	
	/**
	 * Creates a new {@link GenParameter}.
	 * 
	 * @param name
	 *            The name to identify the parameter.
	 * @param typ
	 *            The referenced type of the parameter.
	 * @return {@link GenParameter}
	 */
	public static GenParameter create(final String name, final GenTypeReference typ) {
		return new GenParameter(name, typ);
	}
	
	/**
	 * Creates a new {@link GenParameter}.
	 * 
	 * @param name
	 *            The name to identify the parameter.
	 * @param typ
	 *            The referenced type of the parameter.
	 * @return {@link GenParameter}
	 */
	public static GenParameter create(final String name, final GenImportType typ) {
		return GenParameter.create(name, GenTypeReferenceByReference.create(typ));
	}
	
	/**
	 * Creates a new {@link GenParameter}.
	 * 
	 * @param name
	 *            The name to identify the parameter.
	 * @param typ
	 *            The referenced type of the parameter.
	 * @return {@link GenParameter}
	 */
	public static GenParameter create(final String name, final GenType typ) {
		return GenParameter.create(name, GenTypeReferenceByReference.create(typ));
	}
	
	@Override
	public int hashCode() {
		return this.getGenerics().hashCode() ^ this.getModifiers().hashCode() ^ this.getName().hashCode()
				^ this.getTyp().hashCode();
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof GenParameter) {
			return ((GenParameter) obj).getName().equals(this.getName())
					&& ((GenParameter) obj).getTyp().equals(this.getTyp());
		}
		return false;
	}
	
	@Override
	public String toString() {
		final StringBuffer result = new StringBuffer();
		if (!this.modifiers.isEmpty()) {
			for (final GenParameterModifier modifier : this.modifiers) {
				result.append(modifier);
				result.append(' ');
			}
		}
		result.append(this.typ.getFullyQualifiedName());
		if (!this.generics.isEmpty()) {
			result.append('<');
			for (final GenType type : this.generics) {
				result.append(type.getName());
				result.append(',');
			}
			result.deleteCharAt(result.length() - 1);
			result.append('>');
		}
		result.append(' ');
		result.append(this.getName());
		return result.toString();
	}
	
	@Override
	public void accept(final GenerationModelItemVisitor visitor) {
		visitor.handle(this);
	}
	
	/**
	 * Returns the referenced type of the parameter.
	 * 
	 * @return {@link GenTypeReference}
	 */
	public GenTypeReference getTyp() {
		return this.typ;
	}
	
	/**
	 * Returns the modifiers of the parameter.
	 * 
	 * @return Collection<GenParameterModifier>
	 */
	public Collection<GenParameterModifier> getModifiers() {
		return this.modifiers;
	}
	
	/**
	 * Returns the generics that are an addition to the type of the parameter.
	 * 
	 * @return ArrayList<Generic>
	 */
	public ArrayList<Generic> getGenerics() {
		return this.generics;
	}
	
}
