package de.fhdw.wtf.generator.transformer.transformers.classTransformer;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import de.fhdw.wtf.common.ast.Attribute;
import de.fhdw.wtf.common.ast.ConstructorOrOperation;
import de.fhdw.wtf.common.ast.Group;
import de.fhdw.wtf.common.ast.Model;
import de.fhdw.wtf.common.ast.type.ClassType;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.generator.java.generatorModel.GenClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenComment;
import de.fhdw.wtf.generator.java.generatorModel.GenException;
import de.fhdw.wtf.generator.java.generatorModel.GenExternalInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenFullParsedOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceWithClassImplClass;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenOperationModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenParameter;
import de.fhdw.wtf.generator.java.generatorModel.GenPrimitiveClass;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenTypeReference;
import de.fhdw.wtf.generator.java.generatorModel.GenUserClass;
import de.fhdw.wtf.generator.java.generatorModel.GenVisibility;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;
import de.fhdw.wtf.generator.java.generatorModel.Generic;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenOperationStateVisitorReturnException;
import de.fhdw.wtf.generator.transformer.exception.GenTypeNotReferencedException;
import de.fhdw.wtf.walker.walker.SimpleWalkerTask;

/**
 * Transformer who realizes the delegation pattern for the generated model.
 * 
 * @author hfw413sc
 *
 */
public class DelegationTransformer extends SimpleWalkerTask {
	
	/**
	 * The referenced generator model.
	 */
	private final GeneratorModel generatorModel;
	
	/**
	 * Creates a DelegationTransformer.
	 * 
	 * @param m
	 *            the model
	 * @param taskmanager
	 *            the taskmanager
	 * @param generatorModel
	 *            the generator model
	 */
	public DelegationTransformer(final Model m, final TaskExecutor taskmanager, final GeneratorModel generatorModel) {
		super(m, taskmanager);
		this.generatorModel = generatorModel;
	}
	
	@Override
	public void handleClass(final ClassType c) throws TaskException {
		final GenClass clazz = this.generatorModel.getJavaClassForWTFClass(c);
		clazz.accept(new GenClassVisitorException<TaskException>() {
			
			@Override
			public void handle(final GenClassClass classClass) throws GenTypeNotReferencedException {
				// nothing to do
				
			}
			
			@Override
			public void handle(final GenInterfaceClass interfaceClass) throws TaskException {
				interfaceClass.accept(new GenInterfaceClassVisitorException<TaskException>() {
					
					@Override
					public void handle(final GenSimpleInterfaceClass simpleInterface)
							throws GenTypeNotReferencedException {
						// nothing to do
					}
					
					@Override
					public void handle(final GenInterfaceWithClassImplClass interfaceWithImplClass)
							throws TaskException {
						if (DelegationTransformer.this.isInterfaceInheritanceRoot(interfaceWithImplClass)) {
							DelegationTransformer.this.handleInheritanceRoot(interfaceWithImplClass
									.getClassRepresentation());
						}
						DelegationTransformer.this.makeDelegations(c, interfaceWithImplClass.getClassRepresentation());
						
					}
					
					@Override
					public void handle(final GenExternalInterfaceClass iface) throws GenTypeNotReferencedException {
						// nothing to do
					}
				});
				
			}
			
			@Override
			public void handle(final GenPrimitiveClass primitiveClass) throws GenTypeNotReferencedException {
				// nothing to do
			}
		});
	}
	
	/**
	 * Adds delegation code to an inheritance root.
	 * 
	 * @param classRepresentation
	 *            the root.
	 * @throws TaskException
	 *             if an operation has the wrong state.
	 */
	private void handleInheritanceRoot(final GenUserClass classRepresentation) throws TaskException {
		final Iterator<GenJavaOperation> operations = classRepresentation.getOperations().iterator();
		while (operations.hasNext()) {
			final GenJavaOperation genJavaOperation = operations.next();
			final String name = genJavaOperation.getName();
			final List<GenParameter> parameters = genJavaOperation.getParameters();
			final GenComment comment = this.getComment(genJavaOperation);
			final Collection<GenException> exceptions = this.getExceptions(genJavaOperation);
			final GenTypeReference returntyp = this.getReturnType(genJavaOperation);
			final Collection<GenOperationModifier> modifiers = this.getModifiers(genJavaOperation);
			final GenVisibility visibility = this.getVisibility(genJavaOperation);
			if (!name.contains("$") && !name.equals("accept")) {
				final boolean isAbstract = modifiers.contains(GenOperationModifier.ABSTRACT);
				if (isAbstract) {
					final GenJavaOperation operation =
							this.makeAbstractOperation(
									name,
									parameters,
									comment,
									exceptions,
									returntyp,
									modifiers,
									visibility,
									genJavaOperation.getGenerics());
					operation.getGenerics().addAll(genJavaOperation.getGenerics());
					classRepresentation.overrideExistingOperation(operation);
				} else {
					final GenJavaOperation operation =
							this.makeConcreteOperationEmpty(
									name,
									parameters,
									comment,
									exceptions,
									returntyp,
									modifiers,
									visibility,
									genJavaOperation.getGenerics());
					operation.getGenerics().addAll(genJavaOperation.getGenerics());
					classRepresentation.overrideExistingOperation(operation);
				}
			}
		}
	}
	
	/**
	 * Adds delegation to sub classes.
	 * 
	 * @param c
	 *            the root as classType.
	 * @param classRepresentation
	 *            the root.
	 * @throws TaskException
	 *             if an operation has the wrong state.
	 */
	private void makeDelegations(final ClassType c, final GenUserClass classRepresentation) throws TaskException {
		final Iterator<GenJavaOperation> operations = classRepresentation.getOperations().iterator();
		while (operations.hasNext()) {
			final GenJavaOperation genJavaOperation = operations.next();
			final String name = genJavaOperation.getName();
			final Collection<GenOperationModifier> modifiers = this.getModifiers(genJavaOperation);
			if (!name.contains("$")) {
				final boolean isAbstract = modifiers.contains(GenOperationModifier.ABSTRACT);
				final Iterator<ClassType> subTypes = c.getSubTypes().iterator();
				while (subTypes.hasNext()) {
					final ClassType subType = subTypes.next();
					final GenClassClass genSubClass =
							this.getRepresentingClass(this.generatorModel.getJavaClassForWTFClass(subType));
					
					boolean sameNameFound = false;
					final Iterator<GenJavaOperation> i2 = genSubClass.getOperations().iterator();
					while (!sameNameFound && i2.hasNext()) {
						final GenJavaOperation genJavaOperation2 = i2.next();
						if (genJavaOperation.getName().equals("accept")) {
							sameNameFound = genJavaOperation2.toString().equals(genJavaOperation.toString());
						} else {
							sameNameFound = genJavaOperation2.getName().equals(genJavaOperation.getName());
						}
					}
					this.addDelegationsInSubClass(
							subType,
							genSubClass,
							classRepresentation,
							genJavaOperation,
							isAbstract,
							sameNameFound);
				}
			}
		}
	}
	
	/**
	 * Adds delegations to a subclass.
	 * 
	 * @param subType
	 *            the class type of sub.
	 * @param sub
	 *            the representation.
	 * @param superType
	 *            the super type to delegate to.
	 * @param genJavaOperation
	 *            the operation.
	 * @param isAbstract
	 *            true if genJavaOperation is abstract.
	 * @param same
	 *            true if sub already has an operation equal to genJavaOperation.
	 * @throws TaskException
	 *             if an operation is in a wrong state.
	 */
	private void addDelegationsInSubClass(final ClassType subType,
			final GenClassClass sub,
			final GenUserClass superType,
			final GenJavaOperation genJavaOperation,
			final boolean isAbstract,
			final boolean same) throws TaskException {
		GenJavaOperation operation = null;
		if (subType.isAbstract()) {
			if (isAbstract) {
				operation = DelegationTransformer.this.makeAbstractOperation(genJavaOperation);
			} else {
				operation = DelegationTransformer.this.makeOperationWithDelegationToSuper(superType, genJavaOperation);
				this.getModifiers(operation).remove(GenOperationModifier.ABSTRACT);
			}
		} else {
			if (isAbstract) {
				operation = DelegationTransformer.this.makeConcreteOperationEmpty(genJavaOperation);
				this.getModifiers(operation).remove(GenOperationModifier.ABSTRACT);
			} else {
				operation = DelegationTransformer.this.makeOperationWithDelegationToSuper(superType, genJavaOperation);
				this.getModifiers(operation).remove(GenOperationModifier.ABSTRACT);
			}
			
		}
		if (same) {
			sub.overrideExistingOperation(operation);
		} else {
			sub.addOperation(operation);
		}
	}
	
	/**
	 * @param javaClass
	 *            the class to get the representing class from.
	 * @return the Representing Class of javaClass.
	 * @throws TaskException
	 *             if javaClass is not a not a GenInterfaceWithGenClass!
	 */
	private GenClassClass getRepresentingClass(final GenClass javaClass) throws TaskException {
		final GenClassClass classClass = javaClass.accept(new GenClassVisitorReturn<GenClassClass>() {
			
			@Override
			public GenClassClass handle(final GenClassClass genClassClass) {
				return null;
			}
			
			@Override
			public GenClassClass handle(final GenInterfaceClass interfaceClass) {
				return interfaceClass.accept(new GenInterfaceClassVisitorReturn<GenClassClass>() {
					
					@Override
					public GenClassClass handle(final GenSimpleInterfaceClass simpleInterface) {
						return null;
					}
					
					@Override
					public GenClassClass handle(final GenInterfaceWithClassImplClass interfaceWithImplClass) {
						return interfaceWithImplClass.getClassRepresentation();
					}
					
					@Override
					public GenClassClass handle(final GenExternalInterfaceClass iface) {
						return null;
					}
				});
			}
			
			@Override
			public GenClassClass handle(final GenPrimitiveClass primitiveClass) {
				return null;
			}
			
		});
		if (classClass == null) {
			throw new TaskException("Subclass is not a GenInterfaceWithGenClass!");
		}
		return classClass;
	}
	
	/**
	 * Creates an operation that delegates to superType.
	 * 
	 * @param superType
	 * @param genJavaOperation
	 *            the operation.
	 * @return the new operation.
	 * @throws TaskException
	 *             if operation is in an invalid state.
	 */
	private GenJavaOperation makeOperationWithDelegationToSuper(final GenUserClass superType,
			final GenJavaOperation genJavaOperation) throws TaskException {
		final String name = genJavaOperation.getName();
		final List<GenParameter> parameters = genJavaOperation.getParameters();
		final GenComment comment = this.getComment(genJavaOperation);
		final Collection<GenException> exceptions = this.getExceptions(genJavaOperation);
		final GenTypeReference returntyp = this.getReturnType(genJavaOperation);
		final Collection<GenOperationModifier> modifiers = new Vector<>();
		modifiers.addAll(this.getModifiers(genJavaOperation));
		final GenVisibility visibility = this.getVisibility(genJavaOperation);
		return this.makeOperationWithDelegationToSuper(
				superType,
				name,
				parameters,
				comment,
				exceptions,
				returntyp,
				modifiers,
				visibility,
				genJavaOperation.getGenerics());
	}
	
	/**
	 * 
	 * @param superType
	 *            superType.
	 * @param name
	 *            name.
	 * @param parameters
	 *            parameters.
	 * @param comment
	 *            comment.
	 * @param exceptions
	 *            exceptions.
	 * @param returntyp
	 *            returntyp.
	 * @param modifiers
	 *            modifiers.
	 * @param visibility
	 *            visibility.
	 * @param generics
	 *            generics.
	 * @return the new operation.
	 */
	private GenJavaOperation makeOperationWithDelegationToSuper(final GenUserClass superType,
			final String name,
			final List<GenParameter> parameters,
			final GenComment comment,
			final Collection<GenException> exceptions,
			final GenTypeReference returntyp,
			final Collection<GenOperationModifier> modifiers,
			final GenVisibility visibility,
			final List<Generic> generics) {
		final String superTypeName = superType.getFullyQualifiedTypeName();
		final boolean isVoid = returntyp.getFullyQualifiedName().equals("void");
		final String returnString = isVoid ? "" : "return ";
		final String methodBody =
				returnString + "((" + superTypeName
						+ ")this.$generatedObjects.get(new de.fhdw.wtf.context.model.Str(\"" + superTypeName + "\")))."
						+ name + "(" + this.generateParameterNamesList(parameters) + ");";
		final GenFullParsedOperationState state =
				GenFullParsedOperationState.create(comment, exceptions, returntyp, modifiers, visibility, methodBody);
		final GenJavaOperation operation = GenJavaOperation.create(name, parameters, state);
		operation.getGenerics().addAll(generics);
		return operation;
	}
	
	/**
	 * @param parameters
	 *            parameters of an operation.
	 * @return a string with the comma separated parameters.
	 */
	private String generateParameterNamesList(final List<GenParameter> parameters) {
		final StringBuilder parameterStringBuilder = new StringBuilder();
		for (final GenParameter p : parameters) {
			parameterStringBuilder.append(", ");
			parameterStringBuilder.append(p.getName());
		}
		String parameterNamesList = parameterStringBuilder.toString();
		if (parameterNamesList.length() > 0) {
			parameterNamesList = parameterNamesList.replaceFirst("," + " ", "");
		}
		return parameterNamesList;
	}
	
	/**
	 * Creates an operation with todo.
	 * 
	 * @param genJavaOperation
	 *            the operation.
	 * @return the new operation.
	 * @throws TaskException
	 *             if operation is in an invalid state.
	 */
	private GenJavaOperation makeConcreteOperationEmpty(final GenJavaOperation genJavaOperation) throws TaskException {
		final String name = genJavaOperation.getName();
		final List<GenParameter> parameters = genJavaOperation.getParameters();
		final GenComment comment = this.getComment(genJavaOperation);
		final Collection<GenException> exceptions = this.getExceptions(genJavaOperation);
		final GenTypeReference returntyp = this.getReturnType(genJavaOperation);
		final Collection<GenOperationModifier> modifiers = new Vector<>();
		modifiers.addAll(this.getModifiers(genJavaOperation));
		final GenVisibility visibility = this.getVisibility(genJavaOperation);
		return this.makeConcreteOperationEmpty(
				name,
				parameters,
				comment,
				exceptions,
				returntyp,
				modifiers,
				visibility,
				genJavaOperation.getGenerics());
	}
	
	/**
	 * 
	 * @param name
	 *            name.
	 * @param parameters
	 *            parameters.
	 * @param comment
	 *            comment.
	 * @param exceptions
	 *            exceptions.
	 * @param returntyp
	 *            returntyp.
	 * @param modifiers
	 *            modifiers.
	 * @param visibility
	 *            visibility.
	 * @param generics
	 *            generics.
	 * @return the new operation.
	 */
	private GenJavaOperation makeConcreteOperationEmpty(final String name,
			final List<GenParameter> parameters,
			final GenComment comment,
			final Collection<GenException> exceptions,
			final GenTypeReference returntyp,
			final Collection<GenOperationModifier> modifiers,
			final GenVisibility visibility,
			final List<Generic> generics) {
		String methodBody = "";
		if (name.equals("accept")) {
			methodBody = "visitor.handle(this);";
		} else {
			methodBody =
					"//TODO implement operation " + name
							+ (returntyp.getFullyQualifiedName().equals("void") ? "" : "\n\t\treturn null;");
		}
		final GenFullParsedOperationState state =
				GenFullParsedOperationState.create(comment, exceptions, returntyp, modifiers, visibility, methodBody);
		final GenJavaOperation operation = GenJavaOperation.create(name, parameters, state);
		operation.getGenerics().addAll(generics);
		return operation;
	}
	
	/**
	 * Creates an operation with check for this.
	 * 
	 * @param genJavaOperation
	 *            the operation.
	 * @return the new operation.
	 * @throws TaskException
	 *             if operation is in an invalid state.
	 */
	private GenJavaOperation makeAbstractOperation(final GenJavaOperation genJavaOperation) throws TaskException {
		final String name = genJavaOperation.getName();
		final List<GenParameter> parameters = genJavaOperation.getParameters();
		final GenComment comment = this.getComment(genJavaOperation);
		final Collection<GenException> exceptions = this.getExceptions(genJavaOperation);
		final GenTypeReference returntyp = this.getReturnType(genJavaOperation);
		final Collection<GenOperationModifier> modifiers = new Vector<>();
		modifiers.addAll(this.getModifiers(genJavaOperation));
		final GenVisibility visibility = this.getVisibility(genJavaOperation);
		return this.makeAbstractOperation(
				name,
				parameters,
				comment,
				exceptions,
				returntyp,
				modifiers,
				visibility,
				genJavaOperation.getGenerics());
	}
	
	/**
	 * 
	 * @param name
	 *            name.
	 * @param parameters
	 *            parameters.
	 * @param comment
	 *            comment.
	 * @param exceptions
	 *            exceptions.
	 * @param returntyp
	 *            returntyp.
	 * @param modifiers
	 *            modifiers.
	 * @param visibility
	 *            visibility.
	 * @param generics
	 *            generics.
	 * @return the new operation.
	 */
	private GenJavaOperation makeAbstractOperation(final String name,
			final List<GenParameter> parameters,
			final GenComment comment,
			final Collection<GenException> exceptions,
			final GenTypeReference returntyp,
			final Collection<GenOperationModifier> modifiers,
			final GenVisibility visibility,
			final List<Generic> generics) {
		final String methodBody =
				"if(this.$getThis()==this)throw new Error();" + "\n\t\t"
						+ (returntyp.getFullyQualifiedName().equals("void") ? "" : "return ") + "this.$getThis()."
						+ name + "();";
		final GenFullParsedOperationState state =
				GenFullParsedOperationState.create(comment, exceptions, returntyp, modifiers, visibility, methodBody);
		final GenJavaOperation operation = GenJavaOperation.create(name, parameters, state);
		operation.getGenerics().addAll(generics);
		return operation;
	}
	
	/**
	 * Returns true if checkInterface is directly extending Anything.
	 * 
	 * @param checkInterface
	 *            Interface to be checked
	 * @return true if checkInterface is a root of the inheritance
	 * @throws GenTypeNotReferencedException
	 *             possible exception
	 */
	private boolean isInterfaceInheritanceRoot(final GenInterfaceWithClassImplClass checkInterface)
			throws GenTypeNotReferencedException {
		return checkInterface.getImplement().contains(
				this.generatorModel.getGenTypeForType(this.getModel().getAnything()));
	}
	
	@Override
	public void handleGroup(final Group g) throws TaskException {
		// nothing to do
		
	}
	
	@Override
	public void handleAttribute(final Attribute a, final ClassType owner) throws TaskException {
		// nothing to do
		
	}
	
	@Override
	public void handleConstructorOrOperation(final ConstructorOrOperation coo, final ClassType owner)
			throws TaskException {
		// nothing to do
		
	}
	
	@Override
	public void finalizeTask() throws TaskException {
		// Nothing to do.
	}
	
	/**
	 * 
	 * @param genJavaOperation
	 *            an operation.
	 * @return the desired field value.
	 * @throws TaskException
	 *             if operation in wrong state.
	 */
	private GenVisibility getVisibility(final GenJavaOperation genJavaOperation) throws TaskException {
		return genJavaOperation.getState().accept(
				new GenOperationStateVisitorReturnException<GenVisibility, TaskException>() {
					
					@Override
					public GenVisibility handle(final GenSimpleOperationState s) throws TaskException {
						throw new TaskException("Wrong state for operation");
					}
					
					@Override
					public GenVisibility handle(final GenFullParsedOperationState s) {
						return s.getVisibility();
					}
				});
	}
	
	/**
	 * 
	 * @param genJavaOperation
	 *            an operation.
	 * @return the desired field value.
	 * @throws TaskException
	 *             if operation in wrong state.
	 */
	private Collection<GenOperationModifier> getModifiers(final GenJavaOperation genJavaOperation) throws TaskException {
		return genJavaOperation.getState().accept(
				new GenOperationStateVisitorReturnException<Collection<GenOperationModifier>, TaskException>() {
					
					@Override
					public Collection<GenOperationModifier> handle(final GenSimpleOperationState s)
							throws TaskException {
						throw new TaskException("Wrong state for operation");
					}
					
					@Override
					public Collection<GenOperationModifier> handle(final GenFullParsedOperationState s) {
						return s.getModifiers();
					}
				});
	}
	
	/**
	 * 
	 * @param genJavaOperation
	 *            an operation.
	 * @return the desired field value.
	 * @throws TaskException
	 *             if operation in wrong state.
	 */
	private GenTypeReference getReturnType(final GenJavaOperation genJavaOperation) throws TaskException {
		return genJavaOperation.getState().accept(
				new GenOperationStateVisitorReturnException<GenTypeReference, TaskException>() {
					
					@Override
					public GenTypeReference handle(final GenSimpleOperationState s) throws TaskException {
						throw new TaskException("Wrong state for operation");
					}
					
					@Override
					public GenTypeReference handle(final GenFullParsedOperationState s) {
						return s.getReturntyp();
					}
				});
	}
	
	/**
	 * 
	 * @param genJavaOperation
	 *            an operation.
	 * @return the desired field value.
	 * @throws TaskException
	 *             if operation in wrong state.
	 */
	private GenComment getComment(final GenJavaOperation genJavaOperation) throws TaskException {
		return genJavaOperation.getState().accept(
				new GenOperationStateVisitorReturnException<GenComment, TaskException>() {
					
					@Override
					public GenComment handle(final GenSimpleOperationState s) throws TaskException {
						throw new TaskException("Wrong state for operation");
					}
					
					@Override
					public GenComment handle(final GenFullParsedOperationState s) {
						return s.getComment();
					}
				});
	}
	
	/**
	 * 
	 * @param genJavaOperation
	 *            an operation.
	 * @return the desired field value.
	 * @throws TaskException
	 *             if operation in wrong state.
	 */
	private Collection<GenException> getExceptions(final GenJavaOperation genJavaOperation) throws TaskException {
		return genJavaOperation.getState().accept(
				new GenOperationStateVisitorReturnException<Collection<GenException>, TaskException>() {
					
					@Override
					public Collection<GenException> handle(final GenSimpleOperationState s) throws TaskException {
						throw new TaskException("Wrong state for operation");
					}
					
					@Override
					public Collection<GenException> handle(final GenFullParsedOperationState s) {
						return s.getExceptions();
					}
				});
	}
	
	@Override
	public void beginTask() throws TaskException {
		// nothing to do
		
	}
	
}
