package de.fhdw.wtf.file;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collection;
import java.util.Iterator;

/**
 * Collection of operations for file read and modification.
 */
public final class FileUtils {
	
	/**
	 * Constructor of {@link FileUtils}.
	 */
	private FileUtils() {
	}
	
	/**
	 * Line-Break.
	 */
	private static final String BREAK = "\n";
	
	/**
	 * Error for Directory-Creating.
	 */
	private static final String ERROR_MKDIR = "Could not create directory.";
	
	/**
	 * Error for File-Deleting.
	 */
	private static final String ERROR_DELETE = "Could not delete file.";
	
	/**
	 * Error for File-Creating.
	 */
	private static final String ERROR_CREATE = "Could not create file.";
	
	/**
	 * First deletes the file if the file exists. Then creates directory if it does not exist and finally creates the
	 * file and writes the content to the new file.
	 * 
	 * @param content
	 *            content
	 * @param path
	 *            Path without Filename
	 * @param file
	 *            Path with Filename
	 * @throws IOException
	 *             possible Exception
	 */
	public static void overrideToFile(final String content, final File path, final File file) throws IOException {
		if (file.exists() && !file.delete()) {
			throw new IOException(ERROR_DELETE);
		}
		if (!path.exists() && !path.mkdirs()) {
			throw new IOException(ERROR_MKDIR);
		}
		if (!file.createNewFile()) {
			throw new IOException(ERROR_CREATE);
		}
		try (final FileWriter writer = new FileWriter(file, false);) {
			writer.write(content);
			writer.flush();
		}
	}
	
	/**
	 * Returns the content of the given {@link File}.
	 * 
	 * @param filePath
	 *            filePath
	 * @return content of the file
	 * @throws IOException
	 *             For Exceptions with reading and file handling.
	 */
	public static String getFileString(final String filePath) throws IOException {
		try (final FileIterator i = new FileIterator(new File(filePath));) {
			final StringBuilder b = new StringBuilder();
			while (i.hasNext()) {
				b.append(i.next());
				if (i.hasNext()) {
					b.append(BREAK);
				}
			}
			return b.toString();
		}
	}
	
	/**
	 * Saves the content to an already created file.
	 * 
	 * @param content
	 *            content
	 * @param file
	 *            file
	 * @throws IOException
	 *             possbile Exception
	 */
	public static void saveToFile(final String content, final File file) throws IOException {
		try (final FileWriter writer = new FileWriter(file, false);) {
			writer.write(content);
			writer.flush();
		}
	}
	
	/**
	 * Returned the files as String.
	 * 
	 * @param files
	 *            list of files
	 * @return files as String
	 * @throws java.io.FileNotFoundException
	 *             possible Exception
	 */
	public static String printFileList(final Collection<File> files) {
		final Iterator<File> i = files.iterator();
		final StringBuilder sb = new StringBuilder();
		while (i.hasNext()) {
			sb.append(i.next().getAbsolutePath());
			sb.append(BREAK);
		}
		return sb.toString();
	}
	
	/**
	 * Delete the directory.
	 * 
	 * @param dir
	 *            directory
	 * @throws IOException
	 *             possible Exception
	 */
	public static void deleteDirectory(final String dir) throws IOException {
		final Path directory = Paths.get(dir);
		if (new File(dir).listFiles() != null) {
			Files.walkFileTree(directory, new SimpleFileVisitor<Path>() {
				@Override
				public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException {
					Files.delete(file);
					return FileVisitResult.CONTINUE;
				}
				
				@Override
				public FileVisitResult postVisitDirectory(final Path dir2, final IOException exc) throws IOException {
					Files.delete(dir2);
					return FileVisitResult.CONTINUE;
				}
				
			});
		}
	}
}
