package de.fhdw.wtf.generator.writer.tasks;

import java.io.File;

import de.fhdw.wtf.common.exception.walker.CyclicDependencyException;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.DependencyTask;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.generator.java.generatorModel.GenAnyType;
import de.fhdw.wtf.generator.java.generatorModel.GenAspect;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenException;
import de.fhdw.wtf.generator.java.generatorModel.GenExternalClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenExternalInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceWithClassImplClass;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaException;
import de.fhdw.wtf.generator.java.generatorModel.GenOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenUserClass;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;
import de.fhdw.wtf.generator.java.visitor.GenClassClassVisitor;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitor;
import de.fhdw.wtf.generator.java.walker.SimpleGeneratorModelWalkerTask;
import de.fhdw.wtf.generator.writer.writer.ExceptionFileWriter;
import de.fhdw.wtf.generator.writer.writer.FileHeaderWriter;
import de.fhdw.wtf.generator.writer.writer.JavaAspectClassFileWriter;
import de.fhdw.wtf.generator.writer.writer.JavaClassFileWriter;
import de.fhdw.wtf.generator.writer.writer.JavaInterfaceClassFileWriter;

/**
 * 
 * {@link DependencyTask} for Writing all Java-Files declared in a {@link GeneratorModel} by {@link GenClassClass}s or
 * {@link GenInterfaceClass} .
 */
public class FileWriterTask extends SimpleGeneratorModelWalkerTask {
	
	private final JavaInterfaceClassFileWriter interfaceWriter;
	private final JavaClassFileWriter classWriter;
	private final JavaAspectClassFileWriter aspectWriter;
	private final ExceptionFileWriter exceptionWriter;
	
	private final File rootDir;
	
	public FileWriterTask(final TaskExecutor taskmanager,
			final GeneratorModel model,
			final String rootDir,
			final DependencyTask clipper) {
		super(taskmanager, model);
		this.interfaceWriter = new JavaInterfaceClassFileWriter(true);
		this.classWriter = new JavaClassFileWriter(true);
		this.aspectWriter = new JavaAspectClassFileWriter(true);
		this.exceptionWriter = new ExceptionFileWriter(true);
		this.rootDir = new File(rootDir);
		
		try {
			this.addDependency(clipper);
		} catch (final CyclicDependencyException e) {
			// should not happen
			e.printStackTrace();
			throw new Error("Dependency tasks are cyclic in FileWriterTask.");
		}
	}
	
	/**
	 * Writes the given {@link GenClassClass}.
	 */
	@Override
	public void handleClassClass(final GenClassClass cc) throws TaskException {
		cc.accept(new GenClassClassVisitor() {
			@Override
			public void handle(final GenAspect aspect) {
				final FileHeaderWriter writer = new FileHeaderWriter(FileWriterTask.this.aspectWriter);
				writer.writeGenClass(aspect, FileWriterTask.this.rootDir);
			}
			
			@Override
			public void handle(final GenUserClass userclass) {
				final FileHeaderWriter writer = new FileHeaderWriter(FileWriterTask.this.classWriter);
				writer.writeGenClass(userclass, FileWriterTask.this.rootDir);
			}
			
			@Override
			public void handle(final GenJavaException javaE) {
				// nothing to do
			}
			
			@Override
			public void handle(final GenAnyType anyType) {
				// nothing to do
			}
			
			@Override
			public void handle(final GenException exceptionClass) {
				final FileHeaderWriter headerWriter = new FileHeaderWriter(FileWriterTask.this.exceptionWriter);
				headerWriter.writeGenClass(exceptionClass, FileWriterTask.this.rootDir);
			}
			
			@Override
			public void handle(final GenExternalClassClass externalClass) {
				// nothing to do
			}
		});
	}
	
	/**
	 * Writes the given {@link GenInterfaceClass}.
	 */
	@Override
	public void handleInterfaceClass(final GenInterfaceClass ic) throws TaskException {
		ic.accept(new GenInterfaceClassVisitor() {
			
			@Override
			public void handle(final GenInterfaceWithClassImplClass interfaceWithImplClass) {
				
				final FileHeaderWriter writer = new FileHeaderWriter(FileWriterTask.this.interfaceWriter);
				writer.writeGenClass(interfaceWithImplClass, FileWriterTask.this.rootDir);
				
				final FileHeaderWriter classWriter = new FileHeaderWriter(FileWriterTask.this.classWriter);
				classWriter.writeGenClass(interfaceWithImplClass.getClassRepresentation(), FileWriterTask.this.rootDir);
				
			}
			
			@Override
			public void handle(final GenSimpleInterfaceClass simpleInterface) {
				
				final FileHeaderWriter writer = new FileHeaderWriter(FileWriterTask.this.interfaceWriter);
				writer.writeGenClass(simpleInterface, FileWriterTask.this.rootDir);
			}
			
			@Override
			public void handle(final GenExternalInterfaceClass iface) {
				// nothing to do, interface must already exist
			}
		});
	}
	
	@Override
	public void finalizeTask() throws TaskException {
		// nothing to do
	}
	
	@Override
	public String toString() {
		return "Filewriting";
	}
	
	@Override
	public void handleOperation(final GenOperation o) throws TaskException {
		// nothing to do
	}
}
