package de.fhdw.wtf.generator.transformer.clipper;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.fhdw.wtf.generator.java.generatorModel.GenComment;
import de.fhdw.wtf.generator.java.generatorModel.GenFullParsedOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenOperationModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenTypeReferenceByName;
import de.fhdw.wtf.generator.java.generatorModel.GenVisibility;
import de.fhdw.wtf.generator.transformer.exception.ClipperImportFormatException;
import de.fhdw.wtf.generator.transformer.exception.NoMatchingMethodException;
import de.fhdw.wtf.generator.transformer.util.Tuple;

/**
 * Helper functions used in the clipper. Used for analysis of java source code, comments, etc...
 * 
 */
public final class ClipperUtils {
	
	/** Beginning of a Char-Literal '. */
	private static final String JAVA_STRING_LITERAL_SINGLE_SIGN = "\'";
	
	/** Beginning of a String-Literal ". */
	private static final String JAVA_STRING_LITERAL_DOUBLE_SIGN = "\"";
	
	/** Regular Expression for Java identifier names. */
	public static final String IDENTIFIER_REGEX = "[A-Za-z_$€][\\.A-Za-z_$€0-9]*";
	
	//
	/** Filename-constant for constructor-files. */
	public static final String CONSTRUCTOR_FILENAME = "#constructor";
	/** Filename-constant for area protected from clipping. */
	public static final String PROTECTED_AREA_FILENAME = "#protectedArea";
	/** Filename-constant for javadoc comments of classes. */
	public static final String CLASSCOMMENT_FILENAME = "#classcomment";
	/** Filename-constant for imports of classes. */
	public static final String IMPORTS_FILENAME = "#imports";
	/**
	 * Filename-constant for clipper error files, where the correct filename cannot be determined.
	 */
	public static final String ERRORFILE_FILENAME = "#errorFile";
	/** Seperator between operation parameters or attributes in operation-filenames. */
	public static final String SEPERATOR = "#";
	/** Filename constant for start of attribute filename. */
	public static final String ATTRIBUTE_START = "#attr#";
	/** Parameter separator. */
	public static final String PARAMETER_SEPARATOR = ",";
	/** Matches opening bracket of generic parameter list. */
	public static final String GENERIC_OPEN = "<";
	/** Replacement for opening bracket of generic parameter list, usable in filenames. */
	public static final String GENERIC_OPEN_REPLACEMENT = "%{";
	/** Matches closing bracket of generic parameter list. */
	public static final String GENERIC_CLOSE = ">";
	/** Replacement for closing bracket of generic parameter list, usable in filenames. */
	public static final String GENERIC_CLOSE_REPLACEMENT = "%}";
	/** Replacement character for any invalid character. */
	public static final String REPLACEMENT_FOR_ILLEGAL_CHAR = "_";
	/** Java source filename ending. */
	public static final String FILE_ENDING = ".java";
	/** Regular Expression for matching the non-generation part. */
	public static final String NON_GEN_PART_MATCH = ".*// --> NON-GENERATION-PART <--(.*)}\\s*";
	/** Start of javadoc comment. */
	public static final String COMMENT_OPEN = "/*";
	/** End of javadoc comment. */
	public static final String COMMENT_CLOSE = "*/";
	
	/** Error constants. */
	public static final String CLIPPER_ERROR_NO_METHOD = "Clipper Error: No method";
	
	/**
	 * Unneccessary, because its a helper class with only static access.
	 */
	private ClipperUtils() {
		
	}
	
	/**
	 * Deletes all java blockcomments in the file and replaces the comment region with equivalent amount of whitespace.
	 * 
	 * @param file
	 *            file, possibly containing block comments
	 * @return file of the same size with whitened (replaced with spaces) comments
	 * @throws ClipperImportFormatException
	 *             when the parsing errors
	 */
	public static String deleteBlockComments(final String file) throws ClipperImportFormatException {
		String result = file;
		final String nextStringLiteralDoubleSign = JAVA_STRING_LITERAL_DOUBLE_SIGN;
		final String nextStringLiteralSingleSign = JAVA_STRING_LITERAL_SINGLE_SIGN;
		final String commentSign = COMMENT_OPEN;
		final String commentSignClose = COMMENT_CLOSE;
		int pos = 0;
		int doublePos = findNextQuotation(nextStringLiteralDoubleSign, result, pos);
		int singlePos = findNextQuotation(nextStringLiteralSingleSign, result, pos);
		int commentPos = result.indexOf(commentSign);
		
		while (true) {
			if (commentPos < 0) {
				break;
			}
			if (doublePos < commentPos && (doublePos < singlePos || singlePos < 0) && doublePos > 0) {
				pos = findNextQuotation(nextStringLiteralDoubleSign, result, doublePos + 1);
			} else if (singlePos < commentPos && (singlePos < doublePos || doublePos < 0) && singlePos > 0) {
				pos = findNextQuotation(nextStringLiteralSingleSign, result, singlePos + 1);
				singlePos = findNextQuotation(nextStringLiteralSingleSign, result, pos + 1);
			} else if (commentPos >= pos) {
				final int oldPos = commentPos;
				pos = result.indexOf(commentSignClose, commentPos) + 2;
				result = replaceFromToWithWhitespace(result, oldPos, pos);
				commentPos = result.indexOf(commentSign, pos);
			} else {
				break;
			}
			if (doublePos < pos && doublePos != -1) {
				doublePos = findNextQuotation(nextStringLiteralDoubleSign, result, pos + 1);
			}
			if (singlePos < pos && singlePos != -1) {
				singlePos = findNextQuotation(nextStringLiteralSingleSign, result, pos + 1);
			}
			if (commentPos < pos && commentPos != -1) {
				commentPos = result.indexOf(commentSign, pos + 1);
			}
			
			if (pos < 0) {
				throw new ClipperImportFormatException("Error while parsing quotes");
			}
		}
		return result;
	}
	
	/**
	 * Replaces the given range of characters (from - to) with exact amount of whitespace.
	 * 
	 * @param text
	 *            given text
	 * @param from
	 *            begin position in text
	 * @param to
	 *            end position in text
	 * @return text with replaced characters
	 */
	public static String replaceFromToWithWhitespace(final String text, final int from, final int to) {
		String result = text;
		final String pre = result.substring(0, from);
		String oldComment;
		if (to - from > 0) {
			oldComment = generateSpaces(to - from);
		} else {
			oldComment = "";
		}
		final String after = result.substring(to);
		result = pre + oldComment + after;
		return result;
	}
	
	/**
	 * Generates a String with the given number of spaces.
	 * 
	 * @param n
	 *            The number of spaces in the return value
	 * @return A given number (n) of spaces
	 * 
	 */
	private static String generateSpaces(final int n) {
		return String.format("%1$" + n + "s", "");
	}
	
	/**
	 * Finds the next position of the given literal. Skips existing escaped literals.
	 * 
	 * @param literal
	 *            java literal (", ') to be searched for.
	 * @param file
	 *            text to be searched in
	 * @param startPos
	 *            start position for searching non-escaped character
	 * @return position in the file (after startPos) where the literal has been found
	 */
	public static int findNextQuotation(final String literal, final String file, final int startPos) {
		int pos = file.indexOf(literal, startPos);
		while (pos > 0) {
			if (isNotEscaped(file, pos)) {
				return pos;
			}
			
			pos = file.indexOf(literal, pos + 1);
		}
		return pos;
	}
	
	private static boolean isNotEscaped(final String file, final int pos) {
		final int startPos = pos;
		int currentPos = pos;
		while (file.charAt(currentPos - 1) == '\\') {
			currentPos--;
		}
		return (currentPos - startPos) % 2 == 0;
	}
	
	/**
	 * Searches for Java end line comments (//) and deletes them.
	 * 
	 * @param file
	 *            text of the java source code
	 * @return file with whithened comments (= replaced by equivalent amount of spaces).
	 */
	public static String deleteEndOfLineComments(final String file) {
		final String pattern = "(?<!\\*)//.*";
		String withoutcomments = file;
		final Pattern p = Pattern.compile(pattern);
		final Matcher m = p.matcher(file);
		while (m.find()) {
			withoutcomments = replaceFromToWithWhitespace(withoutcomments, m.start(), m.end());
		}
		return withoutcomments;
	}
	
	/**
	 * Search beginning from position backwards for possible comments.
	 * 
	 * @param position
	 * @param file
	 * @param withoutcommentFile
	 * @return Returns the start + end position of a possible comment. <operationStart> otherwise.
	 */
	public static Tuple<Integer, Integer> getCommentBefore(final int position,
			final String file,
			final String withoutcommentFile) {
		final int end = file.lastIndexOf(COMMENT_CLOSE, position);
		final int begin = file.lastIndexOf(COMMENT_OPEN, position);
		if (end > begin && end > 0 && begin >= 0 && !withoutcommentFile.substring(end, position).contains(";")
				&& !withoutcommentFile.substring(end, position).contains("{")) {
			return new Tuple<>(begin, end + 2);
		}
		return new Tuple<>(position, position);
	}
	
	/**
	 * Searches for string literals and returns the file without string literals. They are replaced by spaces.
	 * 
	 * @param file
	 *            file possibly containing string literals (String a = "This is a literal"; ==> String a = ;).
	 * @return file containing no literals but spaces instead
	 * @throws ClipperImportFormatException
	 * 
	 *             Attention: requires a file without comments!
	 */
	public static String deleteStringLiterals(final String file) throws ClipperImportFormatException {
		final String nextStringLiteralDoubleSign = JAVA_STRING_LITERAL_DOUBLE_SIGN;
		final String nextStringLiteralSingleSign = JAVA_STRING_LITERAL_SINGLE_SIGN;
		int start1, end1, start2, end2;
		String withoutComments = file;
		
		int pos = 0;
		
		while (true) {
			start2 = findNextQuotation(nextStringLiteralDoubleSign, file, pos);
			end2 = findNextQuotation(nextStringLiteralDoubleSign, file, start2 + 1);
			start1 = findNextQuotation(nextStringLiteralSingleSign, file, pos);
			end1 = findNextQuotation(nextStringLiteralSingleSign, file, start1 + 1);
			if ((end1 < start2 || start2 == -1) && end1 != -1 && start1 != -1) {
				pos = end1 + 1;
				withoutComments = replaceFromToWithWhitespace(withoutComments, start1, end1 + 1);
			} else if ((end2 < start1 || start1 == -1) && end2 != -1 && start2 != -1) {
				pos = end2 + 1;
				withoutComments = replaceFromToWithWhitespace(withoutComments, start2, end2 + 1);
			} else if (start2 != -1 && end2 == -1 || start1 != -1 && end1 == -1) {
				throw new ClipperImportFormatException("Error while parsing quotes");
			} else {
				break;
			}
			
		}
		
		return withoutComments;
	}
	
	/**
	 * Simplifies Java code by removing comments, generics, implementations, visibilities and other modifiers.
	 * 
	 * @param file
	 *            java source code
	 * @return simplified java code
	 * @throws ClipperImportFormatException
	 *             when the java code cannot be parsed
	 */
	public static String simplifyJavaCode(final String file) throws ClipperImportFormatException {
		String simpleFile = ClipperUtils.deleteEndOfLineComments(file);
		simpleFile = ClipperUtils.deleteBlockComments(simpleFile);
		simpleFile = ClipperUtils.deleteStringLiterals(simpleFile);
		simpleFile = ClipperUtils.splitInnerClasses(simpleFile);
		simpleFile = ClipperUtils.deleteAttributeImplementations(simpleFile);
		// simpleFile = ClipperUtils.deleteAllGenerics(simpleFile);
		simpleFile = ClipperUtils.deleteAllAnnotations(simpleFile);
		final Collection<String> modifierList = new ArrayList<>();
		modifierList.add("private");
		modifierList.add("public");
		modifierList.add("synchronized");
		modifierList.add("protected");
		modifierList.add("transient");
		modifierList.add("static");
		modifierList.add("final");
		modifierList.add("abstract");
		
		final Iterator<String> i = modifierList.iterator();
		while (i.hasNext()) {
			final String current = i.next();
			final String regex = "(?<![A-Za-z0-9_])" + current + "(?![A-Za-z0-9_])";
			simpleFile = simpleFile.replaceAll(regex, generateSpaces(current.length()));
		}
		
		return simpleFile;
	}
	
	/**
	 * Replaces attribute implementations by ";", e.g. converts "int a = 5;" to "int a;".
	 * 
	 * @param simpleFile
	 * @return
	 * @throws ClipperImportFormatException
	 */
	@SuppressWarnings("javadoc")
	private static String deleteAttributeImplementations(final String simpleFile) throws ClipperImportFormatException {
		String result = simpleFile;
		while (true) {
			final int start = result.indexOf('=');
			if (start == -1) {
				break;
			}
			final int semicolon = result.indexOf(';', start);
			if (semicolon == -1) {
				throw new ClipperImportFormatException("Semicolon missing!");
			}
			result = replaceFromToWithWhitespace(result, start, semicolon);
		}
		return result;
	}
	
	/**
	 * Detects inner classes and does not remove them if the code will be simpliefied
	 * 
	 * @param simpleFile
	 * @return returns the simple code
	 * @throws ClipperImportFormatException
	 */
	public static String splitInnerClasses(final String simpleFile) throws ClipperImportFormatException {
		
		String result = "";
		String suffix = simpleFile;
		
		final int classIndex = suffix.indexOf(" class ");
		final int interfaceIndex = suffix.indexOf(" interface ");
		int start =
				((classIndex > -1 && interfaceIndex > -1) ? Math.min(classIndex, interfaceIndex) : Math.max(
						classIndex,
						interfaceIndex));
		int open = suffix.indexOf('{');
		
		while (open > -1) {
			final Integer close = ClipToFileTask.findCorrespondingClosingCurlyBracket(open, suffix);
			
			if (start == -1 || start > open) {
				// No class / interface found at the beginning
				final String sub = suffix.substring(0, close);
				result = result + replaceFromToWithWhitespace(sub, open, close) + ";";
				suffix = suffix.substring(close + 1);
			} else {
				// Class / interface found at the beginning
				result = result + suffix.substring(0, open + 1);
				final String core = suffix.substring(open + 1, close);
				result = result + ClipperUtils.splitInnerClasses(core);
				suffix = suffix.substring(close);
			}
			start = Math.max(suffix.indexOf(" class "), suffix.indexOf(" interface "));
			open = suffix.indexOf('{');
		}
		return result + (suffix.length() > 0 ? suffix : "");
	}
	
	private static String deleteAllAnnotations(final String withoutcommentFile) {
		String result = withoutcommentFile;
		while (true) {
			final int indexStart = result.indexOf('@');
			if (indexStart == -1) {
				break;
			}
			final String pattern = "@\\s*[A-Za-z0-9]+.*\\n";
			final Pattern p = Pattern.compile(pattern);
			final Matcher m = p.matcher(result);
			if (m.find(indexStart)) {
				final int indexEnd = m.end() - 1;
				result = ClipperUtils.replaceFromToWithWhitespace(result, indexStart, indexEnd);
			}
		}
		return result;
	}
	
	/**
	 * TODO
	 * 
	 * @param openBracketPos
	 * @param file
	 * @return
	 * @throws ClipperImportFormatException
	 */
	@SuppressWarnings("javadoc")
	public static Integer findCorrespondingClosingCurlyBracket(final int openBracketPos, final String file)
			throws ClipperImportFormatException {
		Integer result = openBracketPos;
		Integer openBracketMinusClosed = 1;
		while (openBracketMinusClosed != 0) {
			final int indexOfClose = file.indexOf(String.valueOf('}'), result + 1);
			final int indexOfOpen = file.indexOf(String.valueOf('{'), result + 1);
			if (indexOfClose < indexOfOpen || indexOfOpen == -1) {
				openBracketMinusClosed--;
				result = indexOfClose;
			} else {
				openBracketMinusClosed++;
				result = indexOfOpen;
			}
			if (indexOfClose == -1) {
				throw new ClipperImportFormatException("Unbalanced Brackets!");
			}
		}
		return result;
	}
	
	/**
	 * TODO
	 * 
	 * @param openBracketPos
	 * @param file
	 * @return
	 * @throws ClipperImportFormatException
	 */
	@SuppressWarnings("javadoc")
	public static Integer findCorrespondingClosingRoundBracket(final int openBracketPos, final String file)
			throws ClipperImportFormatException {
		Integer result = openBracketPos;
		Integer openBracketMinusClosed = 1;
		while (openBracketMinusClosed != 0) {
			final int indexOfClose = file.indexOf(String.valueOf(')'), result + 1);
			final int indexOfOpen = file.indexOf(String.valueOf('('), result + 1);
			if (indexOfClose < indexOfOpen || indexOfOpen == -1) {
				openBracketMinusClosed--;
				result = indexOfClose;
			} else {
				openBracketMinusClosed++;
				result = indexOfOpen;
			}
			if (indexOfClose == -1) {
				throw new ClipperImportFormatException("Unbalanced Brackets!");
			}
		}
		return result;
	}
	
	/**
	 * Searches for generics and returns the file without generics. They are replaced by spaces.
	 * 
	 * @param file
	 *            file possibly containing generics (Collection<Y extends Exception> list ==> Collection list).
	 * @return file containing no generics but spaces instead
	 * 
	 *         Attention: requires a file without comments and literals
	 * @throws ClipperImportFormatException
	 *             on parsing errors, e.g. when generics brackets are not closed
	 */
	public static String deleteAllGenerics(final String file) throws ClipperImportFormatException {
		String withoutComments = file;
		int start, end;
		final String genericsBeginSign = "<";
		final String genericsEndSign = ">";
		
		int bracketCounter = 0;
		int startBracketOpen = 0;
		int pos = 0;
		
		while (true) {
			start = findNextQuotation(genericsBeginSign, file, pos);
			end = findNextQuotation(genericsEndSign, file, pos);
			if (end == -1) {
				if (bracketCounter != 0) {
					throw new ClipperImportFormatException("Error while parsing generics");
				} else {
					break;
				}
			} else if (start < end && start != -1) {
				pos = start + 1;
				if (bracketCounter == 0) {
					startBracketOpen = start;
				}
				bracketCounter++;
				
			} else if (end < start || start == -1) {
				pos = end + 1;
				bracketCounter--;
			}
			if (bracketCounter == 0) {
				withoutComments = replaceFromToWithWhitespace(withoutComments, startBracketOpen, end + 1);
			}
			
		}
		
		return withoutComments;
	}
	
	/**
	 * Retrieves a full method from the string.
	 * 
	 * @param operation
	 *            operation with method
	 * @param opName
	 *            name of the given operation
	 * @return parsed Operation
	 * @throws NoMatchingMethodException
	 *             when the method cannot be parsed
	 */
	public static GenFullParsedOperationState parseOperation(final String operation, final String opName)
			throws NoMatchingMethodException {
		GenVisibility vis = GenVisibility.DEFAULT;
		final Collection<de.fhdw.wtf.generator.java.generatorModel.GenException> exc = new ArrayList<>();
		String met = CLIPPER_ERROR_NO_METHOD;
		String returntype = null;
		final Collection<GenOperationModifier> modifier = new ArrayList<>();
		String comment;
		
		try {
			comment = "";
			String withoutCommentsFile = ClipperUtils.deleteEndOfLineComments(operation);
			withoutCommentsFile = ClipperUtils.deleteBlockComments(withoutCommentsFile);
			final Matcher m = getMatcher(operation, withoutCommentsFile, opName);
			if (m.matches()) {
				if (hasVisi(operation)) {
					vis = parseVisibility(m.group("visi"));
				}
				if (m.group("met") != null) {
					met = m.group("met").trim();
				}
				returntype = m.group("ret");
				if (hasComment(operation)) {
					comment = m.group("comment");
				}
			} else {
				throw new NoMatchingMethodException(operation);
			}
			
		} catch (final ClipperImportFormatException e) {
			System.out.println(e.getMessage());
			throw new NoMatchingMethodException(operation);
		}
		
		return GenFullParsedOperationState.create(
				GenComment.create(comment),
				exc,
				GenTypeReferenceByName.create(returntype),
				modifier,
				vis,
				met);
	}
	
	private static Matcher getMatcher(final String file, final String withoutCommentsFile, final String opName) {
		String commentReg = "";
		if (hasComment(file)) {
			commentReg = "(?<comment>/\\*.*\\*/)";
		}
		String visiReg = "(?<visi>(public|private|protected))";
		if (!hasVisi(withoutCommentsFile)) {
			visiReg += "?";
		}
		String abstractReg = "(?<abstract>\\sabstract\\s)";
		if (!hasAbstract(withoutCommentsFile)) {
			abstractReg = "";
		}
		final String returntypeReg = "(?<ret>([a-zA-Z_$\\.<>,]+))";
		final String pattern =
				"\\s*" + commentReg + "\\s*(@[A-Za-z_<>\\(\\))]+)?\\s*" + visiReg + abstractReg + "\\s*"
						+ returntypeReg + "\\s+" + opName + "\\s*\\([^\\)]*\\)\\s*(\\{(?<met>.*)\\}|;)";
		final Pattern p = Pattern.compile(pattern, Pattern.DOTALL);
		final Matcher m = p.matcher(file);
		return m;
	}
	
	private static boolean hasAbstract(final String withoutCommentsFile) {
		final Pattern patt = Pattern.compile(".*\\sabstract\\s.*", Pattern.DOTALL);
		final Matcher m = patt.matcher(withoutCommentsFile);
		return m.matches();
	}
	
	/**
	 * Checks whether the file contains an operation visibility.
	 * 
	 * @param file
	 * @return See comment
	 */
	private static boolean hasVisi(final String file) {
		final Pattern patt =
				Pattern.compile(
						".*(\\*/)?(\\s*)?(@[A-Za-z_<>\\(\\))]+)?\\s*(public|protected|private).*",
						Pattern.DOTALL);
		final Matcher m = patt.matcher(file);
		return m.matches();
	}
	
	/**
	 * Checks whether the file contains a leading operation comment.
	 * 
	 * @param file
	 * @return See comment
	 */
	private static boolean hasComment(final String file) {
		final Pattern patt = Pattern.compile("\\s*/\\*.*", Pattern.DOTALL);
		final Matcher m = patt.matcher(file);
		return m.matches();
	}
	
	private static GenVisibility parseVisibility(final String group) {
		switch (group) {
		case "public":
			return GenVisibility.PUBLIC;
		case "protected":
			return GenVisibility.PROTECTED;
		case "private":
			return GenVisibility.PRIVATE;
		default:
			return GenVisibility.DEFAULT;
		}
	}
}
