package de.fhdw.wtf.generator.writer.writer;

import java.util.Collection;
import java.util.HashMap;
import java.util.Vector;

import org.apache.velocity.context.Context;

import de.fhdw.wtf.generator.java.generatorModel.GenAspectAttribute;
import de.fhdw.wtf.generator.java.generatorModel.GenAttribute;
import de.fhdw.wtf.generator.java.generatorModel.GenAttributeModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenFullParsedOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaAttribute;
import de.fhdw.wtf.generator.java.generatorModel.GenOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenPrimitiveClass;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleOperationState;
import de.fhdw.wtf.generator.java.visitor.GenAttributeVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitor;
import de.fhdw.wtf.generator.java.visitor.GenOperationStateVisitor;

/**
 * Writes Java files from the JavaGenerationModel type {@link GenClass}.
 */
public abstract class ClassClassFileWriter extends ClassFileWriter {
	
	private static final String CONSTRUCTOR_KEY = "constructors";
	private static final String CONSTRUCTOR_COMMENT_KEY = "constructorComment";
	static final String ATTRIBUTES_KEY = "attributes";
	
	public ClassClassFileWriter(final String template, final boolean fullyQualified) {
		super(template, fullyQualified);
	}
	
	/**
	 * Writes the file for the given {@link GenClass} c.
	 * 
	 * @param c
	 */
	void writeClassClass(final GenClassClass c) {
		this.writeClass(c);
		this.setUpContext(c, this.getCurrentContext());
	}
	
	/**
	 * Sets up the {@link org.apache.velocity.VelocityContext} with all necessary values for classes.
	 * 
	 * @param c
	 *            {@link GenClassClass} class to write
	 * @param currentContext
	 *            the {@link org.apache.velocity.VelocityContext} for the class
	 */
	private void setUpContext(final GenClassClass c, final Context currentContext) {
		this.setClassModifier(c, currentContext);
		this.setConstructors(c, currentContext);
		this.setExtends(c, currentContext);
	}
	
	/**
	 * Sets all values for the {@link Classmodifier} of {@link GenClass} {@code c} to the {@link Context}.
	 * 
	 * @param c
	 * @param currentContext
	 */
	private void setClassModifier(final GenClassClass c, final Context currentContext) {
		final Vector<String> modifiers = new Vector<>();
		for (final GenClassModifier modifier : c.getModifieres()) {
			modifiers.add(modifier.toString());
		}
		currentContext.put(CLASS_MODIFIERS_KEY, modifiers);
	}
	
	Vector<HashMap<String, Object>> getAttributes(final Collection<? extends GenAttribute> attributes) {
		final Vector<HashMap<String, Object>> result = new Vector<>();
		for (final GenAttribute attribute : attributes) {
			final HashMap<String, Object> current = new HashMap<>();
			current.put(VISIBILITY_KEY, attribute.getVisibility().toString());
			current.put(NAME_KEY, attribute.getName());
			current.put(TYP_KEY, this.typeToString(attribute.getTyp()));
			final Vector<String> modifiers = new Vector<>();
			for (final GenAttributeModifier modifier : attribute.getModifiers()) {
				modifiers.add(modifier.toString());
			}
			current.put(MODIFIERS_KEY, modifiers);
			attribute.accept(new GenAttributeVisitorReturn<GenAttribute>() {
				@Override
				public GenAttribute handleJavaAttribute(final GenJavaAttribute attribute) {
					return attribute;
				}
				
				@Override
				public GenAttribute handleAspectAttribute(final GenAspectAttribute attribute) {
					current.put(TARGED_CLASS_KEY, ClassClassFileWriter.this.typeToString(attribute.getOwner()) + ".");
					return attribute;
				}
			});
			result.add(current);
		}
		return result;
	}
	
	/**
	 * Sets all values for the constructor declaration of {@link GenClass} c to the {@link Context}.
	 * 
	 * @param c
	 * @param currentContext
	 */
	private void setConstructors(final GenClass c, final Context currentContext) {
		c.accept(new GenClassVisitor() {
			@Override
			public void handle(final GenInterfaceClass interfaceClass) {
				// nothing to do
			}
			
			@Override
			public void handle(final GenClassClass classClass) {
				final Vector<HashMap<String, Object>> constructors = new Vector<>();
				for (final GenOperation constructor : classClass.getConstructors()) {
					constructor.getState().accept(new GenOperationStateVisitor() {
						@Override
						public void handle(final GenFullParsedOperationState s) {
							final HashMap<String, Object> current = new HashMap<>();
							current.put(VISIBILITY_KEY, s.getVisibility().toString());
							current.put(
									PARAMETER_KEY,
									ClassClassFileWriter.this.getParamString(constructor.getParameters()));
							current.put(METHOD_KEY, s.getMethodBody());
							current.put(CONSTRUCTOR_COMMENT_KEY, s.getComment().getText());
							current.put(EXCEPTIONS_KEY, ClassClassFileWriter.this.getExceptionString(s.getExceptions()));
							constructors.add(current);
						}
						
						@Override
						public void handle(final GenSimpleOperationState s) {
							// nothing to do
						}
					});
					
				}
				currentContext.put(CONSTRUCTOR_KEY, constructors);
			}
			
			@Override
			public void handle(final GenPrimitiveClass primitiveClass) {
				// nothing to do
			}
		});
	}
	
	private void setExtends(final GenClassClass c, final Context currentContext) {
		if (c.getExtend() != null) {
			currentContext.put(EXTENDS_KEY, EXTENDS + this.typeToString(c.getExtend()));
		} else {
			currentContext.put(EXTENDS_KEY, "");
		}
	}
	
}
