package de.fhdw.wtf.generator.transformer.clipper;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.DependencyTask;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.common.task.result.ExceptionalTaskResult;
import de.fhdw.wtf.common.task.result.OKTaskResult;
import de.fhdw.wtf.common.task.result.TaskResult;
import de.fhdw.wtf.file.FileUtils;
import de.fhdw.wtf.generator.transformer.clipper.internal.ClipperJavaFileShadowCopy;
import de.fhdw.wtf.generator.transformer.exception.ClipperImportFormatException;
import de.fhdw.wtf.generator.transformer.util.Tuple;

/**
 * Extracts methods from current java project into Clipper repository.
 * 
 */
public class ClipToFileTask extends DependencyTask {
	
	/**
	 * Constant for "}".
	 */
	private static final String CURLY_BRACKET_CLOSE = "}";
	
	/**
	 * Constant for "{".
	 */
	private static final String CURLY_BRACKET_OPEN = "{";
	
	/**
	 * Constant for "/".
	 */
	private static final String FILESYSTEM_PATH_SEPERATOR = "/";
	
	/**
	 * Constant for "\\".
	 */
	private static final String BRACKET_CLOSE = "\\)";
	
	/**
	 * Constant for "\\(".
	 */
	private static final String BRACKET_OPEN = "\\(";
	
	/**
	 * The associated clipper configuration.
	 */
	private final ClipperConfiguration configuration;
	
	/**
	 * This task extracts methods, attributes and comments from the Java source code into the Clipper repository.
	 * 
	 * @param taskmanager
	 *            manager that e.g. starts this task when its dependencies are met
	 * @param configuration
	 *            the configuration
	 */
	public ClipToFileTask(final TaskExecutor taskmanager, final ClipperConfiguration configuration) {
		super(taskmanager);
		this.configuration = configuration;
	}
	
	@Override
	public boolean containsTransitive(final DependencyTask a) {
		return false;
	}
	
	@Override
	public TaskResult doWork() {
		try {
			FileUtils.deleteDirectory(this.configuration.getRepoDirectoryRoot());
			this.walkThroughJava(new File(this.configuration.getJavaDirectoryRoot()));
		} catch (final TaskException | IOException e) {
			return new ExceptionalTaskResult(e);
		}
		return new OKTaskResult();
	}
	
	private void walkThroughJava(final File startDir) throws TaskException, IOException {
		final File[] directoryListing = startDir.listFiles();
		if (directoryListing != null) {
			for (final File child : directoryListing) {
				if (child.isDirectory()) {
					this.walkThroughJava(child);
				} else {
					String fullClassName = child.getCanonicalFile().getCanonicalPath();
					fullClassName =
							fullClassName.substring(new File(this.configuration.getJavaDirectoryRoot())
									.getCanonicalPath().length() + 1);
					final File classDir = this.createDirectoryForClass(fullClassName);
					try {
						this.scanFileToDirectory(child, classDir);
					} catch (final ClipperImportFormatException | TaskException e) {
						System.err.println("ClipperImportFormatException(" + fullClassName + "): " + e.getMessage());
						// continue with next file!
						// throw new TaskException(e); //enabled in test TODO
					}
					
				}
				
			}
		} else {
			System.err.println("ClipperImportFormatException(" + startDir + "): Java Directory cannot be accessed.");
		}
	}
	
	private void scanFileToDirectoryInternal(final File path, final ClipperJavaFileShadowCopy javaFile)
			throws IOException, ClipperImportFormatException {
		final File classRepoDir = path;
		
		// Erzeuge Unterordner wenn notwendig
		// --------------------------------------------------------------------------
		if (!classRepoDir.exists()) {
			if (!classRepoDir.mkdir()) {
				throw new ClipperImportFormatException("Ordner '" + classRepoDir + "' konnte nicht erzeugt werden!");
			}
		}
		
		// Iteriere über alle inneren Klassen
		// --------------------------------------------------------------------------
		final Iterator<ClipperJavaFileShadowCopy> it = javaFile.getInnerClasses().iterator();
		while (it.hasNext()) {
			final ClipperJavaFileShadowCopy current = it.next();
			final String newPath = path + "/" + current.getName();
			this.scanFileToDirectoryInternal(new File(newPath), current);
		}
		
		// Hier startet der eigentliche Clipper
		// --------------------------------------------------------------------------
		String file = javaFile.getFullClass();
		file = this.extractNonGenerationPart(classRepoDir, file);
		final String withoutcommentFile = ClipperUtils.simplifyJavaCode(file);
		
		// Klammern ermitteln
		final int classStart = this.extractClassComment(classRepoDir, file, withoutcommentFile);
		final int classEnd = findCorrespondingClosingCurlyBracket(classStart, withoutcommentFile);
		
		// Schneidet den Inhalt der Klasse heraus (= entfernen der Klammern & Co.)
		final String cFile = file.substring(classStart + 1, classEnd);
		final String cWithoutComments = withoutcommentFile.substring(classStart + 1, classEnd);
		
		int pos = 0;
		while (true) {
			int i = cWithoutComments.indexOf(String.valueOf(';'), pos);
			if (i == -1) {
				break;
			}
			i++;
			final String opOrAttWithout = cWithoutComments.substring(pos, i);
			String opOrAttFile = cFile.substring(pos, i);
			opOrAttFile = this.removeLeadingWhiteBreaks(opOrAttFile);
			if (opOrAttWithout.contains("(")) {
				this.parseOperationToFile(opOrAttWithout, opOrAttFile, classRepoDir);
			} else {
				this.parseAttributeToFile(opOrAttWithout, opOrAttFile, classRepoDir);
			}
			pos = i;
		}
		int importsBegin = 0;
		int importsEnd = 0;
		importsBegin = file.indexOf("import ", importsBegin);
		
		if (importsBegin >= 0) {
			int importBegin = importsBegin;
			while (importBegin >= 0) {
				importsEnd = file.indexOf(';', importBegin);
				importBegin = file.indexOf("import", importsEnd);
			}
			importsEnd++;
			final String preCFile = file.substring(0, importsEnd);
			
			final String importsBlockWithout = preCFile.substring(importsBegin, importsEnd);
			final String importsBlockInFile = preCFile.substring(importsBegin, importsEnd);
			this.parseImportsToFile(importsBlockWithout, importsBlockInFile, classRepoDir);
		}
	}
	
	/**
	 * Scans the given class file operations to the local clipper repository.
	 * 
	 * @param classFile
	 * @param classRepoDir
	 * @throws IOException
	 * @throws TaskException
	 * @throws ClipperImportFormatException
	 */
	private void scanFileToDirectory(final File classFile, final File classRepoDir) throws IOException, TaskException,
			ClipperImportFormatException {
		
		// Load the source into "file"
		final String file = FileUtils.getFileString(classFile.getAbsolutePath());
		final ClipperJavaFileShadowCopy clss = new ClipperJavaFileShadowCopy();
		clss.analyze(file);
		
		// Analyze
		this.scanFileToDirectoryInternal(classRepoDir, clss);
		
	}
	
	/**
	 * 
	 * @param opOrAttFile
	 *            String to be bettered
	 * @return better String
	 */
	private String removeLeadingWhiteBreaks(final String opOrAttFile) {
		String result = opOrAttFile;
		boolean lastRunWithoutChanges = false;
		while (!lastRunWithoutChanges) {
			lastRunWithoutChanges = true;
			if (result.startsWith("\t\n")) {
				result = result.replaceFirst("\t\n", "");
				lastRunWithoutChanges = false;
			}
			if (result.startsWith("\n")) {
				result = result.replaceFirst("\n", "");
				lastRunWithoutChanges = false;
			}
			if (result.startsWith("\t\t")) {
				result = result.replaceFirst("\t\t", "\t");
				lastRunWithoutChanges = false;
			}
		}
		return result;
	}
	
	/**
	 * 
	 * @param withoutcomments
	 *            probably needed for further changes
	 * @param file
	 *            that will be saved to directory
	 * @param classRepoDir
	 *            path of directory
	 * @throws IOException
	 *             saveToFile may throw this exception
	 */
	private void parseImportsToFile(final String withoutcomments, final String file, final File classRepoDir)
			throws IOException {
		FileUtils.saveToFile(file, new File(classRepoDir.getAbsolutePath() + FILESYSTEM_PATH_SEPERATOR
				+ ClipperUtils.IMPORTS_FILENAME));
		
	}
	
	private void parseAttributeToFile(final String withoutcomments, final String file, final File classRepoDir)
			throws IOException {
		final String opPattern = "\\s*" + "(?<type>.*)?\\s+" + "(?<name>" + ClipperUtils.IDENTIFIER_REGEX + ")\\s*;";
		final Pattern p = Pattern.compile(opPattern, Pattern.DOTALL);
		final Matcher m = p.matcher(withoutcomments);
		if (m.matches()) {
			final String name = m.group("name");
			final String type = replaceIllegalChars(m.group("type"));
			final String filename = ClipperUtils.ATTRIBUTE_START + name + ClipperUtils.SEPERATOR + type;
			FileUtils.saveToFile(file, new File(classRepoDir.getAbsolutePath() + FILESYSTEM_PATH_SEPERATOR + filename));
			
		}
	}
	
	/**
	 * "Repairs" a parameter type list by gluing parts that belong to the same parameter type. This is necessary as a
	 * type parameter may have generic type arguments which may also be separated by ",".
	 * 
	 * @param params
	 *            The parameter type list resulting from splitting at ",".
	 * @return The repaired type list.
	 */
	private List<String> repairGenerics(final String[] params) {
		final List<String> result = new ArrayList<>();
		int angleBracketsDifference = 0;
		for (final String param : params) {
			if (angleBracketsDifference != 0) {
				result.set(result.size() - 1, result.get(result.size() - 1) + ClipperUtils.PARAMETER_SEPARATOR + param);
			} else {
				result.add(param);
			}
			angleBracketsDifference +=
					param.split(ClipperUtils.GENERIC_OPEN).length - param.split(ClipperUtils.GENERIC_CLOSE).length;
		}
		return result;
	}
	
	private void parseOperationToFile(final String withoutcomments, final String file, final File classRepoDir)
			throws ClipperImportFormatException, IOException {
		final String opPattern =
				"\\s*" + "(?<type>.*\\s+)?" + "(?<name>" + ClipperUtils.IDENTIFIER_REGEX + ")\\s*" + BRACKET_OPEN
						+ "(?<params>.*)" + BRACKET_CLOSE + ".*\\s*;";
		final Pattern p = Pattern.compile(opPattern, Pattern.DOTALL);
		final Matcher m = p.matcher(withoutcomments);
		if (m.matches()) {
			String name = m.group("name");
			final String type = m.group("type");
			String params = m.group("params");
			
			if (type == null) {
				name = ClipperUtils.CONSTRUCTOR_FILENAME;
			}
			if (params.trim().length() > 0) {
				params += ClipperUtils.PARAMETER_SEPARATOR;
				final String[] paramStrings = params.split(ClipperUtils.PARAMETER_SEPARATOR);
				final String nextWordPattern = "\\s*(.*)\\s+(" + ClipperUtils.IDENTIFIER_REGEX + ")\\s*";
				final Pattern p1 = Pattern.compile(nextWordPattern, Pattern.DOTALL);
				for (final String para : this.repairGenerics(paramStrings)) {
					final String trimmedPara = para.trim();
					final Matcher m1 = p1.matcher(trimmedPara);
					if (m1.matches()) {
						final String paramType = trimmedPara.substring(m1.start(1), m1.end(1));
						name += ClipperUtils.SEPERATOR + paramType;
					} else {
						throw new ClipperImportFormatException("Parameter seems invalid: " + params);
					}
				}
			}
			
			if (!name.contains(ClipperUtils.CONSTRUCTOR_FILENAME)) {
				name = replaceIllegalChars(name);
				if (!name.startsWith("$super")) {
					FileUtils.saveToFile(file, new File(classRepoDir.getAbsolutePath() + FILESYSTEM_PATH_SEPERATOR
							+ name));
				}
			}
		} else {
			throw new ClipperImportFormatException("Regex does not match operation signature!");
		}
		
	}
	
	/**
	 * Extracts first comment before class keyword as class comment and saves it.
	 * 
	 * @param classRepoDir
	 * @param file
	 * @param withoutcommentFile
	 * 
	 * @return class start
	 * @throws IOException
	 * @throws ClipperImportFormatException
	 */
	private int extractClassComment(final File classRepoDir, final String file, final String withoutcommentFile)
			throws IOException, ClipperImportFormatException {
		final String classKeyword = "\\s(class|interface)\\s";
		final Pattern p2 = Pattern.compile(classKeyword, Pattern.DOTALL);
		final Matcher m2 = p2.matcher(withoutcommentFile);
		if (m2.find()) {
			final int start = m2.start();
			final Tuple<Integer, Integer> commentBefore =
					ClipperUtils.getCommentBefore(start, file, withoutcommentFile);
			final int commentStart = commentBefore.getA();
			final int commentEnd = commentBefore.getB();
			if (!(commentStart == commentEnd)) {
				FileUtils.saveToFile(file.substring(commentStart, commentEnd), new File(classRepoDir.getAbsolutePath()
						+ FILESYSTEM_PATH_SEPERATOR + ClipperUtils.CLASSCOMMENT_FILENAME));
			}
			return file.indexOf(String.valueOf(CURLY_BRACKET_OPEN), start);
		}
		throw new ClipperImportFormatException("Keyword 'class' or 'interface' not found!");
		
	}
	
	/**
	 * Extracts the non-generation part from the file and saves it in the clipper repository.
	 * 
	 * @param classRepoDir
	 * @param file
	 * 
	 * @return the file without the non generation part
	 * @throws IOException
	 */
	private String extractNonGenerationPart(final File classRepoDir, final String file) throws IOException {
		String fileWithoutNonGenerationPart = file;
		final Pattern p2 = Pattern.compile(ClipperUtils.NON_GEN_PART_MATCH, Pattern.DOTALL);
		final Matcher m2 = p2.matcher(file);
		if (m2.matches()) {
			String nongenpart = file.substring(m2.start(1), m2.end(1));
			nongenpart = nongenpart.replaceFirst("\\r|\\n", "");
			FileUtils.saveToFile(nongenpart, new File(classRepoDir.getAbsolutePath() + FILESYSTEM_PATH_SEPERATOR
					+ ClipperUtils.PROTECTED_AREA_FILENAME));
			fileWithoutNonGenerationPart = file.substring(0, m2.start(1)) + "\r\n" + file.substring(m2.end(1));
		}
		return fileWithoutNonGenerationPart;
	}
	
	/**
	 * Returns the position of the corresponding closing curly bracket.
	 * 
	 * @param openBracketPos
	 *            postion of the {-symbol.
	 * @param file
	 *            file to be searched in
	 * @return position
	 * @throws ClipperImportFormatException
	 *             when the bracket cannot be found
	 */
	public static Integer findCorrespondingClosingCurlyBracket(final int openBracketPos, final String file)
			throws ClipperImportFormatException {
		Integer lastOpenedBracketPos = openBracketPos;
		Integer openBracketMinusClosed = 1;
		while (openBracketMinusClosed != 0) {
			final int indexOfClose = file.indexOf(String.valueOf(CURLY_BRACKET_CLOSE), lastOpenedBracketPos + 1);
			final int indexOfOpen = file.indexOf(String.valueOf(CURLY_BRACKET_OPEN), lastOpenedBracketPos + 1);
			if (indexOfClose < indexOfOpen || indexOfOpen == -1) {
				openBracketMinusClosed--;
				lastOpenedBracketPos = indexOfClose;
			} else {
				openBracketMinusClosed++;
				lastOpenedBracketPos = indexOfOpen;
			}
			if (indexOfClose == -1) {
				throw new ClipperImportFormatException("Unbalanced Brackets!");
			}
		}
		return lastOpenedBracketPos;
	}
	
	/**
	 * Replaces illegal chars in String by replacement character (_). Good for filesystem names.
	 * 
	 * @param name
	 *            name possibly containing illegal characters
	 * @return normalized string without illegal characters
	 */
	public static String replaceIllegalChars(final String name) {
		return name.replaceAll(ClipperUtils.GENERIC_OPEN, ClipperUtils.GENERIC_OPEN_REPLACEMENT).replaceAll(
				ClipperUtils.GENERIC_CLOSE,
				ClipperUtils.GENERIC_CLOSE_REPLACEMENT);
	}
	
	/**
	 * Creates the directory for the class repo.
	 * 
	 * @param fullQualifiedName
	 *            with groups qualified name
	 * @throws TaskException
	 *             when directory cannot be created
	 * @return created path
	 */
	private File createDirectoryForClass(final String fullQualifiedName) throws TaskException {
		String directoryName = fullQualifiedName;
		if (directoryName.toLowerCase().endsWith(ClipperUtils.FILE_ENDING)) {
			directoryName = directoryName.substring(0, fullQualifiedName.length() - ClipperUtils.FILE_ENDING.length());
		}
		final String path =
				this.configuration.getRepoDirectoryRoot() + FILESYSTEM_PATH_SEPERATOR
						+ directoryName.replace(".", FILESYSTEM_PATH_SEPERATOR);
		final File result = new File(path);
		final boolean success = result.mkdirs();
		if (!success) {
			throw new TaskException("Directory creation failed: " + path);
		}
		return result;
	}
	
}
