package de.fhdw.wtf.generator.database.tasks;

import de.fhdw.wtf.common.ast.DatabaseIDSetState;
import de.fhdw.wtf.common.ast.Model;
import de.fhdw.wtf.common.ast.type.AtomicType;
import de.fhdw.wtf.common.ast.type.BaseType;
import de.fhdw.wtf.common.ast.type.ClassType;
import de.fhdw.wtf.common.ast.type.CompositeType;
import de.fhdw.wtf.common.ast.type.ListType;
import de.fhdw.wtf.common.ast.type.MapType;
import de.fhdw.wtf.common.ast.type.ProductType;
import de.fhdw.wtf.common.ast.type.SumType;
import de.fhdw.wtf.common.ast.type.ThrownType;
import de.fhdw.wtf.common.ast.type.Type;
import de.fhdw.wtf.common.ast.type.TypeProxy;
import de.fhdw.wtf.common.ast.visitor.AtomicTypeVisitorException;
import de.fhdw.wtf.common.ast.visitor.CompositeTypeVisitorException;
import de.fhdw.wtf.common.ast.visitor.TypeVisitorException;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.context.model.collections.PersistentList;
import de.fhdw.wtf.context.model.collections.PersistentMap;
import de.fhdw.wtf.generator.database.generation.InitialGenerator;
import de.fhdw.wtf.generator.java.generatorModel.GenClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenExternalInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceWithClassImplClass;
import de.fhdw.wtf.generator.java.generatorModel.GenPrimitiveClass;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenType;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.meta.IntegerType;
import de.fhdw.wtf.persistence.meta.StringType;
import de.fhdw.wtf.persistence.meta.UserType;
import de.fhdw.wtf.walker.walker.SimpleWalkerTaskForTypes;

/**
 * The {@link TypeGenerationTask} puts all classes of the Model (AST) into the database. It also adds Type-Ids to the
 * AST.
 * 
 */
public class TypeGenerationTask extends SimpleWalkerTaskForTypes {
	
	/**
	 * Creates a {@link TypeGenerationTask}.
	 * 
	 * @param m
	 *            The underlying model.
	 * @param genModel
	 *            The underlying generator model.
	 * @param man
	 *            The task manager to use.
	 * @param generator
	 *            The generator to use.
	 */
	public TypeGenerationTask(final Model m,
			final GeneratorModel genModel,
			final TaskExecutor man,
			final InitialGenerator generator) {
		super(m, man, true);
		this.genModel = genModel;
		this.generator = generator;
	}
	
	/**
	 * The generator to use.
	 */
	private final InitialGenerator generator;
	/**
	 * The generator model.
	 */
	private final GeneratorModel genModel;
	
	@Override
	public void handleType(final Type c) throws TaskException {
		c.accept(new TypeGenerator());
	}
	
	@Override
	public void finalizeTask() throws TaskException {
		try {
			final de.fhdw.wtf.persistence.meta.Type anythingType =
					this.generator.getClassFacade().getTypeManager().getTypeforName("generated.sums.Anything");
			final UserType persistentListType =
					this.generator.createClass(PersistentList.class.getName(), false, false);
			this.generator.createSpecialisation(anythingType.getId(), persistentListType.getId());
			this.generator.createUnidirectionalAssociation(
					PersistentList.LINKS_ASSOCIATION_NAME,
					false,
					false,
					persistentListType.getId(),
					anythingType.getId());
			
			final UserType persistentMapType = this.generator.createClass(PersistentMap.class.getName(), false, false);
			this.generator.createSpecialisation(anythingType.getId(), persistentMapType.getId());
			this.generator.createMapAssociation(
					PersistentMap.linksAssociationName,
					false,
					persistentMapType.getId(),
					anythingType.getId(),
					anythingType.getId());
		} catch (final PersistenceException e) {
			throw new TaskException(e);
		}
	}
	
	@Override
	public void beginTask() throws TaskException {
		// nothing to do
	}
	
	/**
	 * Handles a type.
	 */
	private class TypeGenerator implements TypeVisitorException<TaskException> {
		@Override
		public void handle(final AtomicType s) throws TaskException {
			s.accept(new AtomicTypeGenerator());
		}
		
		@Override
		public void handle(final CompositeType c) throws TaskException {
			final GenType genType = TypeGenerationTask.this.genModel.getGenTypeForType(c);
			c.accept(new CompositeTypeGenerator(genType));
		}
		
		@Override
		public void handle(final TypeProxy typeProxy) throws TaskException {
			throw new TaskException("TypeProxy found in TypeGenerationTask");
		}
	}
	
	/**
	 * Handles an atomic type.
	 */
	private class AtomicTypeGenerator implements AtomicTypeVisitorException<TaskException> {
		
		@Override
		public void handle(final BaseType baseType) throws TaskException {
			if (baseType.getTypeName().toString().equals(StringType.STRING_NAME)) {
				baseType.setTypeId(new DatabaseIDSetState(StringType.String_ID));
			} else if (baseType.getTypeName().toString().equals(IntegerType.INTEGER_NAME)) {
				baseType.setTypeId(new DatabaseIDSetState(IntegerType.Integer_ID));
			} else {
				throw new TaskException("Unknown base type: " + baseType.getTypeName());
			}
		}
		
		@Override
		public void handle(final ClassType clazz) throws TaskException {
			final GenClass genType = TypeGenerationTask.this.genModel.getJavaClassForWTFClass(clazz);
			if (genType != null) {
				genType.accept(new ClassTypeGenerator(clazz));
			} else {
				throw new TaskException("TypeGenerationTask: No GenerationModel type for " + clazz.getTypeString());
			}
		}
	}
	
	/**
	 * Handles a class type.
	 */
	private class ClassTypeGenerator implements GenClassVisitorException<TaskException> {
		
		/**
		 * The WTF class.
		 */
		private final ClassType clazz;
		
		/**
		 * Creates a ClassTypeGenerator.
		 * 
		 * @param clazz
		 *            The WTF class.
		 */
		ClassTypeGenerator(final ClassType clazz) {
			this.clazz = clazz;
		}
		
		@Override
		public void handle(final GenPrimitiveClass primitiveClass) {
			// nothing to do
		}
		
		@Override
		public void handle(final GenInterfaceClass interfaceClass) throws TaskException {
			final GenClassVisitorException<TaskException> gcv = this;
			
			interfaceClass.accept(new GenInterfaceClassVisitorException<TaskException>() {
				
				@Override
				public void handle(final GenExternalInterfaceClass iface) {
					// nothing to do
				}
				
				@Override
				public void handle(final GenInterfaceWithClassImplClass interfaceWithImplClass) throws TaskException {
					gcv.handle(interfaceWithImplClass.getClassRepresentation());
				}
				
				@Override
				public void handle(final GenSimpleInterfaceClass simpleInterface) throws TaskException {
					try {
						final UserType newClass =
								TypeGenerationTask.this.generator.createClass(
										simpleInterface.getFullyQualifiedTypeName(),
										false,
										ClassTypeGenerator.this.clazz.isService());
						ClassTypeGenerator.this.clazz.setTypeId(new DatabaseIDSetState(newClass.getId()));
					} catch (final PersistenceException e) {
						throw new TaskException(e);
					}
				}
			});
		}
		
		@Override
		public void handle(final GenClassClass classClass) throws TaskException {
			try {
				final UserType newClass =
						TypeGenerationTask.this.generator.createClass(
								classClass.getFullyQualifiedTypeName(),
								false,
								this.clazz.isService());
				this.clazz.setTypeId(new DatabaseIDSetState(newClass.getId()));
			} catch (final PersistenceException e) {
				throw new TaskException(e);
			}
		}
	}
	
	/**
	 * Handles a composite type.
	 */
	private class CompositeTypeGenerator implements CompositeTypeVisitorException<TaskException> {
		/**
		 * The GeneratorModel type.
		 */
		private final GenType genType;
		
		/**
		 * Creates a CompositeTypeGenerator.
		 * 
		 * @param genType
		 *            The GeneratorModel type.
		 */
		CompositeTypeGenerator(final GenType genType) {
			this.genType = genType;
		}
		
		@Override
		public void handle(final ListType list) {
			// nothing to do, no special list type necessary
		}
		
		@Override
		public void handle(final MapType map) {
			// nothing to do, no special map type necessary
		}
		
		@Override
		public void handle(final ProductType product) throws TaskException {
			try {
				final UserType newClass =
						TypeGenerationTask.this.generator.createClass(
								this.genType.getFullyQualifiedTypeName(),
								false,
								false);
				product.setTypeId(new DatabaseIDSetState(newClass.getId()));
			} catch (final PersistenceException e) {
				throw new TaskException(e);
			}
		}
		
		@Override
		public void handle(final SumType sum) throws TaskException {
			try {
				final UserType newClass =
						TypeGenerationTask.this.generator.createClass(
								this.genType.getFullyQualifiedTypeName(),
								true,
								false);
				sum.setTypeId(new DatabaseIDSetState(newClass.getId()));
			} catch (final PersistenceException e) {
				throw new TaskException(e);
			}
		}
		
		@Override
		public void handle(final ThrownType thrownType) throws TaskException {
			// nothing to do
		}
	}
}
