package de.fhdw.wtf.generator.java.generatorModel;

import java.util.Collection;

import de.fhdw.wtf.generator.java.visitor.GenOperationStateVisitor;
import de.fhdw.wtf.generator.java.visitor.GenOperationStateVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenOperationStateVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenOperationStateVisitorReturnException;

/**
 * This state indicates a full parsed operation. It contains attributes for all elements of the java operation.
 * 
 */
public final class GenFullParsedOperationState implements GenOperationState {
	
	/**
	 * The comment of the operation.
	 */
	private GenComment comment;
	
	/**
	 * The exceptions that can be thrown by calling this operation.
	 */
	private final Collection<GenException> exceptions;
	
	/**
	 * The type that should be returned.
	 */
	private final GenTypeReference returntyp;
	
	/**
	 * The list of modifiers for this operation.
	 */
	private final Collection<GenOperationModifier> modifiers;
	
	/**
	 * The visibility of the operation.
	 */
	private GenVisibility visibility;
	
	/**
	 * The method-body of this operation.
	 */
	private String methodBody;
	
	/**
	 * Constructor of {@link GenFullParsedOperationState}.
	 * 
	 * @param comment
	 *            The comment of the operation.
	 * @param exceptions
	 *            The exceptions that can be thrown by calling this operation.
	 * @param returntyp
	 *            The type that should be returned.
	 * @param modifiers
	 *            The list of modifiers for this operation.
	 * @param visibility
	 *            The visibility of the operation.
	 * @param methodbody
	 *            The method-body of this operation.
	 */
	private GenFullParsedOperationState(final GenComment comment,
			final Collection<GenException> exceptions,
			final GenTypeReference returntyp,
			final Collection<GenOperationModifier> modifiers,
			final GenVisibility visibility,
			final String methodbody) {
		this.comment = comment;
		this.exceptions = exceptions;
		this.modifiers = modifiers;
		this.returntyp = returntyp;
		this.setVisibility(visibility);
		this.methodBody = methodbody;
	}
	
	/**
	 * Creates a new Object of {@link GenFullParsedOperationState}.
	 * 
	 * @param comment
	 *            The comment of the operation.
	 * @param exceptions
	 *            The exceptions that can be thrown by calling this operation.
	 * @param returntyp
	 *            The type that should be returned.
	 * @param modifiers
	 *            The list of modifiers for this operation.
	 * @param visibility
	 *            The visibility of the operation.
	 * @param methodbody
	 *            The method-body of this operation.
	 * @return Object of {@link GenFullParsedOperationState}
	 */
	public static GenFullParsedOperationState create(final GenComment comment,
			final Collection<GenException> exceptions,
			final GenTypeReference returntyp,
			final Collection<GenOperationModifier> modifiers,
			final GenVisibility visibility,
			final String methodbody) {
		return new GenFullParsedOperationState(comment, exceptions, returntyp, modifiers, visibility, methodbody);
	}
	
	/**
	 * The comment of the operation.
	 * 
	 * @return comment
	 */
	public GenComment getComment() {
		return this.comment;
	}
	
	/**
	 * The exceptions that can be thrown by calling of the operation.
	 * 
	 * @return exceptions
	 */
	public Collection<GenException> getExceptions() {
		return this.exceptions;
	}
	
	/**
	 * The return type of the operation.
	 * 
	 * @return returntyp
	 */
	public GenTypeReference getReturntyp() {
		return this.returntyp;
	}
	
	/**
	 * The modifiers of the operation.
	 * 
	 * @return modifiers
	 */
	public Collection<GenOperationModifier> getModifiers() {
		return this.modifiers;
	}
	
	/**
	 * The visibility of the operation.
	 * 
	 * @return visibility
	 */
	public GenVisibility getVisibility() {
		return this.visibility;
	}
	
	/**
	 * The method-body of the operation.
	 * 
	 * @return methodbody
	 */
	public String getMethodBody() {
		return this.methodBody;
	}
	
	/**
	 * Change the comment of the operation.
	 * 
	 * @param comment
	 *            The new comment
	 */
	public void setComment(final GenComment comment) {
		this.comment = comment;
	}
	
	/**
	 * Change the visibility of the operation.
	 * 
	 * @param visibility
	 *            The new visibility
	 */
	public void setVisibility(final GenVisibility visibility) {
		this.visibility = visibility;
	}
	
	/**
	 * Change the method of the operation.
	 * 
	 * @param methodBody
	 *            The new method-body
	 */
	public void setMethodBody(final String methodBody) {
		this.methodBody = methodBody;
	}
	
	@Override
	public void accept(final GenOperationStateVisitor visitor) {
		visitor.handle(this);
	}
	
	@Override
	public <X> X accept(final GenOperationStateVisitorReturn<X> visitor) {
		return visitor.handle(this);
	}
	
	@Override
	public <Y extends Exception> void accept(final GenOperationStateVisitorException<Y> visitor) throws Y {
		visitor.handle(this);
	}
	
	@Override
	public <X, Y extends Exception> X accept(final GenOperationStateVisitorReturnException<X, Y> visitor) throws Y {
		return visitor.handle(this);
	}
}
