package de.fhdw.wtf.generator.java.generatorModel;

/**
 * A GenQualifiedPackage represents a full qualified package. It is a concatenation of multiple GenPackages.
 */
public final class GenQualifiedPackage extends GenPackage {
	
	/**
	 * The beginning of this qualified package.
	 */
	private final GenUnqualifiedPackage first;
	
	/**
	 * The rest of this qualified package.
	 */
	private final GenPackage rest;
	
	/**
	 * The symbol to separate unqualified packages.
	 */
	private static final String JAVA_PACKAGE_SEPARATOR = ".";
	
	/**
	 * Instantiates a new GenQualifiedPackage with the given parameters.
	 * 
	 * @param first
	 *            The {@link GenUnqualifiedPackage} that is the begin of this package.
	 * @param rest
	 *            The rest of this qualified package.
	 */
	private GenQualifiedPackage(final GenUnqualifiedPackage first, final GenPackage rest) {
		super();
		this.first = first;
		this.rest = rest;
	}
	
	/**
	 * Returns a new instance of {@link GenQualifiedPackage} and copies all attribute-values from
	 * <code>packageToCopy</code>.
	 * 
	 * @param packageToCopy
	 *            The {@link GenQualifiedPackage} to copy attribute-values from.
	 */
	private GenQualifiedPackage(final GenQualifiedPackage packageToCopy) {
		super();
		this.first = (GenUnqualifiedPackage) packageToCopy.getFirst().copy();
		this.rest = packageToCopy.getRest().copy();
	}
	
	/**
	 * Creates a new GenQualifiedPackage with the given parameters.
	 * 
	 * @param first
	 *            The {@link GenUnqualifiedPackage} that is the begin of this package.
	 * @param rest
	 *            The rest of this qualified package.
	 * @return {@link GenQualifiedPackage}
	 */
	public static GenQualifiedPackage create(final GenUnqualifiedPackage first, final GenPackage rest) {
		return new GenQualifiedPackage(first, rest);
	}
	
	@Override
	public GenPackage addName(final String name) {
		return this.addPackage(GenUnqualifiedPackage.create(name));
	}
	
	@Override
	public String toString() {
		String result = this.getFirst().toString();
		result += JAVA_PACKAGE_SEPARATOR;
		result += this.getRest().toString();
		return result;
	}
	
	@Override
	public GenPackage addPackage(final GenPackage packag) {
		return GenQualifiedPackage.create(this.getFirst(), this.getRest().addPackage(packag));
	}
	
	/**
	 * Returns the beginning of this qualified package.
	 * 
	 * @return {@link GenUnqualifiedPackage}
	 */
	public GenUnqualifiedPackage getFirst() {
		return this.first;
	}
	
	/**
	 * Returns the rest of this qualified package.
	 * 
	 * @return {@link GenPackage}
	 */
	public GenPackage getRest() {
		return this.rest;
	}
	
	@Override
	public GenPackage copy() {
		return new GenQualifiedPackage(this);
	}
	
}
