package de.fhdw.wtf.generator.transformer.transformers.classTransformer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Vector;

import de.fhdw.wtf.common.ast.Model;
import de.fhdw.wtf.common.ast.Operation;
import de.fhdw.wtf.common.ast.UnqualifiedName;
import de.fhdw.wtf.common.ast.type.AtomicType;
import de.fhdw.wtf.common.ast.type.BaseType;
import de.fhdw.wtf.common.ast.type.ClassModifier;
import de.fhdw.wtf.common.ast.type.ClassModifierAbstract;
import de.fhdw.wtf.common.ast.type.ClassModifierService;
import de.fhdw.wtf.common.ast.type.ClassModifierTransient;
import de.fhdw.wtf.common.ast.type.ClassModifierVisitable;
import de.fhdw.wtf.common.ast.type.ClassType;
import de.fhdw.wtf.common.ast.type.CompositeType;
import de.fhdw.wtf.common.ast.type.ExceptionClassType;
import de.fhdw.wtf.common.ast.type.ListType;
import de.fhdw.wtf.common.ast.type.MapType;
import de.fhdw.wtf.common.ast.type.ProductType;
import de.fhdw.wtf.common.ast.type.RegularClassType;
import de.fhdw.wtf.common.ast.type.SumType;
import de.fhdw.wtf.common.ast.type.ThrownType;
import de.fhdw.wtf.common.ast.type.Type;
import de.fhdw.wtf.common.ast.type.TypeProxy;
import de.fhdw.wtf.common.ast.visitor.AtomicTypeVisitorException;
import de.fhdw.wtf.common.ast.visitor.AtomicTypeVisitorReturn;
import de.fhdw.wtf.common.ast.visitor.AtomicTypeVisitorReturnException;
import de.fhdw.wtf.common.ast.visitor.ClassModifierVisitor;
import de.fhdw.wtf.common.ast.visitor.ClassTypeVisitorException;
import de.fhdw.wtf.common.ast.visitor.ClassTypeVisitorReturnException;
import de.fhdw.wtf.common.ast.visitor.CompositeTypeVisitorReturn;
import de.fhdw.wtf.common.ast.visitor.CompositeTypeVisitorReturnException;
import de.fhdw.wtf.common.ast.visitor.TypeVisitorException;
import de.fhdw.wtf.common.ast.visitor.TypeVisitorReturn;
import de.fhdw.wtf.common.ast.visitor.TypeVisitorReturnException;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.facade.PackageConstants;
import de.fhdw.wtf.facade.TypeNameGenerator;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenCollectionType;
import de.fhdw.wtf.generator.java.generatorModel.GenComment;
import de.fhdw.wtf.generator.java.generatorModel.GenDummyType;
import de.fhdw.wtf.generator.java.generatorModel.GenExternalInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenIntegerType;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenMutableList;
import de.fhdw.wtf.generator.java.generatorModel.GenMutableMap;
import de.fhdw.wtf.generator.java.generatorModel.GenPackage;
import de.fhdw.wtf.generator.java.generatorModel.GenStringType;
import de.fhdw.wtf.generator.java.generatorModel.GenType;
import de.fhdw.wtf.generator.java.generatorModel.GenVoidType;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;
import de.fhdw.wtf.generator.transformer.exception.GenTypeNotReferencedException;
import de.fhdw.wtf.generator.transformer.util.Tuple;
import de.fhdw.wtf.walker.walker.HelperUtils;
import de.fhdw.wtf.walker.walker.SimpleWalkerTaskForTypes;

/**
 * Transformation-Task from WTF to JavaGenerationModel. <br>
 * Transforms all {@link Type}s on the given WTF-Model into classes of the JavaGenerationModel. <br>
 * <br>
 * <b>Does not cover inheritance!</b> <br>
 */
public class TypeTransformer extends SimpleWalkerTaskForTypes {
	
	private final GeneratorModel genModel;
	private final UtilTransformer utilTransformer;
	private final Map<Tuple<GenType, Collection<ExceptionClassType>>, Type> typeMapping;
	
	/**
	 * This map contains all {@link GenMutableMap}s, because their key-type or their value-type may reference a
	 * {@link GenDummyType}. The value contains the original {@link MapType} that key-type and value-type should be used
	 * instead.
	 */
	private final Map<GenMutableMap, MapType> mapTypeMapping;
	
	/**
	 * This map contains all {@link GenCollectionType}s, because their value-type may reference a {@link GenDummyType}.
	 * The value contains the original {@link ListType} that value-type should be used instead.
	 */
	private final Map<GenCollectionType, ListType> listTypeMapping;
	
	/**
	 * This map contains all {@link GenInterfaceClass}es, because their value-type may reference a {@link GenDummyType}.
	 * The value contains the original {@link Type} that value-type should be used instead.
	 */
	private final Map<GenInterfaceClass, Type> interfaceString;
	
	// Workaround to handle ClassTypes in InheritanceTransformer
	private final Collection<ClassType> foundClassTypes = new HashSet<>();
	
	public Collection<ClassType> getFoundClassTypes() {
		return this.foundClassTypes;
	}
	
	/**
	 * String represents the default class comment.
	 */
	private static final String DEFAULT_CLASS_COMMENT = "TODO Comment Class.";
	
	/**
	 * String represents the default exception class comment.
	 */
	private static final String DEFAULT_EXCEPTION_COMMENT = "TODO Comment Exception.";
	
	/**
	 * String represents the default product class comment.
	 */
	private static final String DEFAULT_PRODUCT_COMMENT = "This class represents an anonym product.";
	
	/**
	 * String represents the default sum class comment.
	 */
	private static final String DEFAULT_SUM_COMMENT = "";
	
	/**
	 * String represents the default non generated part.
	 */
	private static final String DEFAULT_NON_GEN_PART = "";
	
	/**
	 * String represents the name of the Anything sum.
	 */
	private static final String ANYTHING_SUM_NAME = "Anything";
	
	public TypeTransformer(final Model m, final TaskExecutor taskmanager, final GeneratorModel generatorModel) {
		super(m, taskmanager, true);
		this.genModel = generatorModel;
		this.utilTransformer = UtilTransformer.create(this.getGeneratorModel());
		this.typeMapping = new HashMap<>();
		this.mapTypeMapping = new HashMap<>();
		this.listTypeMapping = new HashMap<>();
		this.interfaceString = new HashMap<>();
	}
	
	@Override
	public void handleType(final Type type) throws TaskException {
		
		final Tuple<GenType, Collection<ExceptionClassType>> genTypes = this.createGenType(type);
		
		if (genTypes.getA() != null) {
			TypeTransformer.this.getGeneratorModel().addToMapping(type, genTypes);
		}
	}
	
	private Tuple<GenType, Collection<ExceptionClassType>> createGenType(final Type type) throws TaskException {
		return type
				.accept(new TypeVisitorReturnException<Tuple<GenType, Collection<ExceptionClassType>>, TaskException>() {
					
					@Override
					public Tuple<GenType, Collection<ExceptionClassType>> handle(final AtomicType s)
							throws TaskException {
						
						return s.accept(new AtomicTypeVisitorReturnException<Tuple<GenType, Collection<ExceptionClassType>>, TaskException>() {
							
							@Override
							public Tuple<GenType, Collection<ExceptionClassType>> handle(final BaseType baseType)
									throws TaskException {
								if (baseType.toString().equals(
										de.fhdw.wtf.common.constants.referencer.BaseTypeConstants.STRING_CONSTANT)) {
									return new Tuple<GenType, Collection<ExceptionClassType>>(GenStringType
											.getInstance(), new ArrayList<ExceptionClassType>());
								} else if (baseType.toString().equals(
										de.fhdw.wtf.common.constants.referencer.BaseTypeConstants.INTEGER_CONSTANT)) {
									return new Tuple<GenType, Collection<ExceptionClassType>>(GenIntegerType
											.getInstance(), new ArrayList<ExceptionClassType>());
								} else {
									return new Tuple<GenType, Collection<ExceptionClassType>>(null,
											new ArrayList<ExceptionClassType>());
								}
							}
							
							@Override
							public Tuple<GenType, Collection<ExceptionClassType>> handle(final ClassType c)
									throws TaskException {
								
								// Workaround to handle ClassTypes in InheritanceTransformer
								TypeTransformer.this.foundClassTypes.add(c);
								
								final String className = c.getName().getLastAddedName().toString();
								final GenPackage packag =
										TypeTransformer.this.getUtilTransformer().getPackage(c.getName());
								final Collection<GenClassModifier> modifiers =
										TypeTransformer.this.handleClassModifier(c.getModifiers());
								
								return c.accept(new ClassTypeVisitorReturnException<Tuple<GenType, Collection<ExceptionClassType>>, TaskException>() {
									
									@Override
									public Tuple<GenType, Collection<ExceptionClassType>> handle(final RegularClassType regularClassType)
											throws TaskException {
										final GenComment comment =
												GenComment.createFromPlainText(DEFAULT_CLASS_COMMENT, false);
										
										if (TypeTransformer.this.isInterface(c)) {
											final GenType result =
													TypeTransformer.this.getGeneratorModel().createInterface(
															className,
															c,
															comment,
															packag,
															DEFAULT_NON_GEN_PART);
											
											return new Tuple<GenType, Collection<ExceptionClassType>>(result,
													new ArrayList<ExceptionClassType>());
										} else {
											final GenType result =
													TypeTransformer.this.getGeneratorModel()
															.createInterfaceWithImplClass(
																	className,
																	c,
																	modifiers,
																	comment,
																	packag,
																	DEFAULT_NON_GEN_PART);
											
											return new Tuple<GenType, Collection<ExceptionClassType>>(result,
													new ArrayList<ExceptionClassType>());
										}
									}
									
									@Override
									public Tuple<GenType, Collection<ExceptionClassType>> handle(final ExceptionClassType exceptionClassType)
											throws TaskException {
										final GenComment comment =
												GenComment.createFromPlainText(DEFAULT_EXCEPTION_COMMENT, false);
										final GenType result =
												TypeTransformer.this.getGeneratorModel().createException(
														className,
														c,
														comment,
														packag,
														DEFAULT_NON_GEN_PART);
										
										return new Tuple<GenType, Collection<ExceptionClassType>>(result,
												new ArrayList<ExceptionClassType>());
									}
								});
							}
						});
					}
					
					@Override
					public Tuple<GenType, Collection<ExceptionClassType>> handle(final CompositeType c)
							throws TaskException {
						return c.accept(new CompositeTypeVisitorReturnException<Tuple<GenType, Collection<ExceptionClassType>>, TaskException>() {
							
							@Override
							public Tuple<GenType, Collection<ExceptionClassType>> handle(final MapType map)
									throws TaskException {
								final GenType key = TypeTransformer.this.tryGetType(map.getKey());
								final GenType value = TypeTransformer.this.tryGetType(map.getOf());
								final GenMutableMap coll = GenMutableMap.create(key, value);
								TypeTransformer.this.mapTypeMapping.put(coll, map);
								
								return new Tuple<GenType, Collection<ExceptionClassType>>(coll,
										new ArrayList<ExceptionClassType>());
							}
							
							@Override
							public Tuple<GenType, Collection<ExceptionClassType>> handle(final ListType list)
									throws TaskException {
								final GenType valuetype = TypeTransformer.this.tryGetType(list.getOf());
								final GenMutableList coll = GenMutableList.create(valuetype);
								TypeTransformer.this.listTypeMapping.put(coll, list);
								
								return new Tuple<GenType, Collection<ExceptionClassType>>(coll,
										new ArrayList<ExceptionClassType>());
							}
							
							@Override
							public Tuple<GenType, Collection<ExceptionClassType>> handle(final ProductType product)
									throws TaskException {
								final String pName = TypeNameGenerator.getInstance().getTypeName(product).toString();
								final GenComment comment =
										GenComment.createFromPlainText(TypeTransformer.DEFAULT_PRODUCT_COMMENT, false);
								final GenClassClass prod =
										TypeTransformer.this.getGeneratorModel().createClass(
												pName,
												new Vector<GenClassModifier>(),
												product,
												comment,
												PackageConstants.PRODUCT_PACKAGE,
												"");
								
								return new Tuple<GenType, Collection<ExceptionClassType>>(prod,
										new ArrayList<ExceptionClassType>());
							}
							
							@Override
							public Tuple<GenType, Collection<ExceptionClassType>> handle(final SumType sum)
									throws TaskException {
								
								final Collection<ExceptionClassType> exceptions = new ArrayList<>();
								final Collection<Type> otherElements = new ArrayList<>();
								final GenType genType;
								
								final Iterator<Type> subElementsIterator = sum.getElements().iterator();
								while (subElementsIterator.hasNext()) {
									final Type current = subElementsIterator.next();
									
									if (current.isThrownType()) {
										final ThrownType thrownType = (ThrownType) current;
										thrownType.getUnderlyingType().accept(
												new TypeVisitorException<TaskException>() {
													
													@Override
													public void handle(final TypeProxy typeProxy) throws TaskException {
														throw new TaskException(
																"TypeProxy cannot be the underlying type of a thrownType.");
													}
													
													@Override
													public void handle(final CompositeType compositeType)
															throws TaskException {
														throw new TaskException(
																"CompositeType cannot be the underlying type of a thrownType.");
													}
													
													@Override
													public void handle(final AtomicType atomicType)
															throws TaskException {
														atomicType
																.accept(new AtomicTypeVisitorException<TaskException>() {
																	
																	@Override
																	public void handle(final ClassType clazz)
																			throws TaskException {
																		clazz.accept(new ClassTypeVisitorException<TaskException>() {
																			
																			@Override
																			public void handle(final ExceptionClassType exceptionClassType)
																					throws TaskException {
																				exceptions.add(exceptionClassType);
																			}
																			
																			@Override
																			public void handle(final RegularClassType regularClassType)
																					throws TaskException {
																				throw new TaskException(
																						"RegularClassType cannot be the underlying type of a thrownType.");
																			}
																		});
																	}
																	
																	@Override
																	public void handle(final BaseType baseType)
																			throws TaskException {
																		throw new TaskException(
																				"BaseType cannot be the underlying type of a thrownType.");
																	}
																});
													}
												});
										
									} else {
										otherElements.add(current);
									}
								}
								
								if (otherElements.size() == 0) {
									genType = GenVoidType.getInstance();
								} else if (otherElements.size() == 1) {
									genType =
											TypeTransformer.this.createGenType(otherElements.iterator().next()).getA();
								} else {
									final UnqualifiedName sumName;
									if (sum.equals(TypeTransformer.this.getModel().getAnything().getPrototype())) {
										sumName =
												TypeNameGenerator.getInstance().createName(
														TypeTransformer.ANYTHING_SUM_NAME);
									} else {
										sumName = TypeNameGenerator.getInstance().getTypeName(sum);
									}
									final GenComment comment =
											GenComment.createFromPlainText(DEFAULT_SUM_COMMENT, false);
									
									final GenInterfaceClass sumInterface =
											TypeTransformer.this.getGeneratorModel().createInterface(
													sumName.toString(),
													sum,
													comment,
													PackageConstants.SUM_PACKAGE,
													"");
									
									// Make the generated Anything type a subinterface of
									// de.fhdw.wtf.context.model.Anything.
									// This allows to use context collection classes with generated interface types, as
									// the
									// collection classes require their element types to specialize
									// de.fhdw.wtf.context.model.Anything.
									if (sum.equals(TypeTransformer.this.getModel().getAnything().getPrototype())) {
										sumInterface.getImplement().add(
												GenExternalInterfaceClass
														.getInstance("de.fhdw.wtf.context.model.Anything"));
									}
									
									genType = sumInterface;
								}
								
								return new Tuple<>(genType, exceptions);
							}
							
							@Override
							public Tuple<GenType, Collection<ExceptionClassType>> handle(final ThrownType thrownType)
									throws TaskException {
								final ArrayList<ExceptionClassType> list = new ArrayList<>();
								if (thrownType.getUnderlyingType() instanceof ExceptionClassType) {
									list.add((ExceptionClassType) thrownType.getUnderlyingType());
								}
								return new Tuple<GenType, Collection<ExceptionClassType>>(GenVoidType.getInstance(),
										list);
							}
						});
					}
					
					@Override
					public Tuple<GenType, Collection<ExceptionClassType>> handle(final TypeProxy s)
							throws TaskException {
						return TypeTransformer.this.createGenType(UtilTransformer.getTypeProxyFreePrototype(s));
					}
				});
	}
	
	/**
	 * Changes the {@link GenDummyType}s in the internal collection to the right types.
	 *
	 * @throws GenTypeNotReferencedException
	 */
	private void changeAspectJInterfaceName() throws GenTypeNotReferencedException {
		final Iterator<Entry<GenInterfaceClass, Type>> i = this.getInterfaceString().entrySet().iterator();
		while (i.hasNext()) {
			final Entry<GenInterfaceClass, Type> current = i.next();
			current.getKey().setName(this.getGeneratorModel().getGenTypeForType(current.getValue()).toString());
		}
	}
	
	/**
	 * Change the {@link GenType} of all {@link GenDummyType}s in <code>mapTypeMapping</code> to the right
	 * {@link GenType}.
	 *
	 * @throws GenTypeNotReferencedException
	 */
	private void changeDummyTypeMapType() throws GenTypeNotReferencedException {
		final Iterator<Entry<GenMutableMap, MapType>> iMapType = this.mapTypeMapping.entrySet().iterator();
		while (iMapType.hasNext()) {
			final Entry<GenMutableMap, MapType> current = iMapType.next();
			final GenType currentKey = this.getGeneratorModel().getGenTypeForType(current.getValue().getKey());
			final GenType currentValue = this.getGeneratorModel().getGenTypeForType(current.getValue().getOf());
			
			current.getKey().setKey(currentKey);
			current.getKey().setType(currentValue);
		}
	}
	
	/**
	 * Change the {@link GenType} of all {@link GenDummyType}s in <code>listTypeMapping</code> to the right
	 * {@link GenType}.
	 *
	 * @throws GenTypeNotReferencedException
	 */
	private void changeDummyTypeListType() throws GenTypeNotReferencedException {
		final Iterator<Entry<GenCollectionType, ListType>> iListType = this.listTypeMapping.entrySet().iterator();
		while (iListType.hasNext()) {
			final Entry<GenCollectionType, ListType> current = iListType.next();
			final GenType currentValue = this.getGeneratorModel().getGenTypeForType(current.getValue().getOf());
			
			current.getKey().setType(currentValue);
		}
	}
	
	/**
	 * Tries to get the {@link GenType} for the given {@link Type}. Could fail, because the Type may not be referenced
	 * yet. If it fails, it returns a {@link GenDummyType} instead.
	 *
	 * @param type
	 *            The AST-Type to find a GenType for.
	 * @return {@link GenType}. May be {@link GenDummyType}.
	 */
	private GenType tryGetType(final Type type) {
		GenType result;
		try {
			result = this.getGeneratorModel().getGenTypeForType(type);
		} catch (final GenTypeNotReferencedException e) {
			return GenDummyType.create();
		}
		return result;
	}
	
	/**
	 * Returns true if <code>c</code> is an Interface. To be an Interface <code>c</code> must be abstract and does not
	 * have any attributes or a operation with method, otherwise false.
	 * 
	 * @return {@code true} if {@code type} is an interface. {@code false} otherwise.
	 */
	private boolean isInterface(final Type type) {
		// Check that ALL super types are interfaces too
		final Iterator<Type> superTypeIterator = type.getSuperTypes().iterator();
		while (superTypeIterator.hasNext()) {
			final Type currentSuperType = superTypeIterator.next();
			if (!TypeTransformer.this.isInterface(currentSuperType)) {
				return false;
			}
		}
		// ALL super types are interfaces
		return type.accept(new TypeVisitorReturn<Boolean>() {
			
			@Override
			public Boolean handle(final AtomicType atomicType) {
				return atomicType.accept(new AtomicTypeVisitorReturn<Boolean>() {
					
					@Override
					public Boolean handle(final BaseType baseType) {
						// "Int" and "Str" will be generated as classes
						return false;
					}
					
					@Override
					public Boolean handle(final ClassType clazz) {
						return clazz.isAbstract() && !TypeTransformer.this.hasAtLeastOneConcreteOperation(clazz)
								&& clazz.getAttributes().isEmpty();
					}
				});
			}
			
			@Override
			public Boolean handle(final CompositeType compositeType) {
				return compositeType.accept(new CompositeTypeVisitorReturn<Boolean>() {
					
					@Override
					public Boolean handle(final ListType list) {
						// Lists are classes with generics
						return false;
					}
					
					@Override
					public Boolean handle(final MapType map) {
						// Maps are classes with generics
						return false;
					}
					
					@Override
					public Boolean handle(final ProductType product) {
						// Products are classes
						return false;
					}
					
					@Override
					public Boolean handle(final SumType sum) {
						// sum will be generated as interfaces
						return true;
					}
					
					@Override
					public Boolean handle(final ThrownType thrownType) {
						// Thrown Types are only Exceptions and exceptions aren't interfaces
						return false;
					}
				});
			}
			
			@Override
			public Boolean handle(final TypeProxy typeProxy) {
				return TypeTransformer.this.isInterface(HelperUtils.getReferencedType(typeProxy));
			}
		});
	}
	
	/**
	 * Returns true if and only if class c has one or more concrete operation. Therefore it returns false if all
	 * operations are abstract.
	 * 
	 * @param clazz
	 *            class that will be checked
	 * @return boolean. If this is true, then the class <code>clazz</code> contains at least one concrete Operation. If
	 *         this is false, then the class doesn't contain any operation.
	 */
	private boolean hasAtLeastOneConcreteOperation(final ClassType clazz) {
		final Iterator<Operation> iOp = clazz.getOperations().iterator();
		while (iOp.hasNext()) {
			final Operation current = iOp.next();
			if (!current.isAbstract()) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Handles the occurrence of {@link ClassModifier}.
	 *
	 * @param modifiers
	 *            the given {@link ClassModifier}
	 * @return the list of class modifiers as Strings.
	 */
	private Collection<GenClassModifier> handleClassModifier(final Collection<ClassModifier> modifiers) {
		final Vector<GenClassModifier> modifier = new Vector<>();
		final Iterator<ClassModifier> i = modifiers.iterator();
		while (i.hasNext()) {
			final ClassModifier current = i.next();
			current.accept(new ClassModifierVisitor() {
				@Override
				public boolean handle(final ClassModifierService service) {
					// TODO Handle Service Modifier
					return false;
				}
				
				@Override
				public boolean handle(final ClassModifierTransient tranzient) {
					// TODO handle transient class
					return true;
				}
				
				@Override
				public boolean handle(final ClassModifierAbstract abstract1) {
					modifier.add(GenClassModifier.ABSTRACT);
					return true;
				}
				
				@Override
				public boolean handle(final ClassModifierVisitable visitable) {
					// TODO handle visitable modifier
					return true;
				}
			});
		}
		return modifier;
	}
	
	public GeneratorModel getGeneratorModel() {
		return this.genModel;
	}
	
	public Map<Tuple<GenType, Collection<ExceptionClassType>>, Type> getTypeMapping() {
		return this.typeMapping;
	}
	
	public UtilTransformer getUtilTransformer() {
		return this.utilTransformer;
	}
	
	@Override
	public String toString() {
		return "Type Generation";
	}
	
	public Map<GenInterfaceClass, Type> getInterfaceString() {
		return this.interfaceString;
	}
	
	@Override
	public void beginTask() throws TaskException {
		// create Classes for BaseTypes and special Types
		this.handleType(this.getModel().getInteger());
		this.handleType(this.getModel().getString());
	}
	
	@Override
	public void finalizeTask() throws TaskException {
		// create Classes for BaseTypes and special Types
		this.handleType(this.getModel().getAnything().getPrototype());
		
		this.changeDummyTypeMapType();
		this.changeDummyTypeListType();
		this.changeAspectJInterfaceName();
	}
}
