package de.fhdw.wtf.generator.java.generatorModel;

import de.fhdw.wtf.generator.java.visitor.GenTypeVisitor;
import de.fhdw.wtf.generator.java.visitor.GenTypeVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenTypeVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenericTypeStateVisitorReturn;

/**
 * Instances of the class Generic represent Generics like known from Java, that shall for example be generated amongst
 * some Attributes or Methods.
 */
public final class Generic extends GenType implements Comparable<Generic> {
	
	/**
	 * Separates a generic's name and bound.
	 */
	private static final String BOUND_SEPARATOR = " extends ";
	
	/**
	 * The name of this Generic.
	 */
	private final String name;
	/**
	 * The state whether the Generic actually has Generics or not. TODO is this correct? Also may correct comments for
	 * GenericTypeState.
	 */
	private final GenericTypeState type;
	
	/**
	 * Creates a new Generic with the given name and type.
	 * 
	 * @param name
	 *            The name that represents the created Generic.
	 * @param type
	 *            The type whether the Generic actually has Generics or not. TODO is this correct? Also may correct
	 *            comments for GenericTypeState.
	 */
	private Generic(final String name, final GenericTypeState type) {
		super(name);
		this.name = name;
		this.type = type;
	}
	
	/**
	 * Creates a new Generic with the given name and type.
	 * 
	 * @param name
	 *            The name that represents the created Generic.
	 * @param type
	 *            The type whether the Generic actually has Generics or not. TODO is this correct? Also may correct
	 *            comments for GenericTypeState.
	 * @return Returns the created Generic.
	 */
	public static Generic create(final String name, final GenericTypeState type) {
		return new Generic(name, type);
	}
	
	/*
	 * Projections.
	 */
	
	/**
	 * Returns the GenericName of the Generic.
	 * 
	 * @return GenericName
	 */
	public String getGenericName() {
		
		return this.name;
	}
	
	/**
	 * Return the type of the Generic.
	 * 
	 * @return GenericTypeState
	 */
	public GenericTypeState getGenericType() {
		return this.type;
	}
	
	@Override
	public String getFullyQualifiedTypeName() {
		
		return this.getName();
	}
	
	/**
	 * Returns the name of the Type referenced by this generic. Returns the empty string if no real type is referenced
	 * by this generic
	 * 
	 * @return the type name.
	 */
	public String getTypeName() {
		final String stateName = this.type.accept(new GenericTypeStateVisitorReturn<String>() {
			
			@Override
			public String handle(final GenHasGenericType hasType) {
				return hasType.getType().getFullyQualifiedTypeNameWithGenericArguments();
			}
			
			@Override
			public String handle(final GenHasNoGenericType hasNoType) {
				return "";
			}
			
		});
		return stateName;
	}
	
	@Override
	public String getFullyQualifiedTypeNameWithGenericArguments() {
		return this.getFullyQualifiedTypeName();
	}
	
	@Override
	public void accept(final GenTypeVisitor visitor) {
		visitor.handle(this);
		
	}
	
	@Override
	public <X> X accept(final GenTypeVisitorReturn<X> visitor) {
		return visitor.handle(this);
	}
	
	@Override
	public <X extends java.lang.Exception> void accept(final GenTypeVisitorException<X> visitor) throws X {
		visitor.handle(this);
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof Generic) {
			final Generic genObj = (Generic) obj;
			if (this.getFullyQualifiedTypeName().equals(genObj.getFullyQualifiedTypeName())
					&& this.getGenericName().equals(genObj.getGenericName())
					&& this.getGenericType().equals(genObj.getGenericType()) && this.getName().equals(genObj.getName())) {
				return true;
			}
		}
		return false;
	}
	
	@Override
	public int hashCode() {
		return this.getFullyQualifiedTypeName().hashCode() ^ this.getGenericName().hashCode()
				^ this.getGenericType().hashCode() ^ this.getName().hashCode();
	}
	
	@Override
	public int compareTo(final Generic o) {
		return this.getGenericName().compareTo(o.getGenericName());
	}
	
	@Override
	public String toString() {
		String result = this.getName();
		result += this.getGenericType().accept(new GenericTypeStateVisitorReturn<String>() {
			@Override
			public String handle(final GenHasGenericType hasType) {
				return BOUND_SEPARATOR + hasType.getType().getFullyQualifiedTypeName();
			}
			
			@Override
			public String handle(final GenHasNoGenericType hasNoType) {
				return "";
			}
		});
		return result;
	}
	
}
