package de.fhdw.wtf.generator.java.generatorModel;

import java.util.Collection;
import java.util.Vector;

import de.fhdw.wtf.generator.java.visitor.GenClassVisitor;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitor;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenerateAsStateVisitor;

/**
 * A GenInterfaceWithClassImplClass represents a GenInterfaceClass that contains a GenClassClass to represent an
 * AST-type. This makes it possible for interfaces to implement the AST-type that is represented by the GenClassClass.
 */
public final class GenInterfaceWithClassImplClass extends GenInterfaceClass {
	
	/**
	 * This state decides whether this should be generated as interface containing an "Impl"-class or only as class.
	 */
	private GenerateAsState generateAsState;
	
	/**
	 * If this shall be generated as interface with a inner class, this will be the name of the inner class.
	 */
	private static final String CLASS_NAME = "Impl";
	
	/**
	 * Represents the String This.
	 */
	private static final String THIS = "This";
	
	/**
	 * The classRepresentation represents the AST-type as class and is a private inner class "Impl" inside of this
	 * interface.
	 */
	private final GenUserClass classRepresentation;
	
	/**
	 * Instantiates a new GenInterfaceWithClassImplClass.
	 * 
	 * @param name
	 *            The name of the interface and the java-file to generate.
	 * @param operations
	 *            The operations the interface and the class have.
	 * @param implement
	 *            The interfaces the interface implements.
	 * @param comment
	 *            The comment that describes this interface.
	 * @param packag
	 *            The package the interface will be generated in.
	 * @param nonGeneratedPart
	 *            Additional lines of code.
	 * @param classRepresentation
	 *            The "Impl"-class inside the interface that represents the AST-type.
	 */
	private GenInterfaceWithClassImplClass(final String name,
			final Collection<GenJavaOperation> operations,
			final Collection<GenInterfaceClass> implement,
			final GenPackage packag,
			final GenComment comment,
			final String nonGeneratedPart,
			final GenUserClass classRepresentation) {
		super(name, operations, implement, packag, comment, nonGeneratedPart);
		this.generateAsState = new GenerateAsClassState();
		this.classRepresentation = classRepresentation;
		classRepresentation.getAttributes().add(
				GenJavaAttribute.create(
						GenInterfaceWithClassImplClass.THIS,
						GenVisibility.PRIVATE,
						this,
						new Vector<GenAttributeModifier>()));
		classRepresentation.getAttributes().add(
				GenJavaAttribute.create(
						"$generatedObjects",
						GenVisibility.PRIVATE,
						GenMutableMap.create(GenStringType.getInstance(), GenAnyType.getInstance()),
						new Vector<GenAttributeModifier>()));
		classRepresentation.addOperation(GenJavaOperation.create(
				"$getThis",
				new Vector<GenParameter>(),
				GenFullParsedOperationState.create(
						GenComment.create("/**\n\t * Returns This.\n\t */"),
						new Vector<GenException>(),
						GenTypeReferenceByName.create(this.getFullyQualifiedTypeName()),
						new Vector<GenOperationModifier>(),
						GenVisibility.PRIVATE,
						"return this.This;")));
		
		classRepresentation.addOperation(GenJavaOperation.create(
				"get$generatedObjects",
				new Vector<GenParameter>(),
				GenFullParsedOperationState.create(
						GenComment.create("/**\n\t * Returns the $generatedObjects map.\n\t */"),
						new Vector<GenException>(),
						GenTypeReferenceByReference.create(GenMutableMap.create(
								GenStringType.getInstance(),
								GenAnyType.getInstance())),
						new Vector<GenOperationModifier>(),
						GenVisibility.PRIVATE,
						"return this.$generatedObjects;")));
		// TODO da FileWriter keine inneren Klassen generieren, soll die "Impl" Klasse derzeit extern als "name$Impl"
		// generiert werden (siehe FileWriterTask)
		// this.addInnerClass(this.classRepresentation);
		
		// TODO standardmäßig soll zunächst jede Userklasse mit extrahiertem Interface generiert werden, um potenzielle
		// Fehler oder weitere Folgen durch uneinheitliches Behandeln von UserKlassen zu vermeiden.
		this.setGenerateAsState(new GenerateAsInterfaceWithImplState());
	}
	
	/**
	 * Creates a GenInterfaceWithClassImplClass for a Type with the given parameters.
	 * 
	 * @param name
	 *            The name of the interface and the java-file to generate.
	 * @param operations
	 *            The operations the interface and the class have.
	 * @param implement
	 *            The interfaces the interface implements.
	 * @param comment
	 *            The comment that describes this interface.
	 * @param packag
	 *            The package the interface will be generated in.
	 * @param nonGeneratedPart
	 *            Additional lines of code.
	 * @param classRepresentation
	 *            The "Impl"-class inside the interface that represents the AST-type.
	 * @return Returns a new GenInterfaceWithClassImplClass with the given parameters.
	 */
	public static GenInterfaceWithClassImplClass create(final String name,
			final Collection<GenJavaOperation> operations,
			final Collection<GenInterfaceClass> implement,
			final GenPackage packag,
			final GenComment comment,
			final String nonGeneratedPart,
			final GenUserClass classRepresentation) {
		return new GenInterfaceWithClassImplClass(name, operations, implement, packag, comment, nonGeneratedPart,
				classRepresentation);
	}
	
	/**
	 * Returns the "Impl"-class inside the interface that represents the AST-type.
	 * 
	 * @return GenUserClass
	 */
	public GenUserClass getClassRepresentation() {
		return this.classRepresentation;
	}
	
	/**
	 * This state decides whether this should be generated as interface containing an "Impl"-class or only as class.
	 * 
	 * @return GenerateAsState
	 */
	public GenerateAsState getGenerateAsState() {
		return this.generateAsState;
	}
	
	/**
	 * Sets the generateAsState to the given state. Un- or resetting the state to "GenerateAsInterface..." is not
	 * possible once it has been changed from "GenerateAsClassState".
	 * 
	 * @param state
	 *            The state decides whether this should be generated as interface containing an "Impl"-class or only as
	 *            class.
	 */
	private void setGenerateAsState(final GenerateAsState state) {
		this.generateAsState.accept(new GenerateAsStateVisitor() {
			
			@Override
			public void handle(final GenerateAsClassState asClassState) {
				GenInterfaceWithClassImplClass.this.generateAsState = state;
				// convert class to private inner class schematic
				// TODO da FileWriter keine inneren Klassen generieren, soll die "Impl" Klasse derzeit extern als
				// "name$Impl"
				// generiert werden (siehe FileWriterTask)
				GenInterfaceWithClassImplClass.this.getClassRepresentation().setName(
						GenInterfaceWithClassImplClass.this.getName() + "$" + CLASS_NAME);
				// TODO da FileWriter keine inneren Klassen generieren, soll die "Impl" Klasse derzeit extern als
				// "name$Impl"
				// generiert werden (siehe FileWriterTask). Des Weiteren kann nicht eine Datei und ein Ordner mit dem
				// gleichen Namen erzeugt werden, daher muss sich das Package der "inneren" Klasse vom Namen des
				// Interfaces unterscheiden
				final GenPackage newClassPackag =
						GenInterfaceWithClassImplClass.this
								.getPackag()
								.copy()
								.addPackage(
										GenUnqualifiedPackage.create(GenInterfaceWithClassImplClass.this.getName()
												+ "_InnerPackage"));
				GenInterfaceWithClassImplClass.this.getClassRepresentation().setPackag(newClassPackag);
				
				// class implements his interface-representation
				GenInterfaceWithClassImplClass.this.getClassRepresentation().getImplement()
						.add(GenInterfaceWithClassImplClass.this);
				// make sure class has interface operations
				for (final GenJavaOperation op : GenInterfaceWithClassImplClass.this.getOperations()) {
					GenInterfaceWithClassImplClass.this.classRepresentation.addOperation(op);
				}
				
			}
			
			@Override
			public void handle(final GenerateAsInterfaceWithImplState asInterfaceState) {
				// Do not set state of GenInterfaceWithClassImplClass if it is already GenerateAsInterfaceWithImplState!
			}
		});
	}
	
	@Override
	public void accept(final GenClassVisitor visitor) {
		visitor.handle(this);
	}
	
	@Override
	public <X> X accept(final GenClassVisitorReturn<X> visitor) {
		return visitor.handle(this);
	}
	
	@Override
	public <X extends Exception> void accept(final GenClassVisitorException<X> visitor) throws X {
		visitor.handle(this);
	}
	
	@Override
	public void accept(final GenInterfaceClassVisitor visitor) {
		visitor.handle(this);
	}
	
	@Override
	public <X> X accept(final GenInterfaceClassVisitorReturn<X> visitor) {
		return visitor.handle(this);
	}
	
	@Override
	public <Y extends Exception> void accept(final GenInterfaceClassVisitorException<Y> visitor) throws Y {
		visitor.handle(this);
	}
	
}
