package de.fhdw.wtf.facade;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.fhdw.wtf.common.ast.Name;
import de.fhdw.wtf.common.ast.UnqualifiedName;
import de.fhdw.wtf.common.ast.type.AtomicType;
import de.fhdw.wtf.common.ast.type.CompositeType;
import de.fhdw.wtf.common.ast.type.ListType;
import de.fhdw.wtf.common.ast.type.MapType;
import de.fhdw.wtf.common.ast.type.ProductElementType;
import de.fhdw.wtf.common.ast.type.ProductType;
import de.fhdw.wtf.common.ast.type.SumType;
import de.fhdw.wtf.common.ast.type.ThrownType;
import de.fhdw.wtf.common.ast.type.Type;
import de.fhdw.wtf.common.ast.type.TypeProxy;
import de.fhdw.wtf.common.ast.visitor.CompositeTypeVisitorReturn;
import de.fhdw.wtf.common.ast.visitor.CompositeTypeVisitorReturnException;
import de.fhdw.wtf.common.ast.visitor.TypeVisitorReturn;
import de.fhdw.wtf.common.ast.visitor.TypeVisitorReturnException;
import de.fhdw.wtf.common.token.DummyToken;
import de.fhdw.wtf.common.token.IdentifierToken;
import de.fhdw.wtf.walker.walker.HelperUtils;

/**
 * The TypeNameGenerator provides a way to get class/interface-names for (even anonymous) Ast-Types.
 */
public final class TypeNameGenerator {
	
	/**
	 * The prefix for the name of SumTypes.
	 */
	public static final String SUM_BEGIN = "Sum$0";
	/**
	 * The suffix for the name of SumTypes.
	 */
	public static final String SUM_END = "$0$";
	/**
	 * The symbol for the name of SumTypes to seperate contained Types.
	 */
	public static final String SUM_SEPERATE_ELEMENT = "€_";
	
	/**
	 * The prefix for the name of ProductTypes.
	 */
	public static final String PRODUCT_BEGIN = "Product$0";
	/**
	 * The suffix for the name of ProductTypes.
	 */
	public static final String PRODUCT_END = "$0$";
	/**
	 * The symbol for the name of ProductTypes to seperate contained Attributes.
	 */
	public static final String PRODUCT_SEPERATE_ATTRIBUTES = "€_";
	/**
	 * The symbol for the name of ProductTypes to seperate the Type and the Name of contained Attributes.
	 */
	public static final String PRODUCT_SEPERATE_TYPE_AND_NAME = "€0";
	
	/**
	 * The prefix for the name of ListTypes.
	 */
	public static final String LIST_BEGIN = "List$_$";
	/**
	 * The suffix for the name of ListTypes.
	 */
	public static final String LIST_END = "$_$";
	
	/**
	 * the suffix for the name of ThrownTypes.
	 */
	public static final String THROWN_BEGIN = "Thrown$0";
	
	/**
	 * the suffix for the name of ThrownTypes.
	 */
	public static final String THROWN_END = "$!$";
	
	/**
	 * The prefix for the name of MapTypes.
	 */
	public static final String MAP_BEGIN = "Map$_$";
	/**
	 * The suffix for the name of MapTypes.
	 */
	public static final String MAP_END = "$_$";
	/**
	 * The symbol for the name of MapTypes to seperate the key-Type and the value-Type.
	 */
	public static final String MAP_SEPERATE_KEY_AND_VALUE = "$";
	
	/**
	 * The one instance of TypeNameGenerator.
	 */
	private static TypeNameGenerator instance;
	
	/**
	 * Returns the one instance of a TypeNameGenerator.
	 * 
	 * @return TypeNameGenerator
	 */
	public static synchronized TypeNameGenerator getInstance() {
		if (instance == null) {
			instance = new TypeNameGenerator();
		}
		return instance;
	}
	
	// TODO TypeNameGenerator richtig bauen nach entsprechenden richtigen
	// Vorschlag!
	
	/**
	 * Creates a instance of TypeNameGenerator.
	 */
	private TypeNameGenerator() {
		this.userDefinedTypeNames = new HashMap<>();
	}
	
	/**
	 * Defines the maximal allowed length of generated Names.
	 */
	private static final Integer MAX_NAME_LENGTH = 200;
	// TODO maximale Länge? 50?
	// ~255 ist die maximal erlaubt Pfadlänge in Windows... besser wäre eine
	// intelligente ermittlung auf grundlage des Packages und des Namens und der
	// Root-Pfades
	
	/**
	 * A Map that contains generated Names as key and a userdefined Name as value to map long names on shorter ones.
	 */
	private final Map<Name, Name> userDefinedTypeNames;
	
	/**
	 * Defines a new TypeName. This name will be used instead of the generated name. Use this method to solve the
	 * problem that some generated names exceed the maximal allowed length.
	 * 
	 * @param generateTypeName
	 *            The generated Name to substitute.
	 * @param userDefinedTypeName
	 *            The userdefined Name to use instead.
	 */
	public void defineTypeName(final Name generateTypeName, final Name userDefinedTypeName) {
		// TODO was ist mit benutzer definierten Namen, die dann auch generiert
		// eventuell irgendwo auftauchen!? Siehe Unten! => Am besten im Core
		// lösen.
		this.userDefinedTypeNames.put(generateTypeName, userDefinedTypeName);
	}
	
	/**
	 * Returns the maximal allowed name length.
	 * 
	 * @return Integer
	 */
	public Integer getMaxNameLength() {
		return TypeNameGenerator.MAX_NAME_LENGTH;
	}
	
	/**
	 * Return the typeName of the type. Throws an exception when the generated Name is too long, because some Operating
	 * Systems have problems with long filenames.
	 * 
	 * @param type
	 *            The Ast-Type that a Name shall be provided.
	 * @return Name The generated Name for type.
	 * @throws GeneratedNameTooLongException
	 *             Thrown when the generated Name exceeds the maximal allowed Name length.
	 */
	public UnqualifiedName getTypeName(final Type type) throws GeneratedNameTooLongException {
		final UnqualifiedName name = type.accept(new GetTypeNameTypeVisitorReturnException());
		
		// TODO 1. Gucken ob der generierte Name irgendwo als user definierter
		// auftaucht dann neuen Fehler werfen
		// (USerDefinedNameIsQualToGenratedNameException).
		// 2. Gucken, ob es einen user definierten Namen gibt.
		
		if (name.toString().length() > TypeNameGenerator.MAX_NAME_LENGTH) {
			throw new GeneratedNameTooLongException(name);
		}
		
		return name;
	}
	
	/**
	 * A visitor to provides Names for different Ast-Types.
	 */
	private final class GetTypeNameTypeVisitorReturnException implements
			TypeVisitorReturnException<UnqualifiedName, GeneratedNameTooLongException> {
		
		@Override
		public UnqualifiedName handle(final AtomicType s) throws GeneratedNameTooLongException {
			return s.getTypeName().getLastAddedName();
		}
		
		@Override
		public UnqualifiedName handle(final CompositeType c) throws GeneratedNameTooLongException {
			return c.accept(new GetTypeNameCompositeTypeVisitorReturnException());
		}
		
		/**
		 * private class to handle CompositeTypes for GetTypeNameTypeVisitorReturnException.
		 */
		private final class GetTypeNameCompositeTypeVisitorReturnException implements
				CompositeTypeVisitorReturnException<UnqualifiedName, GeneratedNameTooLongException> {
			@Override
			public UnqualifiedName handle(final ListType list) throws GeneratedNameTooLongException {
				final String name = LIST_BEGIN + TypeNameGenerator.this.getTypeName(list.getOf()).toString() + LIST_END;
				return TypeNameGenerator.this.createName(name);
			}
			
			@Override
			public UnqualifiedName handle(final MapType map) throws GeneratedNameTooLongException {
				final String name =
						MAP_BEGIN + TypeNameGenerator.this.getTypeName(map.getKey()).toString()
								+ MAP_SEPERATE_KEY_AND_VALUE
								+ TypeNameGenerator.this.getTypeName(map.getOf()).toString() + MAP_END;
				return TypeNameGenerator.this.createName(name);
			}
			
			@Override
			public UnqualifiedName handle(final ProductType product) throws GeneratedNameTooLongException {
				final StringBuilder name = new StringBuilder();
				name.append(PRODUCT_BEGIN);
				final List<ProductElementType> elements = product.getElements();
				for (final ProductElementType productElement : elements) {
					name.append(productElement.getName());
					name.append(PRODUCT_SEPERATE_TYPE_AND_NAME);
					name.append(TypeNameGenerator.this.getTypeName(productElement.getType()).toString());
					
					if (elements.indexOf(productElement) < elements.size() - 1) {
						name.append(PRODUCT_SEPERATE_ATTRIBUTES);
					}
				}
				name.append(PRODUCT_END);
				return TypeNameGenerator.this.createName(name.toString());
			}
			
			@Override
			public UnqualifiedName handle(final SumType sum) throws GeneratedNameTooLongException {
				final StringBuilder name = new StringBuilder();
				name.append(SUM_BEGIN);
				final List<Type> elements = sum.getElements();
				for (final Type type : elements) {
					if (type.accept(new TypeVisitorReturn<Boolean>() {
						
						@Override
						public Boolean handle(final AtomicType atomicType) {
							return true;
						}
						
						@Override
						public Boolean handle(final CompositeType compositeType) {
							return compositeType.accept(new CompositeTypeVisitorReturn<Boolean>() {
								
								@Override
								public Boolean handle(final ListType list) {
									return true;
								}
								
								@Override
								public Boolean handle(final MapType map) {
									return true;
								}
								
								@Override
								public Boolean handle(final ProductType product) {
									return true;
								}
								
								@Override
								public Boolean handle(final SumType sum) {
									return true;
								}
								
								@Override
								public Boolean handle(final ThrownType thrownType) {
									return false;
								}
							});
						}
						
						@Override
						public Boolean handle(final TypeProxy typeProxy) {
							return true;
						}
					})) {
						
						if (!name.toString().equals(SUM_BEGIN)) {
							name.append(SUM_SEPERATE_ELEMENT);
						}
						
						name.append(TypeNameGenerator.this.getTypeName(type).toString());
					}
				}
				name.append(SUM_END);
				return TypeNameGenerator.this.createName(name.toString());
			}
			
			@Override
			public UnqualifiedName handle(final ThrownType thrownType) throws GeneratedNameTooLongException {
				final StringBuilder name = new StringBuilder();
				name.append(THROWN_BEGIN);
				name.append(TypeNameGenerator.this.getTypeName(thrownType.getUnderlyingType()).toString());
				name.append(THROWN_END);
				return TypeNameGenerator.this.createName(name.toString());
			}
		}
		
		@Override
		public UnqualifiedName handle(final TypeProxy typeProxy) throws GeneratedNameTooLongException {
			return TypeNameGenerator.this.getTypeName(HelperUtils.getTargetAtomicType(typeProxy));
		}
	}
	
	/**
	 * Converts a String to an UnqualifiedName.
	 * 
	 * @param name
	 *            The value that the UnqualifiedName shall represent.
	 * @return UnqualifiedName
	 */
	public UnqualifiedName createName(final String name) {
		return UnqualifiedName.create(IdentifierToken.create(name, DummyToken.getDummyPosition()));
	}
}
