package de.fhdw.wtf.generator.transformer.clipper;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import de.fhdw.wtf.common.exception.walker.CyclicDependencyException;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.file.FileUtils;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenComment;
import de.fhdw.wtf.generator.java.generatorModel.GenExternalInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceWithClassImplClass;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenOperationSignature;
import de.fhdw.wtf.generator.java.generatorModel.GenParameter;
import de.fhdw.wtf.generator.java.generatorModel.GenPrimitiveClass;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenTypeReferenceByName;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorException;
import de.fhdw.wtf.generator.java.walker.SimpleGeneratorModelWalkerTask;
import de.fhdw.wtf.generator.transformer.util.Tuple;

/**
 * This task imports all existing methods into all classes in the JavaST It overwrites existing operations.
 * 
 */
public class LinkToGenClassTask extends SimpleGeneratorModelWalkerTask {
	
	/**
	 * Constant for "\n".
	 */
	private static final String BREAK = "\n";
	
	private final ClipperConfiguration configuration;
	
	/**
	 * @param taskmanager
	 * @param javaGeneratorModel
	 * @param configuration
	 * @param clipperImport
	 */
	public LinkToGenClassTask(final TaskExecutor taskmanager,
			final GeneratorModel javaGeneratorModel,
			final ClipperConfiguration configuration,
			final ClipToFileTask clipperImport) {
		super(taskmanager, javaGeneratorModel);
		this.configuration = configuration;
		
		try {
			this.addDependency(clipperImport);
		} catch (final CyclicDependencyException e) {
			// should not happen
			e.printStackTrace();
			throw new Error("Dependency tasks are cyclic in ImportAllFromClipperTask.");
		}
	}
	
	/**
	 * Inserts all methods from clipper into the given type.
	 * 
	 * @param t
	 * @throws TaskException
	 */
	private void handleType(final de.fhdw.wtf.generator.java.generatorModel.GenClass t) throws TaskException {
		// TODO: Wenn die InnerClasses so wie geplant generiert werden, kann hier die Struktur wieder vereinfacht
		// werden. ACHTUNG: Auch an die Testfälle denken!
		final String filename =
				LinkToGenClassTask.this.configuration.getRepoDirectoryRoot()
						+ t.getFullyQualifiedTypeName().replace(".", "/");
		final File[] directoryListing = new File(filename).listFiles();
		if (directoryListing != null) { // is a directory
			t.accept(new GenClassVisitorException<TaskException>() {
				
				@Override
				public void handle(final GenClassClass classClass) throws TaskException {
					LinkToGenClassTask.this.readClippedFile(directoryListing, filename, t);
				}
				
				@Override
				public void handle(final GenInterfaceClass interfaceClass) throws TaskException {
					interfaceClass.accept(new GenInterfaceClassVisitorException<TaskException>() {
						
						@Override
						public void handle(final GenInterfaceWithClassImplClass interfaceWithImplClass)
								throws TaskException {
							
							LinkToGenClassTask.this.readClippedFile(directoryListing, filename, t);
							
							LinkToGenClassTask.this.handleType(interfaceWithImplClass.getClassRepresentation());
						}
						
						@Override
						public void handle(final GenSimpleInterfaceClass simpleInterface) throws TaskException {
							LinkToGenClassTask.this.readClippedFile(directoryListing, filename, t);
						}
						
						@Override
						public void handle(final GenExternalInterfaceClass iface) throws TaskException {
							throw new TaskException("Cannot restore contents of external interface "
									+ iface.getFullyQualifiedTypeName());
						}
					});
					
				}
				
				@Override
				public void handle(final GenPrimitiveClass primitiveClass) throws TaskException {
					LinkToGenClassTask.this.readClippedFile(directoryListing, filename, t);
				}
			});
		} else {
			return;
		}
	}
	
	/**
	 * Read all files in clipper directory and sets the contents of the files to the GenClasses fields.
	 * 
	 * @param directoryListing
	 *            files in clipper directory
	 * @param filename
	 *            clipper directory path
	 * @param t
	 *            GenClass to be generated
	 * @throws TaskException
	 *             possible exception
	 */
	private void readClippedFile(final File[] directoryListing,
			final String filename,
			final de.fhdw.wtf.generator.java.generatorModel.GenClass t) throws TaskException {
		for (final File file : directoryListing) {
			try {
				if (file.isDirectory()) {
					final Iterator<GenClassClass> innerClasses = t.getInnerClasses().iterator();
					while (innerClasses.hasNext()) {
						final GenClassClass current = innerClasses.next();
						if (current.getName().equals(file.getName())) {
							this.handleType(current);
							break;
						}
					}
				} else {
					final String fileString = FileUtils.getFileString(file.getAbsolutePath());
					final String fileName = file.getName();
					switch (fileName) {
					case ClipperUtils.CLASSCOMMENT_FILENAME:
						t.setComment(GenComment.create(fileString));
						break;
					case ClipperUtils.PROTECTED_AREA_FILENAME:
						t.setNonGeneratedPart(fileString);
						break;
					case ClipperUtils.ERRORFILE_FILENAME:
						break;
					case ClipperUtils.IMPORTS_FILENAME:
						t.setImports(BREAK + fileString + BREAK);
						break;
					default:
						if (fileName.startsWith("#attr#")) {
							// final GenAttribute a =
							// this.createNewAttribute(fileName, fileString);
							
							// TODO Attributes are not yet supported in Java-AST
						} else if (fileName.startsWith("#constructor")) {
							// Constructors will be no longer clipped
							final GenJavaOperation constructor =
									GenJavaOperation.create(t.getName(), this.parseClipperFileName(fileName, true)
											.getB(), GenSimpleOperationState.create(fileString));
							t.accept(new GenClassVisitorException<TaskException>() {
								@Override
								public void handle(final GenClassClass classClass) throws TaskException {
									LinkToGenClassTask.this.deleteConstructorLike(
											classClass.getConstructors(),
											constructor);
									classClass.getConstructors().add(constructor);
								}
								
								@Override
								public void handle(final GenInterfaceClass interfaceClass) throws TaskException {
									throw new TaskException("Clipper: Tried to add constructor to interface" + fileName);
								}
								
								@Override
								public void handle(final GenPrimitiveClass primitiveClass) throws TaskException {
									// nothing to do
								}
							});
						} else {
							final GenJavaOperation m = this.createNewMethod(fileName, fileString);
							if (!t.overrideExistingOperation(m)) {
								t.tryToOverrideAbstractOperation(m);
							}
						}
					}
				}
			} catch (final IOException e) {
				throw new TaskException("Clipper: Retrieve method failed: " + filename);
			}
		}
	}
	
	/**
	 * Creates the new method and deletes possible existing operations.
	 * 
	 * @param fileName
	 *            - the name of the clipped file (it contains the signature of the operation).
	 * @param fileString
	 *            - the clipped filestring.
	 * 
	 * @return the clipped {@link GenJavaOperation}
	 */
	private GenJavaOperation createNewMethod(final String fileName, final String fileString) {
		final Tuple<String, List<GenParameter>> res = this.parseClipperFileName(fileName, false);
		final GenJavaOperation newOperation =
				GenJavaOperation.create(res.getA(), res.getB(), GenSimpleOperationState.create(fileString));
		return newOperation;
	}
	
	private void deleteConstructorLike(final Collection<GenJavaOperation> collection, final GenJavaOperation constructor) {
		final Iterator<GenJavaOperation> i = collection.iterator();
		while (i.hasNext()) {
			final GenJavaOperation current = i.next();
			if (GenOperationSignature.equalsParamTypes(current.getParameters(), constructor.getParameters())) {
				i.remove();
			}
		}
	}
	
	private Tuple<String, List<GenParameter>> parseClipperFileName(final String fileName, final boolean constructor) {
		final List<String> split = new ArrayList<>(Arrays.asList(fileName.split(ClipperUtils.SEPERATOR)));
		if (constructor) { // Beim Konstruktor-Dateinamen ist der erste Split
							// leer und wird übersprungen!
			split.remove(0);
		}
		final String name = split.remove(0);
		final List<GenParameter> result = new ArrayList<>();
		for (final String current : split) {
			final GenTypeReferenceByName ref = GenTypeReferenceByName.create(current);
			final GenParameter para = GenParameter.create("DUMMY-PARA-NAME", ref); // TODO
																					// Name
																					// anpassen
			result.add(para);
		}
		return new Tuple<>(name, result);
	}
	
	@Override
	public void handleClassClass(final GenClassClass cc) throws TaskException {
		this.handleType(cc);
	}
	
	@Override
	public void handleInterfaceClass(final GenInterfaceClass ic) throws TaskException {
		this.handleType(ic);
	}
	
	@Override
	public void finalizeTask() throws TaskException {
		// nothing to do
	}
	
	@Override
	public void handleOperation(final GenOperation o) throws TaskException {
		// nothing to do
	}
	
}
