package de.fhdw.wtf.generator.transformer.clipper.internal;

import java.util.ArrayList;
import java.util.List;

import de.fhdw.wtf.generator.transformer.clipper.ClipperUtils;
import de.fhdw.wtf.generator.transformer.exception.ClipperImportFormatException;

/**
 * 
 * Diese Klasse analysiert eine Java-Source-Datei und schneidet die darin enthaltenen inneren Klassen/Interfaces heraus.
 * 
 * @author hfw413hy
 *
 */
public class ClipperJavaFileShadowCopy {
	
	/**
	 * Enthält den Inhalt einer Klasse/Interface zwischen den geschweiften Klammern.
	 */
	protected String content = "";
	
	/**
	 * Enthält den kompletten Text einer Klasse oder eines Interfaces.
	 */
	protected String fullClass = "";
	
	/**
	 * Enthält den Typ ("class", "interface").
	 */
	protected String type = "";
	
	/**
	 * Enthält den zugewiesenen Namen (Person, Person$Impl, etc.).
	 */
	protected String name = "";
	
	/**
	 * Enthält die Klassen/Interfaces die innerhalb einer Klasse/eines Interfaces definiert wurden.
	 */
	protected List<ClipperJavaFileShadowCopy> innerClasses = new ArrayList<>();
	
	/**
	 * Erzeugt eine bestimmte Anzahl an Leerzeichen.
	 * 
	 * @param count
	 *            Anzahl der zu erzeugenden Leerzeichen.
	 * @return String mit <count> Leerzeichen
	 */
	private String getWhitespaces(final int count) {
		return new String(new char[count]).replace('\0', ' ');
	}
	
	/**
	 * Liefert den Typ zurück.
	 * 
	 * @return Der Typ
	 */
	public String getType() {
		return this.type;
	}
	
	/**
	 * Setzt den Typ.
	 * 
	 * @param type
	 */
	public void setType(final String type) {
		this.type = type;
	}
	
	/**
	 * Liefert den Namen zurück.
	 * 
	 * @return Der Name
	 */
	public String getName() {
		return this.name;
	}
	
	/**
	 * Setzt den Namen.
	 * 
	 * @param name
	 */
	public void setName(final String name) {
		this.name = name;
	}
	
	/**
	 * Liefert den Inhalt einer Klasse/Eines Interfaces.
	 * 
	 * @return Der Inhalt zwischen den geschweiften Klammern
	 */
	public String getContent() {
		return this.content;
	}
	
	/**
	 * Liefert eine komplette Klasse, nur ohne ihre "inneren Klassen".
	 * 
	 * @return Die komplette Klasse
	 */
	public String getFullClass() {
		return this.fullClass;
	}
	
	/**
	 * Eine Liste an Klassen/Interfaces die in dieser Klasse/diesem Interface definiert wurden.
	 * 
	 * @return Die Liste an inneren Klassen
	 */
	public List<ClipperJavaFileShadowCopy> getInnerClasses() {
		return this.innerClasses;
	}
	
	/**
	 * Starts the analysis of inner classes.
	 * 
	 * @param baseClassContent
	 * @throws ClipperImportFormatException
	 */
	public void analyze(final String baseClassContent) throws ClipperImportFormatException {
		final String tmpCopy = ClipperUtils.simplifyJavaCode(baseClassContent);
		
		if (baseClassContent.indexOf("class ") > -1 || baseClassContent.indexOf("interface ") > -1) {
			// Diesen Teil nur ausführen, wenn wir "class" oder "interface" gefunden haben!
			this.parseType(tmpCopy);
			this.parseName(tmpCopy);
			
			final int idxOpen = tmpCopy.indexOf('{');
			final int idxClose = ClipperUtils.findCorrespondingClosingCurlyBracket(idxOpen, tmpCopy);
			
			final String strContentOfThisClass = baseClassContent.substring(idxOpen + 1, idxClose);
			
			this.content = this.parseInnerClasses(strContentOfThisClass);
			this.fullClass = baseClassContent.substring(0, idxOpen + 1) + "\n" + this.content + "}";
		} else {
			// Diesen Teil ausführen, wenn beispielsweise nur Kommentare in der Datei enthalten sind.
			// Sonst kann es dazu kommen, dass in "parseType" oder "parseName" eine Exception geworfen wird!
			this.fullClass = baseClassContent;
		}
	}
	
	/**
	 * 
	 * @param baseClassContent
	 * @return Returns the direct content of this class/interface and parses the containing classes/interfaces
	 * @throws ClipperImportFormatException
	 */
	private String parseInnerClasses(final String baseClassContent) throws ClipperImportFormatException {
		String result = baseClassContent;
		String resultShadow = ClipperUtils.simplifyJavaCode(baseClassContent);
		int idxOpen = resultShadow.indexOf('{');
		while (idxOpen > -1) {
			final int idxClose = ClipperUtils.findCorrespondingClosingCurlyBracket(idxOpen, resultShadow);
			final int idxNL = resultShadow.substring(0, idxOpen).lastIndexOf('\n');
			
			final String subClass = result.substring(idxNL + 1, idxClose + 1);
			result = result.substring(0, idxNL) // + this.getWhitespaces(subClass.length())
					+ result.substring(idxClose + 1);
			resultShadow = resultShadow.substring(0, idxNL) // + this.getWhitespaces(subClass.length())
					+ resultShadow.substring(idxClose + 1);
			idxOpen = resultShadow.indexOf('{');
			
			final ClipperJavaFileShadowCopy inner = new ClipperJavaFileShadowCopy();
			inner.analyze(subClass);
			this.getInnerClasses().add(inner);
		}
		
		return result;
	}
	
	/**
	 * Parses the name of the first class/interface in the given shadowFile.
	 * 
	 * @param shadowFile
	 * @throws ClipperImportFormatException
	 */
	private void parseName(final String shadowFile) throws ClipperImportFormatException {
		final int idxOpen = shadowFile.indexOf('{');
		String subStr =
				shadowFile.substring(shadowFile.indexOf(this.getType()) + this.getType().length(), idxOpen).trim();
		if (subStr.contains(" ")) {
			subStr = subStr.split(" ")[0];
		}
		this.setName(subStr);
	}
	
	/**
	 * Parses the type (if it is class or interface) of the first class/interface in the given shadowFile.
	 * 
	 * @param shadowFile
	 * @throws ClipperImportFormatException
	 */
	private void parseType(final String shadowFile) throws ClipperImportFormatException {
		final int idxOpen = shadowFile.indexOf('{');
		final int idxInterface = shadowFile.substring(0, idxOpen).indexOf(" interface ");
		final int idxClass = shadowFile.substring(0, idxOpen).indexOf(" class ");
		
		if (idxInterface > -1 && idxClass > -1) {
			if (idxInterface > idxClass) {
				this.setType("class");
			} else {
				this.setType("interface");
			}
		} else if (idxInterface > -1 && idxClass == -1) {
			this.setType("interface");
		} else if (idxClass > -1 && idxInterface == -1) {
			this.setType("class");
		} else if (idxClass == -1 && idxInterface == -1) {
			throw new ClipperImportFormatException("Bezeichnung 'class' und 'interface' nicht gefunden!");
		}
	}
	
}
