package de.fhdw.wtf.generator.database.generation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import de.fhdw.wtf.generator.database.exception.GenerationException;
import de.fhdw.wtf.generator.database.exception.NotSameAttributeException;
import de.fhdw.wtf.persistence.exception.NoSubtypeException;
import de.fhdw.wtf.persistence.exception.NotAbstractException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.facade.ClassFacade;
import de.fhdw.wtf.persistence.facade.ObjectFacade;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.persistence.meta.UserType;

public class RefactoringsGenerator {
	
	private final ClassFacade classFacade;
	private final ObjectFacade objectFacade;
	
	public RefactoringsGenerator(final ClassFacade classFacade, final ObjectFacade objectFacade) {
		this.classFacade = classFacade;
		this.objectFacade = objectFacade;
		
	}
	
	/**
	 * Moves the given association to all subtypes of its current owner.
	 * 
	 * @param association
	 *            the association that should be moved.
	 * @param subtypes
	 *            the sub types
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 * @throws NoSubtypeException
	 *             Without subtypes push down is not possible.
	 * @throws NotSameAttributeException
	 *             if given types are no subtyps of owner
	 * @throws NotAbstractException
	 *             thrown when the owner of asso is abstract
	 */
	public void pushDown(final UnidirectionalAssociation association, final Collection<UserType> subtypes)
			throws PersistenceException, NoSubtypeException, NotAbstractException, NotSameAttributeException {
		this.checkAreSubTypesOf(subtypes, association.getOwner());
		if (!association.getOwner().isAbs()) {
			throw new NotAbstractException();
		}
		final Collection<Long> newAssociations = this.createUnidirectionalAssociationsForAllSubtypes(subtypes, association);
		this.classFacade.updateLinksToNewAssociation(association.getId(), newAssociations);
		this.classFacade.deleteAssociation(association.getId());
	}
	
	/**
	 * checks if superType is a super type of the collection subtypes. When not a Exception is thrown.
	 * 
	 * @param subtypes
	 *            the collection of subtypes
	 * @param superType
	 *            the super type
	 * @throws NotSameAttributeException
	 *             thrown when a subtype is not a subtype of superType
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	private void checkAreSubTypesOf(final Collection<UserType> subtypes, final UserType superType)
			throws NotSameAttributeException, PersistenceException {
		final Iterator<UserType> i = subtypes.iterator();
		while (i.hasNext()) {
			final UserType current = i.next();
			this.checkIsSuperTypeOf(current, superType);
		}
		
	}
	
	/**
	 * creates unidirectional associations for all subtypes.
	 * 
	 * @param subtypes
	 *            the collection of subtypes.
	 * @param template
	 *            a unidirectional Association used as a template for the new associations
	 * @return a collection of the ids of the created associations
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	private Collection<Long> createUnidirectionalAssociationsForAllSubtypes(final Collection<UserType> subtypes,
			final UnidirectionalAssociation template) throws PersistenceException {
		if (subtypes.isEmpty()) {
			throw new NoSubtypeException();
		}
		final Collection<Long> result = new ArrayList<>();
		final Iterator<UserType> i = subtypes.iterator();
		while (i.hasNext()) {
			final UserType current = i.next();
			final UnidirectionalAssociation create =
					this.classFacade.createUnidirectionalAssociation(
							template.getName(),
							template.isEssential(),
							template.isUnique(),
							current,
							template.getTarget());
			result.add(create.getId());
		}
		return result;
	}
	
	/**
	 * Replaces current association target with a more general target type.
	 * 
	 * @param asso
	 *            the association to expand.
	 * @param newType
	 *            the type to expand the association.
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 * @throws NotSameAttributeException
	 *             throws when newType is not a super type of the old target.
	 */
	public void expandUnidirectionalAssociationTarget(final UnidirectionalAssociation asso, final UserType newType)
			throws PersistenceException, NotSameAttributeException {
		this.checkIsSuperTypeOf(asso.getTarget(), newType);
		final UnidirectionalAssociation newAsso =
				this.classFacade.createUnidirectionalAssociation(
						asso.getName(),
						asso.isEssential(),
						asso.isUnique(),
						asso.getOwner(),
						newType);
		final Collection<Long> newAssociations = new ArrayList<>();
		newAssociations.add(newAsso.getId());
		this.classFacade.updateLinksToNewAssociation(asso.getId(), newAssociations);
		this.classFacade.deleteAssociation(asso.getId());
	}
	
	/**
	 * Moves the given associations to the supertype of its current owner. All Associations must have different owners.
	 * All Associations must have same essential, unique flag, target and name. All owners must be subtypes of
	 * 'supertype'.
	 * 
	 * @param asso
	 *            the association that should be moved
	 * @param supertype
	 *            the super type
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 * @throws GenerationException
	 *             if a prerequisite is violated
	 * @throws NoSubtypeException
	 *             Without subtypes push down is not possible.
	 * @throws NotSameAttributeException
	 *             thrown when the owner of the assocation is not a subtype of supertype
	 */
	public void pullUp(final Collection<UnidirectionalAssociation> asso, final UserType supertype)
			throws PersistenceException, NotSameAttributeException, GenerationException {
		if (asso.isEmpty()) {
			throw new GenerationException("You must specify associations!") {
				private static final long serialVersionUID = 1L;
			};
		}
		this.checkIsSuperTypeOfOwners(asso, supertype);
		final boolean essential = this.checkEssential(asso);
		final boolean unique = this.checkUnique(asso);
		final Type target = this.checkSameTarget(asso);
		final String name = this.checkName(asso);
		
		final UnidirectionalAssociation newAsso =
				this.classFacade.createUnidirectionalAssociation(name, essential, unique, supertype, target);
		final Collection<Long> newAssociation = new ArrayList<>();
		newAssociation.add(newAsso.getId());
		final Iterator<UnidirectionalAssociation> i = asso.iterator();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			this.classFacade.updateLinksToNewAssociation(current.getId(), newAssociation);
			this.classFacade.deleteAssociation(current.getId());
		}
	}
	
	/**
	 * check if all the associations of the collection asso are sub types of super type. Throws Exception when not.
	 * 
	 * @param asso
	 *            collection of the association which should checked.
	 * @param supertype
	 *            the super type to check
	 * @throws NotSameAttributeException
	 *             throws when min. 1 association is not a subtype of supertype
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	private void checkIsSuperTypeOfOwners(final Collection<UnidirectionalAssociation> asso, final UserType supertype)
			throws NotSameAttributeException, PersistenceException {
		final Iterator<UnidirectionalAssociation> i = asso.iterator();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			this.checkIsSuperTypeOf(current.getOwner(), supertype);
		}
	}
	
	/**
	 * Check if subtype is a sub type of supertype. Throws NotSameAttributeException when not.
	 * 
	 * @param subtype
	 *            the sub type to check.
	 * @param superType
	 *            the super type to check.
	 * @throws NotSameAttributeException
	 *             thrown when subtype is not a sub type of super type
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	private void checkIsSuperTypeOf(final Type subtype, final UserType superType) throws NotSameAttributeException,
			PersistenceException {
		if (!this.classFacade.isSuperClassTo(superType, subtype)) {
			throw new NotSameAttributeException("supertype");
		}
	}
	
	/**
	 * checks if all the associations of the collection asso have the same target. Throws NotSameAttributeException when
	 * not.
	 * 
	 * @param asso
	 *            the collection of the associations to check.
	 * @return the target of all associations
	 * @throws NotSameAttributeException
	 *             thrown when min. 1 association has a other target.
	 */
	private Type checkSameTarget(final Collection<UnidirectionalAssociation> asso) throws NotSameAttributeException {
		final Iterator<UnidirectionalAssociation> i = asso.iterator();
		final Type result = i.next().getTarget();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			if (!current.getTarget().equals(result)) {
				throw new NotSameAttributeException("target");
			}
		}
		return result;
	}
	
	/**
	 * checks if all the associations of the collection asso have the same owner. Throws NotSameAttributeException when
	 * not.
	 * 
	 * @param asso
	 *            the collection of the associations to check.
	 * @return the owner of all associations
	 * @throws NotSameAttributeException
	 *             thrown when min. 1 association has a other owner.
	 */
	private UserType checkSameOwner(final Collection<UnidirectionalAssociation> asso) throws NotSameAttributeException {
		final Iterator<UnidirectionalAssociation> i = asso.iterator();
		final UserType result = i.next().getOwner();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			if (!current.getOwner().equals(result)) {
				throw new NotSameAttributeException("owner");
			}
		}
		return result;
	}
	
	/**
	 * checks if all the associations of the collection asso have unique flag. Throws NotSameAttributeException when
	 * not.
	 * 
	 * @param asso
	 *            the collection of the associations to check.
	 * @return the Unique-Flag of all associations
	 * @throws NotSameAttributeException
	 *             thrown when min. 1 association has a other unique flag.
	 */
	private boolean checkUnique(final Collection<UnidirectionalAssociation> asso) throws NotSameAttributeException {
		final Iterator<UnidirectionalAssociation> i = asso.iterator();
		final boolean result = i.next().isUnique();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			if (!current.isUnique() == result) {
				throw new NotSameAttributeException("unique");
			}
		}
		return result;
	}
	
	/**
	 * checks if all the associations of the collection asso have essential flag. Throws NotSameAttributeException when
	 * not.
	 * 
	 * @param asso
	 *            the collection of the associations to check.
	 * @return the Essential-Flag of all associations
	 * @throws NotSameAttributeException
	 *             thrown when min. 1 association has a other essential flag.
	 */
	private Boolean checkEssential(final Collection<UnidirectionalAssociation> asso) throws NotSameAttributeException {
		final Iterator<UnidirectionalAssociation> i = asso.iterator();
		final boolean result = i.next().isEssential();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			if (!current.isEssential() == result) {
				throw new NotSameAttributeException("essential");
			}
		}
		return result;
	}
	
	/**
	 * checks if all the associations of the collection asso the same name. Throws NotSameAttributeException when not.
	 * 
	 * @param asso
	 *            the collection of the associations to check.
	 * @return the name of all associations
	 * @throws NotSameAttributeException
	 *             thrown when min. 1 association has a other name.
	 */
	private String checkName(final Collection<UnidirectionalAssociation> asso) throws NotSameAttributeException {
		final Iterator<UnidirectionalAssociation> i = asso.iterator();
		final String result = i.next().getName();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			if (!current.getName().equals(result)) {
				throw new NotSameAttributeException("Name");
			}
		}
		return result;
	}
	
	/**
	 * Removes this abstract user type while keeping all data.
	 * 
	 * @param ownedAssociations
	 *            the owned associations
	 * @param type
	 *            to be removed
	 * @param subtypes
	 *            a collection of subtypes
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 * @throws NotSameAttributeException
	 *             wrong subtypes list
	 */
	public void deleteUserType(final UserType type,
			final Collection<UnidirectionalAssociation> ownedAssociations,
			final Collection<UserType> subtypes) throws PersistenceException, NotSameAttributeException {
		if (!type.isAbs()) {
			throw new NotAbstractException();
		}
		
		final Iterator<UnidirectionalAssociation> i = ownedAssociations.iterator();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			this.pushDown(current, subtypes);
		}
		this.classFacade.deleteUserType(type.getId());
	}
	
	/**
	 * Inserts a new abstract class between an existing specialisation relation (lowerType, upperType).
	 * 
	 * @param lowerType
	 *            the type under the specialisation
	 * @param upperType
	 *            the type over the specialisation
	 * @param name
	 *            the name of the new user type
	 * @return the created user type
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 * @throws NotSameAttributeException
	 *             thrown when upperType is not a super type of lowerType
	 * @return
	 */
	public UserType insertUserTypeBetween(final UserType lowerType, final UserType upperType, final String name)
			throws NotSameAttributeException, PersistenceException {
		this.checkIsSuperTypeOf(lowerType, upperType);
		final UserType newType = this.classFacade.createUserType(name, true, false);
		this.classFacade.createSpecializationBetween(newType, lowerType);
		this.classFacade.createSpecializationBetween(upperType, newType);
		this.classFacade.finalizeSpecialization();
		return newType;
	}
	
	/**
	 * Creates a new type including all old associations and adds a new association from the old owner to the new type.
	 * 
	 * @param assoOld
	 *            associations to be moved - same owner!
	 * @param typeName
	 *            name for new type
	 * @throws NotSameAttributeException
	 *             thrown then not all associations have the same owner
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	public void extractClassFromUnidirectionalAssociations(final List<UnidirectionalAssociation> assoOld,
			final String typeName) throws NotSameAttributeException, PersistenceException {
		final UserType from = this.checkSameOwner(assoOld);
		final UserType newType = this.classFacade.createUserType(typeName, false, false);
		final UnidirectionalAssociation newAsso =
				this.classFacade.createUnidirectionalAssociation("my" + typeName, true, true, from, newType);
		final List<UnidirectionalAssociation> newAssos =
				this.createUnidirectionalAssociationCopiesWithNewOwner(assoOld, newType);
		
		this.classFacade.moveLinksAndCreateObjects(this.getIds(assoOld), newAsso, newType, this.getIds(newAssos));
		
		this.deleteUnidirectionalAssociations(assoOld);
	}
	
	/**
	 * get a list of all associations of the collection assoOld.
	 * 
	 * @param assoOld
	 *            the collection of associations.
	 * @return a list of all given associations.
	 */
	private List<Long> getIds(final List<UnidirectionalAssociation> assoOld) {
		final List<Long> result = new ArrayList<>();
		final Iterator<UnidirectionalAssociation> i = assoOld.iterator();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			result.add(current.getId());
		}
		return result;
	}
	
	/**
	 * delete all associations of the collection assoOld.
	 * 
	 * @param assoOld
	 *            the collection of associations should be deleted.
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	private void deleteUnidirectionalAssociations(final Collection<UnidirectionalAssociation> assoOld)
			throws PersistenceException {
		final Iterator<UnidirectionalAssociation> i = assoOld.iterator();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			this.classFacade.deleteAssociation(current.getId());
		}
	}
	
	/**
	 * creates new associations between the targets of the associations of the collection assoOld and the owner newType.
	 * 
	 * @param assoOld
	 *            a collection of associations.
	 * @param newType
	 *            the owner for the new associations.
	 * @return the list of the created associations.
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	private List<UnidirectionalAssociation> createUnidirectionalAssociationCopiesWithNewOwner(final Collection<UnidirectionalAssociation> assoOld,
			final UserType newType) throws PersistenceException {
		final List<UnidirectionalAssociation> result = new ArrayList<>();
		final Iterator<UnidirectionalAssociation> i = assoOld.iterator();
		while (i.hasNext()) {
			final UnidirectionalAssociation current = i.next();
			final UnidirectionalAssociation copy =
					this.classFacade.createUnidirectionalAssociation(
							current.getName(),
							current.isEssential(),
							current.isUnique(),
							newType,
							current.getTarget());
			result.add(copy);
		}
		return result;
	}
	
	public ClassFacade getClassFacade() {
		return this.classFacade;
	}
	
	public ObjectFacade getObjectFacade() {
		return this.objectFacade;
	}
	
}
