package de.fhdw.wtf.generator.java.generatorModel;

import java.util.HashMap;
import java.util.Map;
import java.util.Vector;

import de.fhdw.wtf.generator.java.visitor.GenClassClassVisitor;
import de.fhdw.wtf.generator.java.visitor.GenClassClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenClassClassVisitorReturn;

/**
 * Represents an external class. Such a class may not be changed by e.g. adding operations to it. It is used to link
 * generated classes via extends to externally available classes, or to reference external classes in parameter types.
 * This is used e.g. to generate load constructors that require a parameter of type UserObject.
 * 
 * The class features a static getInstance() method which does a bit of bookkeeping in order to create at most one
 * GenExternalClassClass object per external interface.
 */
public class GenExternalClassClass extends GenClassClass {
	
	/**
	 * Maps fully qualified names of external interfaces to GenExternalClassClass objects.
	 */
	private static Map<String, GenExternalClassClass> extIfaceMap = new HashMap<>();
	
	/**
	 * Creates a GenExternalClassClass object.
	 * 
	 * @param packageName
	 *            The package name of the external interface.
	 * @param typeName
	 *            The type name of the external interface.
	 */
	protected GenExternalClassClass(final String packageName, final String typeName) {
		super(typeName, new Vector<GenJavaOperation>(), new Vector<GenInterfaceClass>(),
				new Vector<GenJavaAttribute>(), new Vector<GenClassModifier>(), new Vector<GenJavaOperation>(), null,
				GenUnqualifiedPackage.create(packageName), null, null);
	}
	
	/**
	 * Returns a GenExternalClassClass object for a given external Class, specified by a fully qualified name.
	 * 
	 * @param fullyQualifiedName
	 *            The fully qualified name of an external interface.
	 * @return The GenExternalClassClass object.
	 */
	public static synchronized GenExternalClassClass getInstance(final String fullyQualifiedName) {
		GenExternalClassClass iface = extIfaceMap.get(fullyQualifiedName);
		if (iface == null) {
			final int pos = fullyQualifiedName.lastIndexOf('.');
			if (pos < 0) {
				iface = new GenExternalClassClass("", fullyQualifiedName);
			} else {
				iface =
						new GenExternalClassClass(fullyQualifiedName.substring(0, pos),
								fullyQualifiedName.substring(pos + 1));
			}
			
			extIfaceMap.put(fullyQualifiedName, iface);
		}
		return iface;
	}
	
	@Override
	public void accept(final GenClassClassVisitor visitor) {
		visitor.handle(this);
	}
	
	@Override
	public <X> X accept(final GenClassClassVisitorReturn<X> visitor) {
		return visitor.handle(this);
	}
	
	@Override
	public <Y extends Exception> void accept(final GenClassClassVisitorException<Y> visitor) throws Y {
		visitor.handle(this);
	}
	
}
