package de.fhdw.wtf.generator.java.generatorModel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import de.fhdw.wtf.generator.java.visitor.GenClassVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenClassVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenInterfaceClassVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenOperationStateVisitor;
import de.fhdw.wtf.generator.java.visitor.GenTypeVisitor;
import de.fhdw.wtf.generator.java.visitor.GenTypeVisitorException;
import de.fhdw.wtf.generator.java.visitor.GenTypeVisitorReturn;

public abstract class GenClass extends GenType {
	
	@Override
	public String toString() {
		return this.getName();
	}
	
	/**
	 * The comment that describes this class.
	 */
	private GenComment comment;
	
	/**
	 * The operations the class has.
	 */
	private final Collection<GenJavaOperation> operations;
	
	/**
	 * The interfaces the class implements.
	 */
	private final Collection<GenInterfaceClass> implement;
	
	/**
	 * The package the class will be generated in.
	 */
	private GenPackage packag;
	
	/**
	 * Additional lines of code.
	 */
	private String nonGeneratedPart;
	
	/**
	 * The generics of the class.
	 */
	private final ArrayList<Generic> generics;
	
	/**
	 * Field for import block.
	 */
	private String imports = "";
	
	/**
	 * The inner-classes of the class.
	 */
	private final Collection<GenClassClass> innerClasses;
	
	/**
	 * Calls the handle method in the given visitor.
	 * 
	 * @param visitor
	 *            The visitor to handle this type.
	 */
	public abstract void accept(de.fhdw.wtf.generator.java.visitor.GenClassVisitor visitor);
	
	/**
	 * Calls the handle method in the given visitor.
	 * 
	 * @param visitor
	 *            The visitor to handle this type.
	 * @param <X>
	 *            The returntype of the handle and the accept method.
	 * @return Returns the result of type <X> that is returned by the handle method.
	 */
	public abstract <X> X accept(GenClassVisitorReturn<X> visitor);
	
	/**
	 * Calls the handle method in the given visitor.
	 * 
	 * @param visitor
	 *            The visitor to handle this type.
	 * @param <Y>
	 *            The exception that can be thrown by the handle method.
	 * @throws Y
	 *             Throws an exception of type <X> when the handle method throws it.
	 */
	public abstract <Y extends java.lang.Exception> void accept(GenClassVisitorException<Y> visitor) throws Y;
	
	/**
	 * Constructor for GenClass.
	 * 
	 * @param name
	 *            The name of the GenClass and the java-file to generate.
	 * @param operations
	 *            The operations the GenClass has.
	 * @param implement
	 *            The interfaces the GenClass implements.
	 * @param comment
	 *            The comment that describes this GenClass.
	 * @param packag
	 *            The package the GenClass will be generated in.
	 * @param nonGeneratedPart
	 *            Additional lines of code.
	 */
	protected GenClass(final String name,
			final Collection<GenJavaOperation> operations,
			final Collection<GenInterfaceClass> implement,
			final GenPackage packag,
			final GenComment comment,
			final String nonGeneratedPart) {
		super(name);
		this.comment = comment;
		this.operations = operations;
		this.implement = implement;
		this.packag = packag;
		this.nonGeneratedPart = nonGeneratedPart;
		this.generics = new ArrayList<>();
		this.innerClasses = new Vector<>();
	}
	
	@Override
	public void accept(final GenTypeVisitor visitor) {
		visitor.handle(this);
	}
	
	@Override
	public <X> X accept(final GenTypeVisitorReturn<X> visitor) {
		return visitor.handle(this);
	}
	
	@Override
	public <X extends java.lang.Exception> void accept(final GenTypeVisitorException<X> visitor) throws X {
		visitor.handle(this);
	}
	
	/**
	 * Returns a collection of all operations this GenClass has.
	 * 
	 * BE CAREFUL! To make sure operations are properly added using "addOperation" this method returns a COPY of the
	 * collection of the operations the GenClass has. The contained operations are the SAME, but the collection is a new
	 * reference.
	 * 
	 * @return collection of operations
	 */
	public Collection<GenJavaOperation> getOperations() {
		final Collection<GenJavaOperation> copy = new Vector<>();
		copy.addAll(this.operations);
		return copy;
	}
	
	/**
	 * Returns a collection of all operations this GenClass has.
	 * 
	 * @return operations
	 */
	protected Collection<GenJavaOperation> getOperationsInner() {
		return this.operations;
	}
	
	/**
	 * Adds a GenJavaOperation to this GenClass.
	 * 
	 * @param newOperation
	 *            The operation that shall be generated in this class.
	 */
	public void addOperation(final GenJavaOperation newOperation) {
		this.getOperationsInner().add(newOperation);
	}
	
	/**
	 * Searches the given operation in the list of operations compared by the operation signature (
	 * {@link GenOperationSignature}). If the operation is found it will be overriden by the new operation "op" and true
	 * will be returned. If no operation with the signature is found the operation "op" will not be added and false will
	 * be returned.
	 * 
	 * @param op
	 *            - operation for override.
	 * @return if an existing operation was overridden.
	 */
	public boolean overrideExistingOperation(final GenJavaOperation op) {
		final Iterator<GenJavaOperation> it = this.getOperationsInner().iterator();
		while (it.hasNext()) {
			final GenJavaOperation next = it.next();
			if (op.getName().startsWith("$getTuple_")) {
				if (next.getName().contains(op.getName())) {
					op.getState().accept(new GenOperationStateVisitor() {
						
						@Override
						public void handle(final GenSimpleOperationState s) {
							next.getState().accept(new GenOperationStateVisitor() {
								
								@Override
								public void handle(final GenSimpleOperationState p) {
									// Nothing
									
								}
								
								@Override
								public void handle(final GenFullParsedOperationState p) {
									p.setMethodBody(s.getFullOperationWithPossibleImplementation().substring(
											s.getFullOperationWithPossibleImplementation().indexOf('{') + 4,
											s.getFullOperationWithPossibleImplementation().lastIndexOf('}')));
								}
								
							});
							
						}
						
						@Override
						public void handle(final GenFullParsedOperationState s) {
							// Nothing
						}
						
					});
					return true;
				}
			} else if (op.getSignature().equals(next.getSignature())) {
				this.getOperationsInner().remove(next);
				this.getOperationsInner().add(op);
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Searches the given operation in the list of operations compared by the operation signature (
	 * {@link GenOperationSignature}) in all super classes. If the operation is found it will be added to the operations
	 * of this class. If no operation with the signature is found the operation "m" will not be added.
	 * 
	 * @param m
	 *            - operation for override abstract operation.
	 */
	public void tryToOverrideAbstractOperation(final GenJavaOperation m) {
		if (this.checkForAbstractOperation(m)) {
			this.addOperation(m);
			return;
		}
		
	}
	
	/**
	 * Searches the given operation in the list of operations compared by the operation signature (
	 * {@link GenOperationSignature}) in all super classes. If the operation is found true will be returned. If no
	 * operation with the signature is found false will be returned.
	 * 
	 * @param m
	 *            - operation for override abstract operation.
	 * @return true if a operation with the same signature as m is found in this class or super classes.
	 */
	public boolean checkForAbstractOperation(final GenJavaOperation m) {
		final Iterator<GenJavaOperation> i = this.getOperations().iterator();
		while (i.hasNext()) {
			final GenJavaOperation genJavaOperation = i.next();
			if (genJavaOperation.getSignature().equals(m.getSignature())) {
				return true;
			}
		}
		final Iterator<GenInterfaceClass> interfacesIterator = this.getImplement().iterator();
		while (interfacesIterator.hasNext()) {
			final GenInterfaceClass genInterfaceClass = interfacesIterator.next();
			if (genInterfaceClass.checkForAbstractOperation(m)) {
				return true;
			}
			
		}
		return false;
	}
	
	/**
	 * Returns the Collection of GenInterfaceClass with the interfaces the class implements.
	 * 
	 * @return implement
	 */
	public Collection<GenInterfaceClass> getImplement() {
		return this.implement;
	}
	
	/**
	 * Returns the comment of the class.
	 * 
	 * @return comment.
	 */
	public GenComment getComment() {
		return this.comment;
	}
	
	/**
	 * Returns the package of the class.
	 * 
	 * @return packag
	 */
	public GenPackage getPackag() {
		return this.packag;
	}
	
	/**
	 * Returns the additional lines of code.
	 * 
	 * @return nonGeneratedPart
	 */
	public String getNonGeneratedPart() {
		return this.nonGeneratedPart;
	}
	
	/**
	 * Returns all inner classes of this GenClass.
	 * 
	 * @return Collection<GenClassClass>
	 */
	public Collection<GenClassClass> getInnerClasses() {
		return this.innerClasses;
	}
	
	/**
	 * Returns the imports block.
	 * 
	 * @return imports block
	 */
	public String getImports() {
		return this.imports;
	}
	
	/**
	 * Returns the generics of the class.
	 * 
	 * @return generics
	 */
	public ArrayList<Generic> getGenerics() {
		return this.generics;
	}
	
	/**
	 * Set the package of the class.
	 * 
	 * @param packag
	 *            The new package for the class.
	 */
	public void setPackag(final GenPackage packag) {
		this.packag = packag;
	}
	
	/**
	 * Set the comment of the class.
	 * 
	 * @param comment
	 *            The new comment for the class.
	 */
	public void setComment(final GenComment comment) {
		this.comment = comment;
	}
	
	/**
	 * Adds the {@link GenClassClass} <code>c</code> to the innerClasses collection.
	 * 
	 * @param c
	 *            The inner GenClassClass for the GenClass.
	 */
	public void addInnerClass(final GenClassClass c) {
		this.getInnerClasses().add(c);
	}
	
	/**
	 * Set the additional lines of code of the class.
	 * 
	 * @param nonGeneratedPart
	 *            The new additional lines of code for the class.
	 */
	public void setNonGeneratedPart(final String nonGeneratedPart) {
		this.nonGeneratedPart = nonGeneratedPart;
	}
	
	/**
	 * Sets field imports to imports.
	 * 
	 * @param imports
	 *            block of imports
	 */
	public void setImports(final String imports) {
		this.imports = imports;
	}
	
	@Override
	public String getFullyQualifiedTypeName() {
		String result = "";
		if (this.getPackag().toString().equals("")) {
			result = this.getName();
		} else {
			result = this.getPackag() + "." + this.getName();
		}
		return result;
	}
	
	@Override
	public String getFullyQualifiedTypeNameWithGenericArguments() {
		return this.getFullyQualifiedTypeName();
	}
	
	/**
	 * Returns the file ending to use in order to write this class to a file.
	 * 
	 * @return The file ending, including the leading dot.
	 */
	public String getFileEnding() {
		return ".java";
	}
	
	/**
	 * Returns the implementing class. This is defined for GenClassClass classes (which are their own implementing
	 * classes) and for GenInterfaceWithClassImplClass interfaces (where the implementing class is determined by calling
	 * getClassRepresentation()). In all other cases, null is returned.
	 * 
	 * @return The implementing class or null if none available.
	 */
	public GenClassClass getImplementor() {
		return this.accept(new GenClassVisitorReturn<GenClassClass>() {
			
			@Override
			public GenClassClass handle(final GenClassClass classClass) {
				return classClass;
			}
			
			@Override
			public GenClassClass handle(final GenInterfaceClass interfaceClass) {
				return interfaceClass.accept(new GenInterfaceClassVisitorReturn<GenClassClass>() {
					
					@Override
					public GenClassClass handle(final GenSimpleInterfaceClass simpleInterface) {
						return null;
					}
					
					@Override
					public GenClassClass handle(final GenInterfaceWithClassImplClass interfaceWithImplClass) {
						return interfaceWithImplClass.getClassRepresentation();
					}
					
					@Override
					public GenClassClass handle(final GenExternalInterfaceClass iface) {
						return null;
					}
				});
			}
			
			@Override
			public GenClassClass handle(final GenPrimitiveClass primitiveClass) {
				return null;
			}
			
		});
	}
	
}
