/**
 * 
 */
package de.fhdw.wtf.generator.transformer.transformers.classTransformer;

import java.util.Collection;
import java.util.List;
import java.util.Vector;

import de.fhdw.wtf.common.ast.Attribute;
import de.fhdw.wtf.common.ast.Model;
import de.fhdw.wtf.common.ast.type.AtomicType;
import de.fhdw.wtf.common.ast.type.BaseType;
import de.fhdw.wtf.common.ast.type.ClassType;
import de.fhdw.wtf.common.ast.type.CompositeType;
import de.fhdw.wtf.common.ast.type.ExceptionClassType;
import de.fhdw.wtf.common.ast.type.Type;
import de.fhdw.wtf.common.ast.type.TypeProxy;
import de.fhdw.wtf.common.ast.visitor.AtomicTypeVisitor;
import de.fhdw.wtf.common.ast.visitor.TypeVisitor;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.common.task.TaskExecutor;
import de.fhdw.wtf.generator.java.generatorModel.GenAnyType;
import de.fhdw.wtf.generator.java.generatorModel.GenClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassClass;
import de.fhdw.wtf.generator.java.generatorModel.GenClassModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenComment;
import de.fhdw.wtf.generator.java.generatorModel.GenException;
import de.fhdw.wtf.generator.java.generatorModel.GenFullParsedOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenHasGenericType;
import de.fhdw.wtf.generator.java.generatorModel.GenInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaAttribute;
import de.fhdw.wtf.generator.java.generatorModel.GenJavaOperation;
import de.fhdw.wtf.generator.java.generatorModel.GenOperationModifier;
import de.fhdw.wtf.generator.java.generatorModel.GenPackage;
import de.fhdw.wtf.generator.java.generatorModel.GenParameter;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleInterfaceClass;
import de.fhdw.wtf.generator.java.generatorModel.GenSimpleOperationState;
import de.fhdw.wtf.generator.java.generatorModel.GenType;
import de.fhdw.wtf.generator.java.generatorModel.GenTypeReferenceByName;
import de.fhdw.wtf.generator.java.generatorModel.GenUnqualifiedPackage;
import de.fhdw.wtf.generator.java.generatorModel.GenUserClass;
import de.fhdw.wtf.generator.java.generatorModel.GenVisibility;
import de.fhdw.wtf.generator.java.generatorModel.GeneratorModel;
import de.fhdw.wtf.generator.java.generatorModel.Generic;
import de.fhdw.wtf.generator.transformer.util.Tuple;

/**
 * If an class has a symmetric attribute the class must have some overhead for the correct functionality of symmetry.
 * This Task generates an inner class which implements a setter for symmetric attributes as a part of this overhead.
 */
public class InnerClassCreatorForSymmetricAttributesTransformer extends TypeTransformer {
	
	/**
	 * Represents the name for the set-operation in the setter-interface.
	 */
	private static final String SETOPERATIONNAME = "set";
	
	/**
	 * @param model
	 *            ASTModel
	 * @param taskmanager
	 *            TaskExecutor
	 * @param generatorModel
	 *            GeneratorModel
	 */
	public InnerClassCreatorForSymmetricAttributesTransformer(final Model model,
			final TaskExecutor taskmanager,
			final GeneratorModel generatorModel) {
		super(model, taskmanager, generatorModel);
	}
	
	/**
	 * Represents the name for the setter interface for symmetric attributes.
	 */
	private static final String SYMMETRICSETTERINTERFACENAME = "Setter";
	
	@Override
	public void finalizeTask() throws TaskException {
		// nothing to do
	}
	
	/**
	 * Represents the setter interface for symmetric attributes.
	 */
	private GenClass setterInterface;
	
	@Override
	public void beginTask() throws TaskException {
		this.setterInterface = this.getSymmetrySetterInterface();
		this.getGeneratorModel().setSymmetricSetterInterface(this.setterInterface);
	}
	
	@Override
	public void handleType(final Type c) throws TaskException {
		c.accept(new TypeVisitor() {
			
			@Override
			public void handle(final TypeProxy typeProxy) {
				// nothing to do here
			}
			
			@Override
			public void handle(final CompositeType compositeType) {
				// nothing to do here
			}
			
			@Override
			public void handle(final AtomicType atomicType) {
				atomicType.accept(new AtomicTypeVisitor() {
					
					@Override
					public void handle(final ClassType clazz) {
						try {
							InnerClassCreatorForSymmetricAttributesTransformer.this
									.addprivateSetterInterfaceIfNecessary(clazz);
						} catch (final Exception e) {
							// TODO: error treatment
						}
					}
					
					@Override
					public void handle(final BaseType baseType) {
						// nothing to do here
					}
				});
			}
		});
	}
	
	/**
	 * Adds the private static setter class to the given class for each symmetric attribute.
	 * 
	 * @param classType
	 *            represents the class where this function search for symmetric attributes.
	 * @throws Exception
	 *             Throws an exception if the addInnerSetterClassToImplClass-operation fails.
	 */
	protected void addprivateSetterInterfaceIfNecessary(final ClassType classType) throws Exception {
		for (final Attribute a : classType.getAttributes()) {
			if (a.isSymmetric()) {
				this.addInnerSetterClassToImplClass(classType, a);
			}
		}
	}
	
	/**
	 * 
	 * @param classType
	 *            Represents the classType which contains the given attribute.
	 * @param attribute
	 *            Represents the symmetric attribute which requires an internal setter class.
	 * @throws Exception
	 *             Throws an exception if no genClass for the given class can be found in the genModel.
	 */
	
	private void addInnerSetterClassToImplClass(final ClassType classType, final Attribute attribute) throws Exception {
		for (final Tuple<GenType, Collection<ExceptionClassType>> key : this.getTypeMapping().keySet()) {
			if (this.getTypeMapping().get(key).equals(classType)) {
				((GenClassClass) key.getA()).addInnerClass(this.getInnerStaticClassByAttribute(
						(GenClassClass) key.getA(),
						attribute));
				return;
			}
		}
		// no *$Impl-Class found.
		// will not happen if this.getTypeMapping() is filled correctly.
		throw new Exception("no *$Impl-Class found");
	}
	
	/**
	 * @param aImplClass
	 *            .
	 * @param attribute
	 *            .
	 * @return returns the inner static setter class for the given attribute.
	 * 
	 */
	
	private GenClassClass getInnerStaticClassByAttribute(final GenClassClass aImplClass, final Attribute attribute) {
		/*
		 * private static class ASetter implements Setter<AnyType, AnyType> {
		 * 
		 * @Override public void set(AnyType owner, AnyType target) { ((A) t1).b = t2; } }
		 */
		final Collection<GenJavaOperation> operations = new Vector<>();
		operations.add(this.getOperationImplementation(aImplClass, attribute));
		final Collection<GenInterfaceClass> implement = new Vector<>();
		implement.add((GenInterfaceClass) this.setterInterface);
		final Collection<GenClassModifier> modifiers = new Vector<>();
		modifiers.add(GenClassModifier.STATIC);
		
		final Collection<GenJavaOperation> constructors = new Vector<>();
		final String nonGeneratedPart = "";
		final Collection<GenJavaAttribute> attributes = new Vector<>();
		final GenComment comment = GenComment.create("Setter Implementation for symmetric attributes");
		final GenClassClass result =
				GenUserClass.create(
						attribute.getName() + SYMMETRICSETTERINTERFACENAME,
						operations,
						implement,
						attributes,
						modifiers,
						constructors,
						null, // what is the GenClassClass(-Singleton)-class for a "no extends" statement?
						null, // why do we need a package specification for an inner class?
						comment,
						nonGeneratedPart);
		return result;
	}
	
	/**
	 * @param genClassClass
	 *            .
	 * @param attribute
	 *            represents the assigned symmetric attribute for the operation which will be returned.
	 * @return returns the content of the inner class which implements the setter interface.
	 * 
	 */
	
	private GenJavaOperation getOperationImplementation(final GenClassClass genClassClass, final Attribute attribute) {
		/*
		 * generiert folgendes:
		 * 
		 * public void set(A owner, B target) { owner.b =target; } } problem: das interface definiert set(AnyType owner,
		 * AnyType target), diese Operation implementiert aber nur set(A owner, B target), sodass dies in eclipse als
		 * Fehler markiert wird, weil theoretisch "set(new C(), new D());" aufgerufen werden könnte, aber dafür keine
		 * Methode gefunden wird. TODO: dieser Operation muss eine "@Override"-Annotation hinzugefügt werden.
		 */
		
		final String methodbody = "owner." + attribute.getAttrType().getTypeString() + " = target;";
		final List<GenParameter> parameters = new Vector<>();
		parameters.add(GenParameter.create("owner", genClassClass));
		parameters.add(GenParameter.create(
				"target",
				GenTypeReferenceByName.create(attribute.getAttrType().getTypeString())));
		final GenJavaOperation result =
				GenJavaOperation.create(SETOPERATIONNAME, parameters, GenFullParsedOperationState.create(
						GenComment.create(""),
						new Vector<GenException>(),
						GenTypeReferenceByName.create(""),
						new Vector<GenOperationModifier>(),
						GenVisibility.PUBLIC,
						methodbody));
		return result;
	}
	
	/**
	 * 
	 * @param nameOfType
	 *            of the generic type.
	 * @return returns an generic with name nameOfType and type AnyType.
	 */
	private Generic getGenericForSetterInterface(final String nameOfType) {
		return Generic.create(nameOfType, GenHasGenericType.create(GenAnyType.getInstance()));
	}
	
	/**
	 * 
	 * @param name
	 *            Represents the name of the operation.
	 * @return returns an operation with the given name and generics for the setter interface.
	 */
	private GenJavaOperation getOperationForSetterInterface(final String name) {
		final List<GenParameter> parameters = new Vector<>();
		parameters.add(GenParameter.create(name, this.getGenericForSetterInterface(name.toUpperCase())));
		final GenJavaOperation result = GenJavaOperation.create(name, parameters, GenSimpleOperationState.create(""));
		return result;
	}
	
	/**
	 * 
	 * @return returns the interface for setter for symmetric attributes.
	 */
	private GenSimpleInterfaceClass getSymmetrySetterInterface() {
		final Collection<GenJavaOperation> operations = new Vector<>();
		GenPackage classPackage = GenUnqualifiedPackage.create("generated");
		classPackage = classPackage.addName("symmetry");
		operations.add(this.getOperationForSetterInterface(SETOPERATIONNAME));
		final GenSimpleInterfaceClass result =
				GenSimpleInterfaceClass.create(
						SYMMETRICSETTERINTERFACENAME,
						operations,
						new Vector<GenInterfaceClass>(),
						classPackage,
						GenComment.create("Represents an interface for the setter for symmetric attributes."),
						"");
		result.getGenerics().add(this.getGenericForSetterInterface("T1"));
		result.getGenerics().add(this.getGenericForSetterInterface("T2"));
		return result;
	}
	
	@Override
	public String toString() {
		return "InnerClassCreatorForSymmetricAttributesTransformer";
	}
}
