package de.fhdw.wtf.generator.java.generatorModel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import de.fhdw.wtf.common.ast.type.BaseType;
import de.fhdw.wtf.common.ast.type.ClassType;
import de.fhdw.wtf.common.ast.type.ExceptionClassType;
import de.fhdw.wtf.common.ast.type.Type;
import de.fhdw.wtf.common.exception.walker.TaskException;
import de.fhdw.wtf.facade.PackageConstants;
import de.fhdw.wtf.generator.java.visitor.GenPrimitiveTypeVisitor;
import de.fhdw.wtf.generator.java.visitor.GenPrimitiveTypeVisitorReturn;
import de.fhdw.wtf.generator.java.visitor.GenTypeVisitor;
import de.fhdw.wtf.generator.java.visitor.GenTypeVisitorReturn;
import de.fhdw.wtf.generator.transformer.exception.GenTypeNotReferencedException;
import de.fhdw.wtf.generator.transformer.transformers.classTransformer.UtilTransformer;
import de.fhdw.wtf.generator.transformer.util.Tuple;

/**
 * A GeneratorModel is used to represent a given Model from Core as a Java-Model, that can be generated and that
 * provides all features of WTF with the generated sources.
 */
public class GeneratorModel {
	
	/**
	 * A collection of all GenClasses that additionally to the AST-Model needs to be generated. Those "classes" are for
	 * example the AppStarter, Visitors, Factories, the FinderAspect and additional classes to realize symmetric
	 * relations.
	 */
	private final Vector<GenClass> nonASTClasses;
	/**
	 * TODO
	 */
	private final Map<GenUserClass, Vector<ClassType>> symmetricManagers;
	/**
	 * A mapping from AST-Types to GenPrimitiveClasses. GenPrimitiveClasses will not be generated as they already exist
	 * in de.wtf.context.model.
	 */
	private final Map<Type, GenPrimitiveClass> baseClassMapping;
	/**
	 * A mapping from AST-Types to GenClasses. GenClasses can be Interfaces or Classes that will be generated.
	 */
	private final Map<Type, GenClass> classMapping;
	/**
	 * A mapping that contains all AST-Types to a Tuple of GenType and a collection of ExceptionClassType that are part
	 * of the AST-model.
	 */
	private final Map<Type, Tuple<GenType, Collection<ExceptionClassType>>> typeMapping;
	/**
	 * A mapping from Gen-Types to GenAspects. GenAspects represent AspectJ-files that will be generated and shall
	 * expand the GenClass provided as key. A GenClass represents an AST-Type.
	 */
	private final Map<GenClass, GenAspect> aspectJMapping;
	private GenClass setterInterface;
	
	public GenClass getSetterInterface() {
		return this.setterInterface;
	}
	
	/**
	 * Suffix for aspect-name.
	 */
	private static final String ASPECT_SUFFIX = "Aspect";
	/**
	 * Suffix for aspect-name of an aspect for the expansion of an Ast-BaseType.
	 */
	private static final String BASETYPE_EXPANSION_ASPECT_SUFFIX = "Expansion" + ASPECT_SUFFIX;
	/**
	 * Standard comment for an aspect.
	 */
	private static final String ASPECT_COMMENT = "This is an aspect.";
	
	/**
	 * Adds a class to the classes that are not classes of the AST.
	 * 
	 * @param c
	 *            The GenClass to add
	 */
	public void addNonAstClass(final GenClass c) {
		this.nonASTClasses.add(c);
	}
	
	/**
	 * Adds a symmetric manager (and one of the corresponding classes) to the inner list.
	 * 
	 * @param manager
	 *            The SymmetricRelationAccess-Manager.
	 * @param userClass
	 *            The class.
	 */
	public void addSymmetricManager(final GenUserClass manager, final ClassType userClass) {
		if (!this.symmetricManagers.containsKey(manager)) {
			this.symmetricManagers.put(manager, new Vector<ClassType>());
		}
		this.symmetricManagers.get(manager).add(userClass);
	}
	
	/**
	 * Gets the SymmetricRelationManager that handles the symmetric relation between the two classes. It returns "null"
	 * if the manager is not found.
	 * 
	 * @param clss1
	 *            First Class.
	 * @param clss2
	 *            Second Class.
	 * @return The singleton manager.
	 */
	public GenUserClass getSymmetricManager(final ClassType clss1, final ClassType clss2) {
		final Iterator<GenUserClass> i = this.symmetricManagers.keySet().iterator();
		while (i.hasNext()) {
			final GenUserClass singleton = i.next();
			if (!this.symmetricManagers.get(singleton).contains(clss1)) {
				continue;
			}
			if (!this.symmetricManagers.get(singleton).contains(clss2)) {
				continue;
			}
			return singleton;
		}
		return null;
	}
	
	/**
	 * Set the aspect that shall be generated for <code>genClass</code>.
	 * 
	 * @param genClass
	 *            The GenClass to set the aspect for
	 * @param aspect
	 *            The aspect to set
	 */
	public void addAspect(final GenClass genClass, final GenAspect aspect) {
		this.getAspectJMapping().put(genClass, aspect);
	}
	
	/**
	 * Returns all classes that will be generated. These classes can represent an AST-Type or can be classes like the
	 * AppStarter, Visitors, Factories, the FinderAspect or additional classes to realize symmetric relations.
	 * 
	 * @return Returns a collection of GenClass.
	 */
	public Collection<GenClass> getClasses() {
		final Vector<GenClass> res = new Vector<>();
		res.addAll(this.classMapping.values());
		res.addAll(this.nonASTClasses);
		res.addAll(this.aspectJMapping.values());
		// res.add(this.getSetterInterface());
		return res;
	}
	
	/**
	 * Instantiates a new GeneratorModel. A GeneratorModel is used to represent a given Model from Core as a Java-Model,
	 * that can be generated and that provides all features of WTF with the generated sources.
	 */
	public GeneratorModel() {
		this.baseClassMapping = new HashMap<>();
		this.classMapping = new HashMap<>();
		this.typeMapping = new HashMap<>();
		this.nonASTClasses = new Vector<>();
		this.aspectJMapping = new HashMap<>();
		this.symmetricManagers = new HashMap<>();
	}
	
	/**
	 * Creates a new GeneratorModel. A GeneratorModel is used to represent a given Model from Core as a Java-Model, that
	 * can be generated and that provides all features of WTF with the generated sources.
	 * 
	 * @return Returns the created GeneratorModel
	 */
	public static GeneratorModel create() {
		return new GeneratorModel();
	}
	
	/**
	 * Adds map-entries from the given AST-type to a given GenType. The GenType is added to the type-mapping in every
	 * case. Additionally also a GenClass will be added to the class-mapping.
	 * 
	 * @param type
	 *            The AST-Type to add a GenType- and if possible a GenClass-representation for.
	 * @param genTypes
	 *            A Tuple with the GenType that represents the AST-type as GenType or even as GenClass and a list of
	 *            GenExceptions defined in the AST-type.
	 */
	public void addToMapping(final Type type, final Tuple<GenType, Collection<ExceptionClassType>> genTypes) {
		final Type typeProxyFreePrototype = UtilTransformer.getTypeProxyFreePrototype(type);
		
		// TODO mapped classes shall have an unique name
		// if(this.getTypeMapping().containsKey(wtfType)){
		// throw new TypeNameAlreadyTakenException(typeName);
		// }
		
		this.typeMapping.put(typeProxyFreePrototype, genTypes);
		
		genTypes.getA().accept(new GenTypeVisitor() {
			@Override
			public void handle(final GenPrimitiveType primitiveType) {
				primitiveType.accept(new AddToMappingPrimitiveTypeVisitor(typeProxyFreePrototype));
			}
			
			@Override
			public void handle(final GenCollectionType collectionType) {
				// add to GeneratorModel.classMapping not possible
			}
			
			@Override
			public void handle(final GenClass cla) {
				GeneratorModel.this.getClassMapping().put(typeProxyFreePrototype, cla);
			}
			
			@Override
			public void handle(final GenMapType mapType) {
				// add to GeneratorModel.classMapping not possible
			}
			
			@Override
			public void handle(final Generic generic) {
				// add to GeneratorModel.classMapping not possible
			}
			
			@Override
			public void handle(final GenJavaUtilCollection javaUtilCollection) {
				// add to GeneratorModel.classMapping not possible
			}
			
			@Override
			public void handle(final GenImportType importType) {
				// add to GeneratorModel.classMapping not possible
			}
			
			@Override
			public void handle(final GenDummyType dummy) {
				// add to GeneratorModel.classMapping not possible
			}
		});
	}
	
	/**
	 * The private class to handle primitive Types while adding them to the mapping.
	 */
	private final class AddToMappingPrimitiveTypeVisitor implements GenPrimitiveTypeVisitor {
		/**
		 * The AST-Type to add a GenType- and if possible a GenClass-representation for.
		 */
		private final Type wtfType;
		
		/**
		 * Instantietes a new AddToMappingPrimitiveTypeVisitor.
		 * 
		 * @param wtfType
		 *            The AST-Type to add a GenType- and if possible a GenClass-representation for.
		 */
		private AddToMappingPrimitiveTypeVisitor(final Type wtfType) {
			this.wtfType = wtfType;
		}
		
		@Override
		public void handle(final GenVoidType voidType) {
			// nothing to do
		}
		
		@Override
		public void handle(final GenStringType stringType) {
			GeneratorModel.this.getBaseClassMapping().put(this.wtfType, GenStringType.getCorrespondingClass());
		}
		
		@Override
		public void handle(final GenIntegerType integerType) {
			GeneratorModel.this.getBaseClassMapping().put(this.wtfType, GenIntegerType.getCorrespondingClass());
		}
	}
	
	/**
	 * Returns the GenType for a AST-Type.
	 * 
	 * @param type
	 *            The type to get the GenType for.
	 * @throws GenTypeNotReferencedException
	 *             if no GenType is set for the given type
	 * @return GenType
	 */
	public GenType getGenTypeForType(final Type type) throws GenTypeNotReferencedException {
		final Type protoType = UtilTransformer.getTypeProxyFreePrototype(type);
		final Tuple<GenType, Collection<ExceptionClassType>> result = this.typeMapping.get(protoType);
		if (result == null || result.getA() == null) {
			throw new GenTypeNotReferencedException();
		}
		return result.getA();
	}
	
	/**
	 * Returns the collection of GenExceptions for a AST-Type.
	 * 
	 * @param type
	 *            The type to get the collection of exceptions for.
	 * @throws TaskException
	 *             GenExceptionsNotReferencedException -> if no list of GenExceptions is set for the given type.
	 *             TaskException -> if the underlying type of the thrownType is not a exceptionClassType.
	 * 
	 * @return Collection<GenExceptions>
	 */
	public Collection<GenException> getGenExceptionsForType(final Type type) throws TaskException {
		final Type protoType = UtilTransformer.getTypeProxyFreePrototype(type);
		final Collection<GenException> result = new ArrayList<>();
		final Collection<ExceptionClassType> exceptions = this.typeMapping.get(protoType).getB();
		final Iterator<ExceptionClassType> iterator = exceptions.iterator();
		while (iterator.hasNext()) {
			final ExceptionClassType current = iterator.next();
			final Tuple<GenType, Collection<ExceptionClassType>> types = this.typeMapping.get(current.getPrototype());
			if (types.getA() instanceof GenException) {
				result.add((GenException) types.getA());
			}
		}
		return result;
	}
	
	/**
	 * Returns the GenClass that represents the given Type if possible. May returns null.
	 * 
	 * @param type
	 *            The AST-Type to return the GenClass for.
	 * @return GenClass that represents Type.
	 */
	public GenClass getJavaClassForWTFClass(final Type type) {
		final Type prototype = UtilTransformer.getTypeProxyFreePrototype(type);
		final GenClass result = this.classMapping.get(prototype);
		if (result == null) {
			System.out.println("Impl class not found");
		}
		return result;
	}
	
	/**
	 * Creates a GenClassClass for a Type. Attributes, operations, constructors, implementations and extend will be
	 * initialized in a standard way.
	 * 
	 * @param name
	 *            Name of the class to create
	 * @param modifiers
	 *            Modifiers for the class
	 * @param type
	 *            The corresponding Ast-Type for the class
	 * @param comment
	 *            The comment to describe the class
	 * @param packag
	 *            The package of the class
	 * @param nonGeneratedPart
	 *            The additional part to the part that will be generated
	 * @return GenUserClass that represents a ClassType
	 */
	public GenUserClass createClass(final String name,
			final Collection<GenClassModifier> modifiers,
			final Type type,
			final GenComment comment,
			final GenPackage packag,
			final String nonGeneratedPart) {
		final Collection<GenJavaAttribute> attributes = new Vector<>();
		final Collection<GenJavaOperation> operations = new Vector<>();
		final Collection<GenJavaOperation> constructors = new Vector<>();
		final Collection<GenInterfaceClass> implement = new Vector<>();
		final GenClassClass extend = GenAnyType.getInstance();
		return this.createClassInternal(
				name,
				attributes,
				operations,
				modifiers,
				constructors,
				implement,
				extend,
				type,
				comment,
				packag,
				nonGeneratedPart);
	}
	
	/**
	 * Creates a GenClassClass for a Type with the given parameters.
	 * 
	 * @param name
	 *            The name of the class and the java-file to generate.
	 * @param attributes
	 *            The attribute the class has.
	 * @param operations
	 *            The operations the class has.
	 * @param modifiers
	 *            The class-modifiers for this class. For example "public".
	 * @param constructors
	 *            The constructor-operations for this class.
	 * @param implement
	 *            The interfaces the class implements.
	 * @param extend
	 *            The class the class extends.
	 * @param type
	 *            The AST-Type the class represents.
	 * @param comment
	 *            The comment that describes this class.
	 * @param packag
	 *            The package the class will be generated in.
	 * @param nonGeneratedPart
	 *            Additional lines of code.
	 * @return Returns a new GenUserClass with the given parameters.
	 */
	private GenUserClass createClassInternal(final String name,
			final Collection<GenJavaAttribute> attributes,
			final Collection<GenJavaOperation> operations,
			final Collection<GenClassModifier> modifiers,
			final Collection<GenJavaOperation> constructors,
			final Collection<GenInterfaceClass> implement,
			final GenClassClass extend,
			final Type type,
			final GenComment comment,
			final GenPackage packag,
			final String nonGeneratedPart) {
		final GenUserClass c =
				GenUserClass.create(
						name,
						operations,
						implement,
						attributes,
						modifiers,
						constructors,
						extend,
						packag,
						comment,
						nonGeneratedPart);
		return c;
	}
	
	/**
	 * Creates a GenInterfaceClass for a ClassType. Operations and implementations will be initialized in a standard
	 * way.
	 * 
	 * @param name
	 *            Name of the interface to create
	 * @param type
	 *            The corresponding AST-Type for the class
	 * @param comment
	 *            The comment to describe the class
	 * @param packag
	 *            The package of the class
	 * @param nonGeneratedPart
	 *            The additional part to the part that will be generated
	 * @return GenInterfaceClass that represents a ClassType
	 */
	public GenInterfaceClass createInterface(final String name,
			final Type type,
			final GenComment comment,
			final GenPackage packag,
			final String nonGeneratedPart) {
		final Vector<GenJavaOperation> operations = new Vector<>();
		final Vector<GenInterfaceClass> implement = new Vector<>();
		return this.createInterfaceInternal(name, operations, implement, type, comment, packag, nonGeneratedPart);
	}
	
	/**
	 * Creates a GenInterfaceClass with for a Type with the given parameters.
	 * 
	 * @param name
	 *            The name of the interface and the java-file to generate.
	 * @param operations
	 *            The operations the interface has.
	 * @param implement
	 *            The interfaces the interface implements.
	 * @param type
	 *            The AST-Type the interface represents.
	 * @param comment
	 *            The comment that describes this interface.
	 * @param packag
	 *            The package the interface will be generated in.
	 * @param nonGeneratedPart
	 *            Additional lines of code.
	 * @return Returns a new GenInterfaceClass with the given parameters.
	 */
	private GenInterfaceClass createInterfaceInternal(final String name,
			final Vector<GenJavaOperation> operations,
			final Vector<GenInterfaceClass> implement,
			final Type type,
			final GenComment comment,
			final GenPackage packag,
			final String nonGeneratedPart) {
		final GenInterfaceClass c =
				GenSimpleInterfaceClass.create(name, operations, implement, packag, comment, nonGeneratedPart);
		return c;
	}
	
	/**
	 * Creates a GenInterfaceWithClassImplClass for a ClassType. Operations and implementations will be initialized in a
	 * standard way. Also the contained "Impl"-class will be initialized in a standard way.
	 * 
	 * @param name
	 *            Name of the class to create
	 * @param type
	 *            The corresponding AST-Type for the class
	 * @param classModifiers
	 *            The class-modifiers for the inner class. For example "public".
	 * @param comment
	 *            The comment to describe the class
	 * @param packag
	 *            The package of the class
	 * @param nonGeneratedPart
	 *            The additional part to the part that will be generated
	 * @return GenInterfaceClass that represents a ClassType
	 */
	public GenInterfaceWithClassImplClass createInterfaceWithImplClass(final String name,
			final ClassType type,
			final Collection<GenClassModifier> classModifiers,
			final GenComment comment,
			final GenPackage packag,
			final String nonGeneratedPart) {
		final Vector<GenJavaOperation> operations = new Vector<>();
		final Vector<GenInterfaceClass> implement = new Vector<>();
		
		final Collection<GenClassModifier> modifiers = new Vector<>();
		final Iterator<GenClassModifier> i = classModifiers.iterator();
		while (i.hasNext()) {
			final GenClassModifier current = i.next();
			if (current != GenClassModifier.ABSTRACT) {
				modifiers.add(current);
			}
			
		}
		final GenUserClass classRepresentation =
				this.createClass(name, modifiers, type, comment, packag, nonGeneratedPart);
		return this.createInterfaceWithImplClassInternal(
				name,
				operations,
				implement,
				type,
				comment,
				packag,
				nonGeneratedPart,
				classRepresentation);
	}
	
	/**
	 * Creates a GenInterfaceWithClassImplClass with for a Type with the given parameters.
	 * 
	 * @param name
	 *            The name of the interface and the java-file to generate.
	 * @param operations
	 *            The operations the interface has.
	 * @param implement
	 *            The interfaces the interface implements.
	 * @param type
	 *            The AST-Type the interface represents.
	 * @param comment
	 *            The comment that describes this interface.
	 * @param packag
	 *            The package the interface will be generated in.
	 * @param nonGeneratedPart
	 *            Additional lines of code.
	 * @param classRepresentation
	 *            The "Impl"-class inside the interface that represents the AST-type.
	 * @return Returns a new GenInterfaceClass with the given parameters.
	 */
	private GenInterfaceWithClassImplClass createInterfaceWithImplClassInternal(final String name,
			final Vector<GenJavaOperation> operations,
			final Vector<GenInterfaceClass> implement,
			final Type type,
			final GenComment comment,
			final GenPackage packag,
			final String nonGeneratedPart,
			final GenUserClass classRepresentation) {
		final GenInterfaceWithClassImplClass i =
				GenInterfaceWithClassImplClass.create(
						name,
						operations,
						implement,
						packag,
						comment,
						nonGeneratedPart,
						classRepresentation);
		return i;
	}
	
	/**
	 * Creates a GenException for a ClassType that is an exception. Attributes, operations, modifiers, constructors,
	 * implementations and extend will be initialized in a standard way.
	 * 
	 * @param name
	 *            Name of the exception to create
	 * @param type
	 *            The corresponding AST-Type for the exception
	 * @param comment
	 *            The comment to describe the exception
	 * @param packag
	 *            The package of the exception
	 * @param nonGeneratedPart
	 *            The additional part to the part that will be generated
	 * @return GenException that represents a ClassType
	 */
	public GenException createException(final String name,
			final ClassType type,
			final GenComment comment,
			final GenPackage packag,
			final String nonGeneratedPart) {
		final Collection<GenJavaAttribute> attributes = new Vector<>();
		final Collection<GenJavaOperation> operations = new Vector<>();
		final Collection<GenClassModifier> modifiers = new Vector<>();
		final Collection<GenJavaOperation> constructors = new Vector<>();
		final Collection<GenInterfaceClass> implement = new Vector<>();
		return this.createExceptionInternal(
				name,
				attributes,
				operations,
				modifiers,
				constructors,
				implement,
				type,
				comment,
				packag,
				nonGeneratedPart);
	}
	
	/**
	 * Creates a GenException for a ClassType with the given parameters.
	 * 
	 * @param name
	 *            The name of the exception and the java-file to generate.
	 * @param attributes
	 *            The attribute the exception has.
	 * @param operations
	 *            The operations the exception has.
	 * @param modifiers
	 *            The class-modifiers for this exception. For example "public".
	 * @param constructors
	 *            The constructor-operations for this exception.
	 * @param implement
	 *            The interfaces the exception implements.
	 * @param type
	 *            The AST-ClassType the exception represents.
	 * @param comment
	 *            The comment that describes this exception.
	 * @param packag
	 *            The package the exception will be generated in.
	 * @param nonGeneratedPart
	 *            Additional lines of code.
	 * @return Returns a new GenException with the given parameters.
	 */
	private GenException createExceptionInternal(final String name,
			final Collection<GenJavaAttribute> attributes,
			final Collection<GenJavaOperation> operations,
			final Collection<GenClassModifier> modifiers,
			final Collection<GenJavaOperation> constructors,
			final Collection<GenInterfaceClass> implement,
			final ClassType type,
			final GenComment comment,
			final GenPackage packag,
			final String nonGeneratedPart) {
		final GenException result =
				GenException.create(
						name,
						operations,
						implement,
						attributes,
						modifiers,
						constructors,
						packag,
						comment,
						nonGeneratedPart);
		return result;
	}
	
	/**
	 * Returns the {@link GenAspect} for <code>baseType</code>. If no {@link GenAspect} for <code>baseType</code> was
	 * found it will return a new created {@link GenAspect}. If no {@link GenPrimitiveClass} is mapped for
	 * {@link BaseType} it will throw an exception.
	 * 
	 * @param baseType
	 *            The {@link BaseType} to get or create an GenAspect for to expand the {@link GenPrimitiveClass} that
	 *            represents the baseType.
	 * @return The GenAspect which shall contain additional declarations for the {@link GenPrimitiveClass} that is
	 *         represented by the baseType.
	 * @throws GenTypeNotReferencedException
	 *             Thrown when no {@link GenClass} is mapped for baseType.
	 */
	public GenAspect getAspectForBaseTypeExpansion(final BaseType baseType) throws GenTypeNotReferencedException {
		final GenType genType = this.getGenTypeForType(baseType);
		final GenClass genClass = genType.accept(new GenTypeVisitorReturn<GenClass>() {
			
			@Override
			public GenClass handle(final GenClass cla) {
				return cla;
			}
			
			@Override
			public GenClass handle(final GenCollectionType collectionType) {
				return null;
			}
			
			@Override
			public GenClass handle(final GenPrimitiveType primitiveType) {
				return primitiveType.accept(new GenPrimitiveTypeVisitorReturn<GenClass>() {
					
					@Override
					public GenClass handle(final GenVoidType voidType) {
						return null;
					}
					
					@Override
					public GenClass handle(final GenStringType stringType) {
						return GenStringType.getCorrespondingClass();
					}
					
					@Override
					public GenClass handle(final GenIntegerType integerType) {
						return GenIntegerType.getCorrespondingClass();
					}
				});
			}
			
			@Override
			public GenClass handle(final GenMapType mapType) {
				return null;
			}
			
			@Override
			public GenClass handle(final GenImportType importType) {
				return null;
			}
			
			@Override
			public GenClass handle(final GenDummyType dummyType) {
				return null;
			}
			
			@Override
			public GenClass handle(final Generic generic) {
				return null;
			}
			
			@Override
			public GenClass handle(final GenJavaUtilCollection javaUtilCollection) {
				return null;
			}
		});
		return this.getAspectForType(genClass);
	}
	
	/**
	 * Returns the {@link GenAspect} for <code>primitiveClass</code>. If no {@link GenAspect} for
	 * <code>primitiveClass</code> was found it will return a new created {@link GenAspect}.
	 * 
	 * @param clazz
	 *            The {@link GenPrimitiveClass} to get or create an GenAspect for to expand the primitiveClass.
	 * @return The GenAspect which shall contain additional declarations for the primitiveClass.
	 */
	public GenAspect getAspectForType(final GenClass clazz) {
		if (this.getAspectJMapping().containsKey(clazz)) {
			return this.getAspectJMapping().get(clazz);
		} else {
			final String name = clazz.getName() + GeneratorModel.BASETYPE_EXPANSION_ASPECT_SUFFIX;
			final Vector<GenJavaOperation> operations = new Vector<>();
			final Vector<GenAspectOperation> aspectOperations = new Vector<>();
			final Vector<GenInterfaceClass> impl = new Vector<>();
			final Vector<GenJavaAttribute> attributes = new Vector<>();
			final Vector<GenAspectAttribute> aspectAttributes = new Vector<>();
			final Vector<GenClassModifier> modifiers = new Vector<>();
			final Vector<GenJavaOperation> constructors = new Vector<>();
			final GenPackage packag = PackageConstants.ASPECT_PACKAGE;
			final GenComment comment = GenComment.createFromPlainText(GeneratorModel.ASPECT_COMMENT, false);
			
			final GenAspect aspect =
					GenAspect.create(
							name,
							operations,
							aspectOperations,
							impl,
							attributes,
							aspectAttributes,
							modifiers,
							constructors,
							null,
							packag,
							comment,
							"");
			this.addAspect(clazz, aspect);
			return aspect;
		}
	}
	
	/**
	 * Returns a map from AST-Types to classes that represent a primitive type.
	 * 
	 * @return Map<Type, GenPrimitiveClass>
	 */
	public Map<Type, GenPrimitiveClass> getBaseClassMapping() {
		return this.baseClassMapping;
	}
	
	/**
	 * Returns a map of all AST-Types that can be generated as GenClasses.
	 * 
	 * @return Map<Type, GenClass>
	 */
	public Map<Type, GenClass> getClassMapping() {
		return this.classMapping;
	}
	
	/**
	 * Returns a map from AST-Types to a tuple of GenType that represents them and a collection of
	 * {@link ExceptionClassType}s that represents the thrown exceptions in the AST-type.
	 * 
	 * @return Map<Type, GenType>
	 */
	public Map<Type, Tuple<GenType, Collection<ExceptionClassType>>> getTypeMapping() {
		return this.typeMapping;
	}
	
	/**
	 * Returns a map from {@link GenClass} to {@link GenAspect} that are created to help realizing the given AST-Model.
	 * 
	 * @return Map<GenClass, GenAspect>
	 */
	public Map<GenClass, GenAspect> getAspectJMapping() {
		return this.aspectJMapping;
	}
	
	/**
	 * Returns the collection of all GenClasses that additionally to the AST-Model needs to be generated. Those
	 * "classes" are for example the AppStarter, Visitors, Factories, the FinderAspect and additional classes to realize
	 * symmetric relations.
	 * 
	 * @return Vector<GenClass>
	 */
	public Vector<GenClass> getNonAstType() {
		return this.nonASTClasses;
	}
	
	public void setSymmetricSetterInterface(final GenClass setterInterface) {
		this.setterInterface = setterInterface;
	}
	
	/**
	 * Returns a Collection of all GenClasses that additionally to the AST-Model needs to be generated. All contained
	 * {@link GenClass}es represent an abstract prototype of a {@link ProductType}.
	 */
	// public Set<GenClass> getAbstractPrototypeProducts() {
	// return this.abstractPrototypeProducts;
	// }
	
}
