package de.fhdw.wtf.persistence.facade;

import java.io.IOException;

import de.fhdw.wtf.persistence.exception.ClassFacadeUninitializedException;
import de.fhdw.wtf.persistence.exception.PersistenceException;

public abstract class DatabaseManager {
	
	/**
	 * Loads the configuration from a given file.
	 * 
	 * @param fileName
	 * @throws IOException
	 */
	public abstract void setConnectionConstantsFromFile(String fileName) throws IOException;
	
	/**
	 * Connects to the Database with the given Information. After the successful execution of this method, it is
	 * possible to get Access to the default connection.
	 * 
	 * @throws PersistenceException
	 *             If the Database is not accessible or the given credentials were wrong, a Persistence Exception will
	 *             be thrown.
	 * 
	 */
	public abstract void connect() throws PersistenceException;
	
	/**
	 * Returns the class facade that works with this manager.
	 * 
	 * @return The class facade
	 */
	public abstract ClassFacade getClassFacade();
	
	/**
	 * Returns the object facade that works with this manager.
	 * 
	 * @return The object facade
	 * @throws ClassFacadeUninitializedException
	 *             thrown if the facade is not initialized
	 */
	public abstract ObjectFacade getObjectFacade() throws ClassFacadeUninitializedException;
	
	/**
	 * This Method closes the connection with the database. It should be invoked at last.
	 * 
	 * @throws PersistenceException
	 *             A Persistence Exception is thrown if something on the database side goes wrong.
	 */
	public abstract void disconnect() throws PersistenceException;
	
	/**
	 * Checks if the manager is connected.
	 * 
	 * @return Provides true whether connect has been successfully invoked or not.
	 */
	public abstract boolean isConnected();
	
	/**
	 * Clears the database. Recreates the database-schema and creates database-procedures if necessary.
	 * 
	 * @throws PersistenceException
	 *             if access to database failed.
	 * @throws IOException
	 *             will be thrown if a file is not accessible.
	 */
	public abstract void resetDatabase() throws PersistenceException, IOException;
	
	/**
	 * Clears all database tables.
	 * 
	 * @throws PersistenceException
	 *             access to database failed
	 */
	public abstract void clearTables() throws PersistenceException;
}
