package de.fhdw.wtf.persistence.meta;

import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.facade.ObjectFacade;

/**
 * A class to represent any String Value. A String Value is an object which type is always of type String.
 * 
 */
public class StringValue extends Object { // NOPMD: Object != java.lang.Object

	/**
	 * The String Value will use lazy load to fetch unset data from the database. Therefore an access to the Object
	 * Facade is needed.
	 */
	private static ObjectFacade facade;
	
	/**
	 * set the ObjectFacade, because the String Value will use lazy load to fetch unset data from the database.
	 * Therefore an access to the Object Facade is needed.
	 * 
	 * @param objectFacade
	 */
	public static void setObjectFacade(final ObjectFacade objectFacade) {
		facade = objectFacade;
	}
	
	/**
	 * The value of this Object.
	 */
	private String value;
	
	/**
	 * The unique identity of the string in the database.
	 */
	private Long id;
	
	/**
	 * Creates a new String Value if the id is known. The value will be fetched from the database if needed. Note that
	 * the id has to be a valid Id of a String Value Object in the database.
	 * 
	 * @param id
	 *            The existing Id of a String Value in the database.
	 */
	public StringValue(final long id) {
		super(StringType.getInstance());
		this.id = id;
	}
	
	/**
	 * A constructor for a new String value if only the value is known. The Id will be determined by the database later.
	 * 
	 * @param value
	 *            Any String value.
	 */
	public StringValue(final String value) {
		super(StringType.getInstance());
		this.value = value;
	}
	
	/**
	 * Constructor for a String value if both Id and value are known. The tuple if and value has to exist in the
	 * database otherwise the consistency will be violated.
	 * 
	 * @param id
	 *            The Id of this String Value in the database.
	 * @param value
	 *            The value of this String value in the database.
	 */
	public StringValue(final Long id, final String value) {
		super(StringType.getInstance());
		this.id = id;
		this.value = value;
	}
	
	/**
	 * Getter for the value. Lazy Loads the Value from the database if unset.
	 * 
	 * @return Provides the value of this String Value Object.
	 */
	public String getValue() {
		if (this.value == null) {
			try {
				this.value = facade.getStringForId(this.getId());
			} catch (final PersistenceException e) {
				throw new Error(e);
			}
		}
		return this.value;
	}
	
	/**
	 * Getter for the Id. Lazy Loads the Value from the database if unset.
	 * 
	 * @return Provides the id of this String Value Object.
	 */
	@Override
	public long getId() {
		if (this.id == null) {
			try {
				this.id = facade.getIdForString(this.getValue());
			} catch (final PersistenceException e) {
				throw new Error(e);
			}
		}
		return this.id;
	}
	
	@Override
	public boolean isTheSameAs(final java.lang.Object other) {
		if (!(other instanceof StringValue)) {
			return false;
		}
		
		return this.getId() == ((StringValue) other).getId();
	}
	
}
