package de.fhdw.wtf.persistence.facade;

import java.math.BigInteger;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.fhdw.wtf.persistence.exception.BaseTypeNotFoundException;
import de.fhdw.wtf.persistence.exception.InvalidLinkException;
import de.fhdw.wtf.persistence.exception.NotInstantiatableException;
import de.fhdw.wtf.persistence.exception.NotValidInputException;
import de.fhdw.wtf.persistence.exception.ObjectNotFoundException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.meta.AdhocTransaction;
import de.fhdw.wtf.persistence.meta.Association;
import de.fhdw.wtf.persistence.meta.IntegerType;
import de.fhdw.wtf.persistence.meta.IntegerValue;
import de.fhdw.wtf.persistence.meta.Link;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.MapLink;
import de.fhdw.wtf.persistence.meta.Object;
import de.fhdw.wtf.persistence.meta.StringType;
import de.fhdw.wtf.persistence.meta.StringValue;
import de.fhdw.wtf.persistence.meta.Transaction;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.persistence.meta.UnidirectionalLink;
import de.fhdw.wtf.persistence.meta.UserObject;
import de.fhdw.wtf.persistence.meta.UserType;
import de.fhdw.wtf.persistence.utils.Tuple;

/**
 * The class that should be used as the object facade if you do not use a database in the background.
 * 
 * @author hfw413hy
 *
 */
public class NoDatabaseObjectFacadeImplementation implements ObjectFacade {
	
	/**
	 * Represents the id which will be given to the first thing with an identifier which has been created by this
	 * ObjectFacade. Will be set during the ObjectFacades instantiation or during an execution of the clear-operation.
	 */
	private static final long INITIAL_NEXT_ID = 1;
	
	/**
	 * This ClassFacade is needed to perform consistency checks in regard to the model layer of the meta-model.
	 */
	private final ClassFacade classFacade;
	
	/**
	 * This field contains the next identifier for something that can be created within this ObjectFacade.
	 */
	private long nextId;
	
	/**
	 * Maps an id to its corresponding instance of UserObject.
	 */
	private final Map<Long, UserObject> idToUserObject;
	
	/**
	 * Stores all UserTypes (values) which are instances of a UserType (the key).
	 */
	private final Map<UserType, Collection<UserObject>> userTypeMappingToUserObjects;
	
	/**
	 * Stores the Information "ID -> Value".
	 */
	private final Map<Long, BigInteger> integerValues;
	
	/**
	 * Stores the Information "Value -> ID".
	 */
	private final Map<BigInteger, Long> inverseIntegerValues;
	
	/**
	 * Stores the Information "ID -> String".
	 */
	private final Map<Long, String> stringValues;
	
	/**
	 * Stores the Information "String -> ID".
	 */
	private final Map<String, Long> inverseStringValues;
	
	/**
	 * A map consisting of link-targets (keys) and collections of tuples with the targets corresponding links and their
	 * owners (values).
	 */
	private final Map<Object, Collection<Tuple<UnidirectionalLink, UserObject>>> targetMappingToUnidirectionalLinksWithOwners;
	
	/**
	 * A map consisting of link-owners (keys) and collections of tuples with the owners corresponding links and their
	 * targets (values).
	 */
	private final Map<UserObject, Collection<Tuple<UnidirectionalLink, Object>>> ownerMappingToUnidirectionalLinksWithTargets;
	
	/**
	 * Stores the Information "Owner (of the MapLink) -> MapLink". The MapLink contains information about the
	 * MapAssociation, the Owner, the Target and the Key.
	 */
	private final HashMap<UserObject, Collection<MapLink>> ownerMappingToMapLinks;
	
	/**
	 * Stores the Information "Target (in a MapLink) -> MapLink". The MapLink contains information about the
	 * MapAssociation, the Owner, the Target and the Key.
	 */
	private final HashMap<Object, Collection<MapLink>> targetMappingToMapLinks;
	
	/**
	 * Stores the Information "Association -> Collection of StringValue/UserObject". This is important to improve the
	 * "find"-Method.
	 */
	private final HashMap<UnidirectionalAssociation, Collection<Tuple<UserObject, StringValue>>> associationToStringValue;
	
	/**
	 * Stores the Information "Association -> Collection of IntegerValue/UserObject". This is important to improve the
	 * "find"-Method.
	 */
	private final HashMap<UnidirectionalAssociation, Collection<Tuple<UserObject, IntegerValue>>> associationToIntegerValue;
	
	/**
	 * Constructor for instances of NoDatabaseObjectFacadeImplementation.
	 * 
	 * @param classFacade
	 *            is needed to perform consistency checks in regard to the model layer of the meta-model. It must have
	 *            been initialized to function correctly within this ObjectFacade.
	 */
	public NoDatabaseObjectFacadeImplementation(final ClassFacade classFacade) {
		super();
		IntegerValue.setObjectFacade(this);
		StringValue.setObjectFacade(this);
		this.classFacade = classFacade;
		this.nextId = NoDatabaseObjectFacadeImplementation.INITIAL_NEXT_ID;
		this.userTypeMappingToUserObjects = new HashMap<>();
		this.integerValues = new HashMap<>();
		this.stringValues = new HashMap<>();
		this.inverseIntegerValues = new HashMap<>();
		this.inverseStringValues = new HashMap<>();
		this.idToUserObject = new HashMap<>();
		this.targetMappingToUnidirectionalLinksWithOwners = new HashMap<>();
		this.ownerMappingToUnidirectionalLinksWithTargets = new HashMap<>();
		this.ownerMappingToMapLinks = new HashMap<>();
		this.targetMappingToMapLinks = new HashMap<>();
		this.associationToStringValue = new HashMap<>();
		this.associationToIntegerValue = new HashMap<>();
	}
	
	@Override
	public void clear() throws PersistenceException {
		this.nextId = NoDatabaseObjectFacadeImplementation.INITIAL_NEXT_ID;
		this.userTypeMappingToUserObjects.clear();
		this.integerValues.clear();
		this.stringValues.clear();
		this.inverseIntegerValues.clear();
		this.inverseStringValues.clear();
		this.idToUserObject.clear();
		this.ownerMappingToUnidirectionalLinksWithTargets.clear();
		this.targetMappingToUnidirectionalLinksWithOwners.clear();
		this.ownerMappingToMapLinks.clear();
		this.targetMappingToMapLinks.clear();
		this.associationToStringValue.clear();
		this.associationToIntegerValue.clear();
	}
	
	@Override
	public Collection<UserObject> find(final Association association, final String searchPatternOracle, final Date date)
			throws PersistenceException {
		// Replacing oracle wildcards against standard (perl?) regex wildcards
		// TODO: Currently only replacing % against * -- Take a closer look to the complete oracle LIKE
		// syntax and translate to corresponding standard regular expression wildcards.
		
		final String searchPatternRegex = searchPatternOracle.replaceAll("\\*", "\\\\*").replaceAll("%+", ".*");
		final Pattern searchPattern = Pattern.compile(searchPatternRegex);
		
		final HashSet<UserObject> result = new HashSet<>();
		if (!this.associationToStringValue.containsKey(association)) {
			return result;
		}
		
		final Iterator<Tuple<UserObject, StringValue>> it = this.associationToStringValue.get(association).iterator();
		while (it.hasNext()) {
			final Tuple<UserObject, StringValue> current = it.next();
			final String target = current.getSecond().getValue();
			
			final Matcher targetSearchMatcher = searchPattern.matcher(target);
			if (targetSearchMatcher.matches()) {
				result.add(current.getFirst());
			}
		}
		return result;
	}
	
	@Override
	public Collection<UserObject> find(final Association association, final BigInteger integer, final Date date)
			throws PersistenceException {
		final HashSet<UserObject> result = new HashSet<>();
		if (!this.associationToIntegerValue.containsKey(association)) {
			return result;
		}
		
		final Iterator<Tuple<UserObject, IntegerValue>> it = this.associationToIntegerValue.get(association).iterator();
		while (it.hasNext()) {
			final Tuple<UserObject, IntegerValue> current = it.next();
			final BigInteger target = current.getSecond().getValue();
			if (integer.equals(target)) {
				result.add(current.getFirst());
			}
		}
		return result;
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, Object>> get(final UserObject owner,
			final UnidirectionalAssociation association,
			final Date date) throws PersistenceException {
		return this.get(owner, association);
	}
	
	@Override
	public Collection<Object> get(final UserObject owner,
			final MapAssociation association,
			final String key,
			final Date date) throws PersistenceException {
		final Collection<Object> result = new LinkedList<>();
		if (!this.ownerMappingToMapLinks.containsKey(owner)) {
			return result;
		}
		final Iterator<MapLink> it = this.ownerMappingToMapLinks.get(owner).iterator();
		while (it.hasNext()) {
			final MapLink currentMapLink = it.next();
			if (currentMapLink.getInstanceOf().equals(association) && currentMapLink.getKey() instanceof StringValue) {
				if (((StringValue) currentMapLink.getKey()).getValue().equals(key)) {
					result.add(currentMapLink.getTarget());
				}
			}
		}
		if (result.isEmpty()) {
			throw new ObjectNotFoundException();
		}
		return result;
	}
	
	@Override
	public Collection<Object> get(final UserObject owner,
			final MapAssociation association,
			final BigInteger key,
			final Date date) throws PersistenceException {
		final Collection<Object> result = new LinkedList<>();
		if (!this.ownerMappingToMapLinks.containsKey(owner)) {
			return result;
		}
		final Iterator<MapLink> it = this.ownerMappingToMapLinks.get(owner).iterator();
		while (it.hasNext()) {
			final MapLink currentMapLink = it.next();
			if (currentMapLink.getInstanceOf().equals(association) && currentMapLink.getKey() instanceof IntegerValue) {
				if (((IntegerValue) currentMapLink.getKey()).getValue().equals(key)) {
					result.add(currentMapLink.getTarget());
				}
			}
		}
		if (result.isEmpty()) {
			throw new ObjectNotFoundException();
		}
		return result;
	}
	
	@Override
	public Collection<Object> get(final UserObject owner,
			final MapAssociation association,
			final UserObject key,
			final Date date) throws PersistenceException {
		if (key == null) {
			throw new NotValidInputException("");
		}
		final Collection<Object> result = new LinkedList<>();
		if (!this.ownerMappingToMapLinks.containsKey(owner)) {
			return result;
		}
		final Iterator<MapLink> it = this.ownerMappingToMapLinks.get(owner).iterator();
		while (it.hasNext()) {
			final MapLink currentMapLink = it.next();
			if (currentMapLink.getInstanceOf().equals(association) && currentMapLink.getKey() instanceof UserObject) {
				if (currentMapLink.getKey().equals(key)) {
					result.add(currentMapLink.getTarget());
				}
			}
		}
		if (result.isEmpty()) {
			throw new ObjectNotFoundException();
		}
		return result;
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, UserObject>> inverseGet(final UserObject target,
			final UnidirectionalAssociation association,
			final Date date) throws PersistenceException {
		return this.inverseGet(target, association);
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final UserObject target,
			final MapAssociation association,
			final Date date) throws PersistenceException {
		final Collection<Tuple<UserObject, Object>> result = new LinkedList<>();
		if (!this.targetMappingToMapLinks.containsKey(target)) {
			return result;
		}
		final Iterator<MapLink> it = this.targetMappingToMapLinks.get(target).iterator();
		while (it.hasNext()) {
			final MapLink currentMapLink = it.next();
			if (currentMapLink.getInstanceOf().equals(association)) {
				result.add(new Tuple<>(currentMapLink.getOwner(), currentMapLink.getKey()));
			}
		}
		return result;
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final String target,
			final MapAssociation association,
			final Date date) throws PersistenceException {
		final Collection<Tuple<UserObject, Object>> result = new LinkedList<>();
		final Iterator<Object> itMain = this.targetMappingToMapLinks.keySet().iterator();
		while (itMain.hasNext()) {
			final Object ob = itMain.next();
			if (!(ob instanceof StringValue)) {
				continue;
			}
			final StringValue val = (StringValue) ob;
			if (!val.getValue().equals(target)) {
				continue;
			}
			final Iterator<MapLink> it = this.targetMappingToMapLinks.get(val).iterator();
			while (it.hasNext()) {
				final MapLink currentMapLink = it.next();
				if (currentMapLink.getInstanceOf().equals(association)) {
					result.add(new Tuple<>(currentMapLink.getOwner(), currentMapLink.getKey()));
				}
			}
		}
		return result;
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final BigInteger target,
			final MapAssociation association,
			final Date date) throws PersistenceException {
		final Collection<Tuple<UserObject, Object>> result = new LinkedList<>();
		final Iterator<Object> itMain = this.targetMappingToMapLinks.keySet().iterator();
		while (itMain.hasNext()) {
			final Object ob = itMain.next();
			if (!(ob instanceof IntegerValue)) {
				continue;
			}
			final IntegerValue val = (IntegerValue) ob;
			if (!val.getValue().equals(target)) {
				continue;
			}
			final Iterator<MapLink> it = this.targetMappingToMapLinks.get(val).iterator();
			while (it.hasNext()) {
				final MapLink currentMapLink = it.next();
				if (currentMapLink.getInstanceOf().equals(association)) {
					result.add(new Tuple<>(currentMapLink.getOwner(), currentMapLink.getKey()));
				}
			}
		}
		return result;
	}
	
	@Override
	public Collection<UserObject> find(final Association assoc, final String str, final Transaction trans)
			throws PersistenceException {
		return this.find(assoc, str, new Date());
	}
	
	@Override
	public Collection<UserObject> find(final Association association,
			final BigInteger integer,
			final Transaction transaction) throws PersistenceException {
		return this.find(association, integer, new Date());
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, Object>> get(final UserObject owner,
			final UnidirectionalAssociation association,
			final Transaction transaction) throws PersistenceException {
		return this.get(owner, association);
	}
	
	/**
	 * Provides a collection of Tuples of Links and Objects. The Links are instances of the UnidirectionalAssociation
	 * association. And the Objects are the owner of the Links. type(owner) has to be is sub type of the owner of the
	 * UnidirectionalAssociation association.
	 * 
	 * @param owner
	 *            the owner of the links.
	 * @param association
	 *            the type of the links.
	 * @return a list with tuples consisting the links and their targets.
	 */
	private Collection<Tuple<UnidirectionalLink, Object>> get(final UserObject owner,
			final UnidirectionalAssociation association) {
		final Collection<Tuple<UnidirectionalLink, Object>> result = new LinkedList<>();
		if (!this.ownerMappingToUnidirectionalLinksWithTargets.containsKey(owner)) {
			return result;
		}
		final Iterator<Tuple<UnidirectionalLink, Object>> allLinksForOwner =
				this.ownerMappingToUnidirectionalLinksWithTargets.get(owner).iterator();
		while (allLinksForOwner.hasNext()) {
			final Tuple<UnidirectionalLink, Object> currentTuple = allLinksForOwner.next();
			if (currentTuple.getFirst().getInstanceOf().equals(association)) {
				result.add(currentTuple);
			}
		}
		return result;
	}
	
	@Override
	public Collection<Tuple<MapLink, Object>> get(final UserObject owner,
			final MapAssociation association,
			final String key,
			final Transaction transaction) throws PersistenceException {
		final Collection<Tuple<MapLink, Object>> result = new HashSet<>();
		if (!this.ownerMappingToMapLinks.containsKey(owner)) {
			throw new ObjectNotFoundException();
		}
		final Iterator<MapLink> it = this.ownerMappingToMapLinks.get(owner).iterator();
		while (it.hasNext()) {
			final MapLink currentMapLink = it.next();
			if (currentMapLink.getInstanceOf().equals(association)) {
				if (((StringValue) currentMapLink.getKey()).getValue().equals(key)) {
					result.add(new Tuple<>(currentMapLink, currentMapLink.getTarget()));
				}
			}
		}
		if (result.isEmpty()) {
			throw new ObjectNotFoundException();
		}
		return result;
	}
	
	@Override
	public Collection<Tuple<MapLink, Object>> get(final UserObject owner,
			final MapAssociation association,
			final BigInteger key,
			final Transaction transaction) throws PersistenceException {
		final Collection<Tuple<MapLink, Object>> result = new HashSet<>();
		if (!this.ownerMappingToMapLinks.containsKey(owner)) {
			return result;
		}
		final Iterator<MapLink> it = this.ownerMappingToMapLinks.get(owner).iterator();
		while (it.hasNext()) {
			final MapLink currentMapLink = it.next();
			if (currentMapLink.getInstanceOf().equals(association)) {
				if (((IntegerValue) currentMapLink.getKey()).getValue().equals(key)) {
					result.add(new Tuple<>(currentMapLink, currentMapLink.getTarget()));
				}
			}
		}
		return result;
	}
	
	@Override
	public Collection<Tuple<MapLink, Object>> get(final UserObject owner,
			final MapAssociation association,
			final UserObject key,
			final Transaction transaction) throws PersistenceException {
		if (key == null) {
			throw new NotValidInputException("");
		}
		final Collection<Tuple<MapLink, Object>> result = new HashSet<>();
		if (!this.ownerMappingToMapLinks.containsKey(owner)) {
			return result;
		}
		final Iterator<MapLink> it = this.ownerMappingToMapLinks.get(owner).iterator();
		while (it.hasNext()) {
			final MapLink currentMapLink = it.next();
			if (currentMapLink.getInstanceOf().equals(association)) {
				if (currentMapLink.getKey().equals(key)) {
					result.add(new Tuple<>(currentMapLink, currentMapLink.getTarget()));
				}
			}
		}
		return result;
	}
	
	@Override
	public Collection<Tuple<UnidirectionalLink, UserObject>> inverseGet(final UserObject target,
			final UnidirectionalAssociation association,
			final Transaction transaction) throws PersistenceException {
		return this.inverseGet(target, association);
	}
	
	/**
	 * Provides a collection of Tuples of Links and User Objects. The Links are instances of the
	 * UnidirectionalAssociation association. And the User Objects are the targets of the Links. type(target) has to be
	 * is sub type of the target of the UnidirectionalAssociation association.
	 * 
	 * @param target
	 *            the target of the links.
	 * @param association
	 *            the type of the links.
	 * @return tuples with all links of type association; with the target target as first component and the links owner
	 *         as second component.
	 */
	private Collection<Tuple<UnidirectionalLink, UserObject>> inverseGet(final UserObject target,
			final UnidirectionalAssociation association) {
		final Collection<Tuple<UnidirectionalLink, UserObject>> result = new LinkedList<>();
		if (!this.targetMappingToUnidirectionalLinksWithOwners.containsKey(target)) {
			return result;
		}
		final Iterator<Tuple<UnidirectionalLink, UserObject>> allLinksForTarget =
				this.targetMappingToUnidirectionalLinksWithOwners.get(target).iterator();
		while (allLinksForTarget.hasNext()) {
			final Tuple<UnidirectionalLink, UserObject> currentTuple = allLinksForTarget.next();
			if (currentTuple.getFirst().getInstanceOf().equals(association)) {
				result.add(currentTuple);
			}
		}
		return result;
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final UserObject target,
			final MapAssociation association,
			final Transaction transaction) throws PersistenceException {
		return this.inverseGet(target, association, new Date());
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final String target,
			final MapAssociation association,
			final Transaction transaction) throws PersistenceException {
		return this.inverseGet(target, association, new Date());
	}
	
	@Override
	public Collection<Tuple<UserObject, Object>> inverseGet(final BigInteger target,
			final MapAssociation association,
			final Transaction transaction) throws PersistenceException {
		return this.inverseGet(target, association, new Date());
	}
	
	@Override
	public boolean isInConflict(final Transaction before, final Transaction after) throws PersistenceException {
		// We do not support transactions!
		return false;
	}
	
	@Override
	public boolean isOpenTransaction(final Transaction transaction) throws PersistenceException {
		// We do not support transactions!
		return true;
	}
	
	@Override
	public UserObject create(final UserType type, final Transaction transaction) throws PersistenceException {
		if (type.isAbs()) {
			throw new NotInstantiatableException(new SQLException());
		}
		final UserObject result = UserObject.init(this.getAndIncrementNextId(), type);
		this.idToUserObject.put(result.getId(), result);
		this.mapUserTypeToEmptyListIfNotExists(type);
		this.userTypeMappingToUserObjects.get(type).add(result);
		return result;
	}
	
	@Override
	public void delete(final UserObject object, final Transaction transaction) throws PersistenceException {
		// ID -> UserObject
		this.idToUserObject.remove(object.getId());
		
		// UserType -> UserObject
		this.userTypeMappingToUserObjects.get(object.getInstanceOf()).remove(object);
		
		// Owner of a Link
		this.ownerMappingToUnidirectionalLinksWithTargets.remove(object);
		
		// Target of a Link
		this.targetMappingToUnidirectionalLinksWithOwners.remove(object);
		
		// Owner of a MapLink
		this.ownerMappingToMapLinks.remove(object);
		
		// Target of a MapLink
		this.targetMappingToMapLinks.remove(object);
		
		this.deleteFromFindMap(object.getInstanceOf(), object.getId());
	}
	
	private void deleteFromFindMap(final Type type, final long id) throws PersistenceException {
		// TODO: Remove from associationToStringValue
		// TODO: Remove from associationToIntegerValue
	}
	
	@Override
	public UnidirectionalLink set(final UserObject owner,
			final UnidirectionalAssociation association,
			final UserObject target,
			final Transaction transaction) throws PersistenceException {
		this.checkAssociationUnidirectional(association, owner.getInstanceOf(), target.getInstanceOf());
		return this.set(owner, association, target);
	}
	
	@Override
	public UnidirectionalLink set(final UserObject owner,
			final UnidirectionalAssociation association,
			final BigInteger target,
			final Transaction transaction) throws PersistenceException {
		this.checkAssociationUnidirectional(association, owner.getInstanceOf(), IntegerType.getInstance());
		return this.set(owner, association, new IntegerValue(target));
	}
	
	@Override
	public UnidirectionalLink set(final UserObject owner,
			final UnidirectionalAssociation association,
			final String target,
			final Transaction transaction) throws PersistenceException {
		this.checkAssociationUnidirectional(association, owner.getInstanceOf(), StringType.getInstance());
		return this.set(owner, association, new StringValue(target));
	}
	
	/**
	 * Creates a new UnidirectionalLink of type association consisting of an owner and its target.
	 * 
	 * @param owner
	 *            the owner of the link.
	 * @param association
	 *            the type of the link.
	 * @param target
	 *            the target of the link.
	 * @return a new UnidirectionalLink of type association consisting of an owner and its target.
	 * @throws PersistenceException
	 *             if the type of owner is not a descendant of the associations owner-type and that the type of the
	 *             target is not a descendant of the associations target-type. Throws an exception if the check is
	 *             negative.
	 * 
	 */
	private UnidirectionalLink set(final UserObject owner,
			final UnidirectionalAssociation association,
			final Object target) throws PersistenceException {
		this.throwInvalidLinkExceptionIfWrongTypeForLink(owner, association, target);
		final UnidirectionalLink result =
				new UnidirectionalLink(this.getAndIncrementNextId(), owner, target, association);
		final Tuple<UnidirectionalLink, Object> tupleForOwner = new Tuple<>(result, target);
		if (!this.ownerMappingToUnidirectionalLinksWithTargets.containsKey(owner)) {
			this.ownerMappingToUnidirectionalLinksWithTargets.put(
					owner,
					new LinkedList<Tuple<UnidirectionalLink, Object>>());
		}
		final Tuple<UnidirectionalLink, UserObject> tupleForTarget = new Tuple<>(result, owner);
		if (!this.targetMappingToUnidirectionalLinksWithOwners.containsKey(target)) {
			this.targetMappingToUnidirectionalLinksWithOwners.put(
					target,
					new LinkedList<Tuple<UnidirectionalLink, UserObject>>());
		}
		if (association.isUnique()) {
			this.deleteAllOwnerUnidirectionalLinksWhichAreInstanceOf(owner, association);
			this.deleteAllTargetUnidirectionalLinksWhichAreInstanceOf(target, association);
		}
		this.ownerMappingToUnidirectionalLinksWithTargets.get(owner).add(tupleForOwner);
		this.targetMappingToUnidirectionalLinksWithOwners.get(target).add(tupleForTarget);
		
		// Used to improve the performance of "find" (String)
		if (target instanceof StringValue) {
			if (!this.associationToStringValue.containsKey(association)) {
				this.associationToStringValue.put(association, new LinkedList<Tuple<UserObject, StringValue>>());
			}
			this.associationToStringValue.get(association).add(new Tuple<>(owner, (StringValue) target));
		}
		if (target instanceof IntegerValue) {
			if (!this.associationToIntegerValue.containsKey(association)) {
				this.associationToIntegerValue.put(association, new LinkedList<Tuple<UserObject, IntegerValue>>());
			}
			this.associationToIntegerValue.get(association).add(new Tuple<>(owner, (IntegerValue) target));
		}
		
		return result;
	}
	
	/**
	 * Deletes all links with the target target which are type of association from the
	 * targetMappingToUnidirectionalLinksWithOwners-map.
	 * 
	 * @param target
	 *            the target of those links
	 * @param association
	 *            the type of those links
	 */
	private void deleteAllTargetUnidirectionalLinksWhichAreInstanceOf(final Object target,
			final UnidirectionalAssociation association) {
		if (this.targetMappingToUnidirectionalLinksWithOwners.containsKey(target)) {
			final Iterator<Tuple<UnidirectionalLink, UserObject>> allLinks =
					this.targetMappingToUnidirectionalLinksWithOwners.get(target).iterator();
			while (allLinks.hasNext()) {
				if (allLinks.next().getFirst().getInstanceOf().equals(association)) {
					allLinks.remove();
				}
			}
		}
	}
	
	/**
	 * Deletes all links with the owner owner which are type of association from the
	 * ownerMappingToUnidirectionalLinksWithTargets-map.
	 * 
	 * @param owner
	 *            the owner of those links
	 * @param association
	 *            the type of those links
	 */
	private void deleteAllOwnerUnidirectionalLinksWhichAreInstanceOf(final UserObject owner,
			final UnidirectionalAssociation association) {
		if (this.ownerMappingToUnidirectionalLinksWithTargets.containsKey(owner)) {
			final Iterator<Tuple<UnidirectionalLink, Object>> allLinks =
					this.ownerMappingToUnidirectionalLinksWithTargets.get(owner).iterator();
			while (allLinks.hasNext()) {
				if (allLinks.next().getFirst().getInstanceOf().equals(association)) {
					allLinks.remove();
				}
			}
		}
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final UserObject target,
			final String key,
			final Transaction transaction) throws PersistenceException {
		return this.putInternal(owner, association, target, new StringValue(key));
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final BigInteger target,
			final String key,
			final Transaction transaction) throws PersistenceException {
		return this.putInternal(owner, association, new IntegerValue(target), new StringValue(key));
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final String target,
			final String key,
			final Transaction transaction) throws PersistenceException {
		return this.putInternal(owner, association, new StringValue(target), new StringValue(key));
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final UserObject target,
			final BigInteger key,
			final Transaction transaction) throws PersistenceException {
		return this.putInternal(owner, association, target, new IntegerValue(key));
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final BigInteger target,
			final BigInteger key,
			final Transaction transaction) throws PersistenceException {
		return this.putInternal(owner, association, new IntegerValue(target), new IntegerValue(key));
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final String target,
			final BigInteger key,
			final Transaction transaction) throws PersistenceException {
		return this.putInternal(owner, association, new StringValue(target), new IntegerValue(key));
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final UserObject target,
			final UserObject key,
			final Transaction transaction) throws PersistenceException {
		return this.putInternal(owner, association, target, key);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final BigInteger target,
			final UserObject key,
			final Transaction transaction) throws PersistenceException {
		return this.putInternal(owner, association, new IntegerValue(target), key);
	}
	
	@Override
	public MapLink put(final UserObject owner,
			final MapAssociation association,
			final String target,
			final UserObject key,
			final Transaction transaction) throws PersistenceException {
		return this.putInternal(owner, association, new StringValue(target), key);
	}
	
	private MapLink putInternal(final UserObject owner,
			final MapAssociation association,
			final Object target,
			final Object key) throws PersistenceException {
		this.checkAssociationMap(association, owner.getInstanceOf(), target.getInstanceOf(), key.getInstanceOf());
		
		if (!this.ownerMappingToMapLinks.containsKey(owner)) {
			this.ownerMappingToMapLinks.put(owner, new LinkedList<MapLink>());
		}
		if (!this.targetMappingToMapLinks.containsKey(target)) {
			this.targetMappingToMapLinks.put(target, new LinkedList<MapLink>());
		}
		
		// TODO Check "isUnique()"? IMPORTANT!!!
		
		final MapLink result = new MapLink(this.getAndIncrementNextId(), owner, target, key, association);
		
		this.ownerMappingToMapLinks.get(owner).add(result);
		this.targetMappingToMapLinks.get(target).add(result);
		
		return result;
	}
	
	@Override
	public void unset(final Link link, final Transaction transaction) throws PersistenceException {
		// We do not support transactions!
	}
	
	@Override
	public void commit(final Transaction transaction) throws PersistenceException {
		// We do not support transactions!
	}
	
	@Override
	public void rollback(final Transaction transaction) throws PersistenceException {
		// We do not support transactions!
	}
	
	@Override
	public void savePoint(final Transaction transaction) throws PersistenceException {
		// We do not support transactions!
	}
	
	@Override
	public void rollbackToSavePoint(final Transaction transaction) throws PersistenceException {
		// We do not support transactions!
	}
	
	@Override
	public BigInteger getIntForId(final long id) throws PersistenceException {
		if (!this.integerValues.containsKey(id)) {
			throw new BaseTypeNotFoundException();
		}
		return this.integerValues.get(id);
	}
	
	@Override
	public String getStringForId(final long id) throws PersistenceException {
		if (!this.stringValues.containsKey(id)) {
			throw new BaseTypeNotFoundException();
		}
		return this.stringValues.get(id);
	}
	
	@Override
	public long getIdForString(final String string) throws PersistenceException {
		if (!this.inverseStringValues.containsKey(string)) {
			final long newId = this.getAndIncrementNextId();
			this.stringValues.put(newId, string);
			this.inverseStringValues.put(string, newId);
		}
		return this.inverseStringValues.get(string);
	}
	
	@Override
	public long getIdForInteger(final BigInteger integer) throws PersistenceException {
		if (!this.inverseIntegerValues.containsKey(integer)) {
			final long newId = this.getAndIncrementNextId();
			this.integerValues.put(newId, integer);
			this.inverseIntegerValues.put(integer, newId);
		}
		return this.inverseIntegerValues.get(integer);
	}
	
	@Override
	public Transaction provideAdhocTransaction() throws PersistenceException {
		return new AdhocTransaction(this.getAndIncrementNextId());
	}
	
	@Override
	public TypeManager getTypeManager() {
		return TypeManagerImplementation.getInstance();
	}
	
	@Override
	public Collection<UserObject> findAllObjects(final UserType type, final Transaction transaction)
			throws PersistenceException {
		this.mapUserTypeToEmptyListIfNotExists(type);
		return this.userTypeMappingToUserObjects.get(type);
	}
	
	@Override
	public UserObject checkUserObjectOut(final long id, final Transaction transaction) throws PersistenceException {
		if (!this.idToUserObject.containsKey(id)) {
			throw new ObjectNotFoundException();
		}
		return this.idToUserObject.get(id);
	}
	
	/**
	 * Adds the mapping from a given type to an empty collection if the type doesn't exists as key in the
	 * userTypeMappingToUserObjects-collection.
	 * 
	 * @param type
	 *            the given type.
	 */
	private void mapUserTypeToEmptyListIfNotExists(final UserType type) {
		if (!this.userTypeMappingToUserObjects.containsKey(type)) {
			this.userTypeMappingToUserObjects.put(type, new LinkedList<UserObject>());
		}
	}
	
	/**
	 * @return Returns the next free id.
	 */
	private long getAndIncrementNextId() {
		final long result = this.nextId;
		this.nextId++;
		return result;
	}
	
	/**
	 * Checks if the given components of a new UnidirectionalLink will match the association.
	 * 
	 * @param association
	 *            The given UnidirectionalAssociation where the user wants to create an instance
	 * @param owner
	 *            The new owner of this Link instance
	 * @param target
	 *            The new target of this Link instance
	 * @throws PersistenceException
	 *             A InvalidLinkException will be thrown if the types do not match
	 */
	private void checkAssociationUnidirectional(final UnidirectionalAssociation association,
			final Type owner,
			final Type target) throws PersistenceException {
		if (!association.getOwner().isTheSameAs(owner)
				&& !this.classFacade.isSuperClassTo(association.getOwner(), owner)) {
			throw new InvalidLinkException(new SQLException());
		}
		
		if (!association.getTarget().isTheSameAs(target)
				&& !this.classFacade.isSuperClassTo(association.getTarget(), target)) {
			throw new InvalidLinkException(new SQLException());
		}
	}
	
	/**
	 * Checks if the given components of a new MapLink will match the association.
	 * 
	 * @param association
	 *            The given MapAssociation where the user wants to create an MapLink instance
	 * @param owner
	 *            The new owner of this MapLink instance
	 * @param target
	 *            The new target of this MapLink instance
	 * @param key
	 *            The new key of this MapLink instance
	 * @throws PersistenceException
	 *             A InvalidLinkException will be thrown if the types do not match
	 */
	private void checkAssociationMap(final MapAssociation association,
			final Type owner,
			final Type target,
			final Type key) throws PersistenceException {
		if (!association.getOwner().isTheSameAs(owner)
				&& !this.classFacade.isSuperClassTo(association.getOwner(), owner)
				&& !this.classFacade.isSuperClassTo(owner, association.getOwner())) {
			throw new InvalidLinkException(null);
		}
		
		if (!association.getTarget().isTheSameAs(target)
				&& !this.classFacade.isSuperClassTo(association.getTarget(), target)
				&& !this.classFacade.isSuperClassTo(target, association.getTarget())) {
			throw new InvalidLinkException(new SQLException());
		}
		
		if (!association.getKeyType().isTheSameAs(key)
				&& !this.classFacade.isSuperClassTo(association.getKeyType(), key)
				&& !this.classFacade.isSuperClassTo(key, association.getKeyType())) {
			throw new InvalidLinkException(new SQLException());
		}
	}
	
	/**
	 * Checks if the type of owner is a descendant of the associations owner-type and that the type of the target is a
	 * descendant of the associations target-type. Throws an exception if the check is negative.
	 * 
	 * @param owner
	 *            the potential owner of a link.
	 * @param association
	 *            the type of a potential link consisting of owner and an instance of targetType.
	 * @param target
	 *            the potential target of a link.
	 * @throws PersistenceException
	 *             if the check mentioned above is negative.
	 */
	private void throwInvalidLinkExceptionIfWrongTypeForLink(final UserObject owner,
			final UnidirectionalAssociation association,
			final Object target) throws PersistenceException {
		if (!this.classFacade.isSuperClassTo(association.getTarget(), target.getInstanceOf())
				|| !this.classFacade.isSuperClassTo(association.getOwner(), owner.getInstanceOf())) {
			throw new InvalidLinkException(new SQLException());
		}
	}
	
}
