package de.fhdw.wtf.persistence.facade;

import java.util.Collection;
import java.util.List;

import de.fhdw.wtf.persistence.exception.ClassFacadeUninitializedException;
import de.fhdw.wtf.persistence.exception.DuplicateUserTypeException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.meta.Association;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.persistence.meta.UserType;

/**
 * A facade to access objects in the database which represent types and associations.
 * 
 */
public interface ClassFacade {
	
	/**
	 * Constructor for a new User Type.
	 * 
	 * @param name
	 *            The name of the {@link de.fhdw.wtf.persistence.meta.UserType User Type}.
	 * @param abs
	 *            A boolean flag indicating whether this {@link de.fhdw.wtf.persistence.meta.UserType User Type} is
	 *            abstract or not
	 * @param transaction
	 *            A boolean flag indicating whether this {@link de.fhdw.wtf.persistence.meta.UserType User Type} is also
	 *            an Transaction or not.
	 * @return Provides a new {@link de.fhdw.wtf.persistence.meta.UserType} with the given attributes and the next
	 *         possible id.
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 * @throws DuplicateUserTypeException
	 *             can be thrown if the attempt is made to create an {@link de.fhdw.wtf.persistence.meta.UserType} with
	 *             an already existing name.
	 */
	UserType createUserType(String name, boolean abs, boolean transaction) throws PersistenceException,
			DuplicateUserTypeException;
	
	/**
	 * Constructor for a new {@link de.fhdw.wtf.persistence.meta.UnidirectionalAssociation}.
	 * 
	 * @param name
	 *            The name of the new {@link de.fhdw.wtf.persistence.meta.UnidirectionalAssociation}.
	 * @param essential
	 *            A boolean flag indicating whether there is an existing object necessary.
	 * @param unique
	 *            A boolean flag indicating whether the association is set-valued.
	 * @param owner
	 *            The owning side of the {@link de.fhdw.wtf.persistence.meta.UnidirectionalAssociation}
	 * @param target
	 *            The targeting side of the {@link de.fhdw.wtf.persistence.meta.UnidirectionalAssociation}.
	 * @return Provides a new {@link de.fhdw.wtf.persistence.meta.UnidirectionalAssociation} with the given attributes
	 *         and the next possible ID.
	 * @throws PersistenceException
	 *             A {@link de.fhdw.wtf.persistence.exception.PersistenceException persistence exception} is thrown if
	 *             any error contacting the database occurs.
	 */
	UnidirectionalAssociation createUnidirectionalAssociation(String name,
			boolean essential,
			boolean unique,
			UserType owner,
			Type target) throws PersistenceException;
	
	/**
	 * Constructor for a new MapAssociation.
	 * 
	 * @param name
	 *            The name of the new MapAssociation.
	 * @param essential
	 *            A boolean flag indicating whether there is an existing object necessary.
	 * @param owner
	 *            The owner side of the MapAssociation.
	 * @param target
	 *            The targeting side of the MapAssociation.
	 * @param keyType
	 *            The map-key type of the MapAssociation, when the MapAssociation is a map.
	 * @return Provides a new MapAssociation with the given attributes and the next possible ID.
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	MapAssociation createMapAssociation(String name, boolean essential, UserType owner, Type target, Type keyType)
			throws PersistenceException;
	
	/**
	 * Creates a specialization-relationship between two user types.
	 * 
	 * @param ancestor
	 *            The Ancestor which means the super type in the specialization
	 * @param descendant
	 *            The descendant which means the sub type in the specialization
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	void createSpecializationBetween(UserType ancestor, Type descendant) throws PersistenceException;
	
	/**
	 * Checks if one Type is Superior to another.
	 * 
	 * @param ancestor
	 *            The given super type.
	 * @param descendant
	 *            The given sub type.
	 * @return Provides true if the ancestor is a supertype to descendant.
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	boolean isSuperClassTo(Type ancestor, Type descendant) throws PersistenceException;
	
	/**
	 * When creating specializations, they will be stored in a normalized manner inside the database, which means that
	 * the Facade can only determine if direct specializations exist. Reflexive and transitive Specializations could not
	 * be detected. This Method creates the reflexive, transitive closure.
	 * 
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	void finalizeSpecialization() throws PersistenceException;
	
	/**
	 * Attention !!! This Method deletes every Type, UnidirectionalAssociation an specialization in the database. This
	 * means it should be used very carefully.
	 * 
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	void clear() throws PersistenceException;
	
	/**
	 * Renaming the given Type discarding the old name.
	 * 
	 * @param typeId
	 *            the id of the Type which should renamed
	 * @param newName
	 *            the new name of the type
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	void renameType(Long typeId, String newName) throws PersistenceException;
	
	/**
	 * Renaming the given Association discarding the old name.
	 * 
	 * @param assoId
	 *            the id of the Association which should renamed
	 * @param newName
	 *            the new name of the association
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	void renameAssociation(Long assoId, String newName) throws PersistenceException;
	
	/**
	 * Removes this association permanently. Attention: This function must only be called on associations that have no
	 * links!
	 * 
	 * @param associationId
	 *            to be removed
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	void deleteAssociation(Long associationId) throws PersistenceException;
	
	/**
	 * Removes this user type and corresponding specializations permanently. Attention: This function must only be
	 * called on types that have no direct objects!
	 * 
	 * @param typeId
	 *            to be removed
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	void deleteUserType(Long typeId) throws PersistenceException;
	
	/**
	 * Updates the instance-of relation of all links of 'association' to a matching new association in
	 * 'newAssociations'.
	 * 
	 * @param associationId
	 *            to be removed
	 * @param newAssociationIds
	 *            new associations that links might have as new 'instance-of'
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	void updateLinksToNewAssociation(Long associationId, Collection<Long> newAssociationIds)
			throws PersistenceException;
	
	/**
	 * Updates all existing links and adds objects and links to perform the move from the old associations to the new
	 * associations, that have moved to the new type.
	 * 
	 * @param oldAssoIds
	 * @param newAsso
	 * @param newType
	 * @param newAssoIds
	 * @throws PersistenceException
	 */
	void moveLinksAndCreateObjects(List<Long> oldAssoIds, Association newAsso, UserType newType, List<Long> newAssoIds)
			throws PersistenceException;
	
	/**
	 * 
	 * This Method creates the Base Types in the Database and makes it possible to fill the model layer with content.
	 * This Method has to be called if there are no Model Items present in the Database and you want to create a new
	 * pool of data.
	 * 
	 * @throws PersistenceException
	 *             A persistence Exception is thrown if any Error contacting the database occurs.
	 */
	void initialize() throws PersistenceException;
	
	/**
	 * This method initializes the classfacade when there is already a model present in the database and the persistence
	 * facade has to be set up for Runtime. It reads the Type Information from the Database and store it into the
	 * TypeManager.
	 * 
	 * @throws PersistenceException
	 *             A persistence exception is thrown if there are any errors contacting the database.
	 */
	void initializeForRuntime() throws PersistenceException;
	
	/**
	 * A Method to determine if the classfacade has been initialized with at least Base type Informations. It is
	 * necessary that this methods provides true before executing other Database operations.
	 * 
	 * @return Provides true if initialize or initializeWithExistingData had been invoked.
	 */
	boolean hasBeenInitialized();
	
	/**
	 * Provides the Type Manager, which represents a Dictionary for all created Classes and Associations, so it can be
	 * used to lookup Classes. This Method could only be used if initialize or initializeWithExistingData are invoked.
	 * 
	 * @return The Type Manager, which stores Information about all Types and Associatons
	 * @throws ClassFacadeUninitializedException
	 *             this exception were thrown when the classfacase isn't initialized
	 */
	TypeManager getTypeManager() throws ClassFacadeUninitializedException;
	
}
