package de.fhdw.wtf.persistence.facade;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Properties;

import de.fhdw.wtf.persistence.exception.ClassFacadeUninitializedException;
import de.fhdw.wtf.persistence.exception.NotConnectableException;
import de.fhdw.wtf.persistence.exception.NotConnectedException;
import de.fhdw.wtf.persistence.exception.OracleJDBCDriverMissing;
import de.fhdw.wtf.persistence.exception.OtherSQLException;
import de.fhdw.wtf.persistence.exception.PersistenceException;

/**
 * A Class to access an oracle database with a JDBC Driver.
 * 
 */
public final class OracleDatabaseManager extends DatabaseManager {
	
	private static final String JDBC_SEPERATOR = ":";
	
	private static final String CONNECTION_PREFIX_ORACLE = "jdbc:oracle:thin:@";
	
	private static final String JDBC_ORACLE_DRIVER_NAME = "oracle.jdbc.OracleDriver";
	
	private ClassFacade classFacade;
	
	/**
	 * The Oracle Database Manager is a singleton, therefore one single instance is stored in this static field.
	 */
	private static OracleDatabaseManager instance;
	
	/**
	 * A boolean flag indicating whether a connection to the database is established or not.
	 */
	private boolean isConnected;
	
	/**
	 * The Name of the Oracle Schema, which the Manager will connect to. Usually its the same as the user name.
	 */
	private String schemaName;
	
	/**
	 * The Username of an valid oracle user account, which has resource privilegies and has full access to packages in
	 * the given Schema.
	 */
	private String userName;
	
	/**
	 * The password of the given user. Password will be stored in plain text.
	 */
	private String userPassword;
	
	/**
	 * The SID of the Oracle Database.
	 */
	private String oracleSID;
	
	/**
	 * The name of the host where the oracle database is running.
	 */
	private String hostname;
	
	/**
	 * The port on which the Oracle Net Listener listens.
	 */
	private String port;
	
	/**
	 * The Oracle Database Manager will establish a single connection.
	 */
	private Connection defaultConnection;
	
	/**
	 * Creates a new unconnected Database Manager.
	 */
	private OracleDatabaseManager() {
		this.isConnected = false;
	}
	
	@Override
	public void setConnectionConstantsFromFile(final String fileName) throws IOException {
		final Properties props = new Properties();
		try (final FileInputStream inputStream = new FileInputStream(new File(fileName))) {
			props.load(inputStream);
		}
		this.schemaName = props.getProperty("schema");
		this.userName = props.getProperty("username");
		this.userPassword = props.getProperty("password");
		this.oracleSID = props.getProperty("sid");
		this.hostname = props.getProperty("hostname");
		this.port = props.getProperty("port");
	}
	
	/**
	 * @see de.fhdw.wtf.persistence.facade.DatabaseManager#connect()
	 * @throws OracleJDBCDriverMissing
	 *             if JDBC driver is missing
	 * @throws NotConnectableException
	 *             if an SQL error occurs
	 */
	@Override
	public void connect() throws PersistenceException {
		try {
			Class.forName(JDBC_ORACLE_DRIVER_NAME);
			this.isConnected = true;
			this.defaultConnection =
					DriverManager.getConnection(
							CONNECTION_PREFIX_ORACLE + this.getHostname() + JDBC_SEPERATOR + this.getPort()
									+ JDBC_SEPERATOR + this.getOracleSID(),
							this.getUserName(),
							this.getUserPassword());
		} catch (final ClassNotFoundException e) {
			throw new OracleJDBCDriverMissing();
		} catch (final SQLException e) {
			throw new NotConnectableException(this, e);
		}
	}
	
	/**
	 * This Method provides the default connection object to the Database. This Method will only works if connect has
	 * been successfully executed before otherwise an exception will be thrown.
	 * 
	 * @return Provides the default database connection.
	 * @throws PersistenceException
	 *             This Exception will be thrown if the Manager has not been connected to Orcle before.
	 */
	public Connection getConnection() throws PersistenceException {
		if (this.isConnected) {
			return this.defaultConnection;
		}
		throw new NotConnectedException();
	}
	
	@Override
	public ClassFacade getClassFacade() {
		if (this.classFacade == null) {
			this.classFacade = new OracleClassFacadeImplementation(this);
		}
		return this.classFacade;
	}
	
	@Override
	public ObjectFacade getObjectFacade() throws ClassFacadeUninitializedException {
		return new OracleObjectFacadeImplementation(this, this.getClassFacade().getTypeManager());
	}
	
	@Override
	public void resetDatabase() throws PersistenceException, IOException {
		final OracleDataBasePreparator prep = new OracleDataBasePreparator();
		final boolean tableStructureValid = prep.isTableStructureValid();
		if (tableStructureValid) {
			this.clearTables();
		} else {
			prep.dropWholeSchema();
			prep.createWholeSchema();
		}
		final boolean areProceduresCreated = prep.areProceduresCreated();
		if (!areProceduresCreated) {
			prep.createProcedures();
		}
	}
	
	@Override
	public void clearTables() throws PersistenceException {
		new OracleObjectFacadeImplementation(this, null).clear();
		this.getClassFacade().clear();
	}
	
	/**
	 * Provides the singleton instance of the Oracle Database Manager.
	 * 
	 * @return The Oracle Database manager.
	 */
	public static synchronized OracleDatabaseManager getInstance() {
		if (instance == null) {
			instance = new OracleDatabaseManager();
		}
		return instance;
	}
	
	@Override
	public void disconnect() throws PersistenceException {
		try {
			this.getConnection().close();
		} catch (final SQLException e) {
			throw new OtherSQLException(e);
		}
		this.isConnected = false;
		
	}
	
	@Override
	public boolean isConnected() {
		return this.isConnected;
	}
	
	/**
	 * 
	 * @return Provides the set Schema Name.
	 */
	public String getSchemaName() {
		return this.schemaName;
	}
	
	/**
	 * 
	 * @return Provides the set Username.
	 */
	public String getUserName() {
		return this.userName;
	}
	
	/**
	 * 
	 * @return Provides the set Password.
	 */
	public String getUserPassword() {
		return this.userPassword;
	}
	
	/**
	 * 
	 * @return Provides the Set Oracle SID.
	 */
	public String getOracleSID() {
		return this.oracleSID;
	}
	
	/**
	 * 
	 * @return Provides the set Hostname.
	 */
	public String getHostname() {
		return this.hostname;
	}
	
	/**
	 * 
	 * @return Provides the set port.
	 */
	public String getPort() {
		return this.port;
	}
	
}
