package de.fhdw.wtf.persistence.facade;

import java.io.IOException;

import de.fhdw.wtf.persistence.exception.ClassFacadeUninitializedException;
import de.fhdw.wtf.persistence.exception.PersistenceException;

/**
 * Manages actions on simulated database (called No-Database).
 */
public final class NoDatabaseManager extends DatabaseManager {
	
	/**
	 * The No Database Manager is a singleton, therefore one single instance is stored in this static field.
	 */
	private static NoDatabaseManager instance;
	
	/**
	 * The class facade.
	 */
	private ClassFacade classFacade;
	
	/**
	 * The object facade.
	 */
	private ObjectFacade objectFacade;
	
	/**
	 * It stores the information if the NoDatabase is connected or not to simulate a database connection. We need this
	 * to use "connect()", "disconnect()" and "isConnected()" correctly.
	 */
	private boolean connected = false;
	
	@Override
	public void setConnectionConstantsFromFile(final String fileName) throws IOException {
		// It is empty because we do not need to read some configurations but we have to implement this method to
		// simulate a database
	}
	
	@Override
	public void connect() throws PersistenceException {
		this.connected = true;
	}
	
	@Override
	public ClassFacade getClassFacade() {
		if (this.classFacade == null) {
			this.classFacade = new NoDatabaseClassFacadeImplementation();
		}
		return this.classFacade;
	}
	
	@Override
	public ObjectFacade getObjectFacade() throws ClassFacadeUninitializedException {
		if (!this.getClassFacade().hasBeenInitialized()) {
			throw new ClassFacadeUninitializedException();
		}
		if (this.objectFacade == null) {
			this.objectFacade = new NoDatabaseObjectFacadeImplementation(this.classFacade);
		}
		return this.objectFacade;
	}
	
	@Override
	public void disconnect() throws PersistenceException {
		this.connected = false;
	}
	
	@Override
	public boolean isConnected() {
		return this.connected;
	}
	
	@Override
	public void resetDatabase() throws PersistenceException, IOException {
		this.clearTables();
	}
	
	@Override
	public void clearTables() throws PersistenceException {
		this.getClassFacade().clear();
		this.getObjectFacade().clear();
	}
	
	/**
	 * Provides the singleton instance of the No Database Manager.
	 * 
	 * @return The No Database manager.
	 */
	public static synchronized NoDatabaseManager getInstance() {
		if (instance == null) {
			instance = new NoDatabaseManager();
		}
		return instance;
	}
	
}
