package de.fhdw.wtf.persistence.facade;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;

import de.fhdw.wtf.persistence.exception.TypeOrAssociationNotFoundException;
import de.fhdw.wtf.persistence.meta.Association;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.Specialization;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;

/**
 * A class to represent an implementation of the Type Manager Interface, wich stores information about Types and
 * Associations.
 * 
 */
public final class TypeManagerImplementation implements TypeManager {
	
	/**
	 * The Package separator.
	 */
	private static final String JAVA_PACKAGE_SEPERATOR = ".";
	
	/**
	 * The Types are stored in a Map indexed by their Ids.
	 */
	private final Map<Long, Type> cacheId;
	
	/**
	 * The Types are store in a Map indexed by their Ids.
	 */
	private final Map<Long, Association> associationCacheId;
	
	/**
	 * The cache which stores the connection between the name of the classes and the types. It is used to find a type by
	 * its name.
	 */
	private final Map<String, Type> cacheName;
	
	/**
	 * The cache contains all associations.
	 */
	private final Map<String, Association> cacheAssociationName;
	
	/**
	 * Holds all specializations stored in this TypeManagerImplementation.
	 */
	private final Collection<Specialization> specializations;
	
	// /**
	// * The implementation of the Type Manager is a singleton
	// */
	// private static TypeManagerImplementation instance;
	
	/**
	 * Provides the only instance of a Type Manager Implementation.
	 * 
	 * @return The Type Manager Implementation Object.
	 */
	public static TypeManagerImplementation getInstance() {
		// if (instance == null) {
		// instance = new TypeManagerImplementation();
		// }
		// return instance;
		return new TypeManagerImplementation();
	}
	
	/**
	 * Private constructor, which initializes the fields.
	 */
	private TypeManagerImplementation() {
		super();
		this.cacheId = new HashMap<>();
		this.associationCacheId = new HashMap<>();
		this.cacheName = new HashMap<>();
		this.cacheAssociationName = new HashMap<>();
		this.specializations = new LinkedList<>();
	}
	
	private String prepareName(final String name) {
		return name.replace('>', '.').replace('$', '.');
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see persistence.facade.TyperManager#getTypeForId(long)
	 */
	@Override
	public Type getTypeForId(final long id) throws TypeOrAssociationNotFoundException {
		if (!this.cacheId.containsKey(id)) {
			throw new TypeOrAssociationNotFoundException(id);
		}
		return this.cacheId.get(id);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see persistence.facade.TyperManager#getTypeforName(java.lang.String)
	 */
	@Override
	public Type getTypeforName(final String name) throws TypeOrAssociationNotFoundException {
		if (!this.cacheName.containsKey(this.prepareName(name))) {
			throw new TypeOrAssociationNotFoundException(name);
		}
		return this.cacheName.get(this.prepareName(name));
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see persistence.facade.TyperManager#getAssociationForId(long)
	 */
	@Override
	public UnidirectionalAssociation getUnidirectionalAssociationForId(final long id)
			throws TypeOrAssociationNotFoundException {
		if (!this.associationCacheId.containsKey(id)) {
			throw new TypeOrAssociationNotFoundException(id);
		}
		final Association association = this.associationCacheId.get(id);
		if (!(association instanceof UnidirectionalAssociation)) {
			throw new TypeOrAssociationNotFoundException(id);
		}
		
		return (UnidirectionalAssociation) association;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see persistence.facade.TyperManager#getAssociationForId(long)
	 */
	@Override
	public MapAssociation getMapAssociationForId(final long id) throws TypeOrAssociationNotFoundException {
		if (!this.associationCacheId.containsKey(id)) {
			throw new TypeOrAssociationNotFoundException(id);
		}
		final Association association = this.associationCacheId.get(id);
		if (!(association instanceof MapAssociation)) {
			throw new TypeOrAssociationNotFoundException(id);
		}
		
		return (MapAssociation) association;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see persistence.facade.TyperManager#getAssociationForName(java.lang.String)
	 */
	@Override
	public UnidirectionalAssociation getUnidirectionalAssociationForName(final String name)
			throws TypeOrAssociationNotFoundException {
		if (!this.cacheAssociationName.containsKey(this.prepareName(name))) {
			throw new TypeOrAssociationNotFoundException(name);
		}
		final Association association = this.cacheAssociationName.get(this.prepareName(name));
		if (!(association instanceof UnidirectionalAssociation)) {
			throw new TypeOrAssociationNotFoundException(name);
		}
		
		return (UnidirectionalAssociation) association;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see persistence.facade.TyperManager#getAssociationForName(java.lang.String)
	 */
	@Override
	public MapAssociation getMapAssociationForName(final String name) throws TypeOrAssociationNotFoundException {
		if (!this.cacheAssociationName.containsKey(this.prepareName(name))) {
			throw new TypeOrAssociationNotFoundException(name);
		}
		final Association association = this.cacheAssociationName.get(this.prepareName(name));
		if (!(association instanceof MapAssociation)) {
			throw new TypeOrAssociationNotFoundException(name);
		}
		
		return (MapAssociation) association;
	}
	
	@Override
	public void saveType(final Type type) {
		this.cacheId.put(type.getId(), type);
		final String name = this.prepareName(type.getName());
		this.cacheName.put(name, type);
	}
	
	@Override
	public void saveAssociation(final Association association) {
		this.associationCacheId.put(association.getId(), association);
		final String name =
				this.prepareName(association.getOwner().getName()) + JAVA_PACKAGE_SEPERATOR
						+ this.prepareName(association.getName());
		this.cacheAssociationName.put(name, association);
	}
	
	@Override
	public long getMaximumTypeId() {
		long result = 0;
		final Iterator<Type> types = this.cacheId.values().iterator();
		while (types.hasNext()) {
			final Type type = types.next();
			if (type.getId() > result) {
				result = type.getId();
			}
		}
		return result;
	}
	
	@Override
	public void clear() {
		this.cacheId.clear();
		this.cacheName.clear();
		this.associationCacheId.clear();
		this.cacheAssociationName.clear();
		this.specializations.clear();
	}
	
	@Override
	public void deleteType(final long typeId) throws TypeOrAssociationNotFoundException {
		if (!this.cacheId.containsKey(typeId)) {
			throw new TypeOrAssociationNotFoundException(typeId);
		}
		this.cacheName.remove(this.prepareName(this.cacheId.remove(typeId).getName()));
	}
	
	@Override
	public void deleteAssociation(final long associationId) throws TypeOrAssociationNotFoundException {
		if (!this.associationCacheId.containsKey(associationId)) {
			throw new TypeOrAssociationNotFoundException(associationId);
		}
		this.cacheAssociationName.remove(this.associationCacheId.remove(associationId));
	}
	
	@Override
	public void saveSpecialization(final Specialization specialization) {
		this.specializations.add(specialization);
	}
	
	@Override
	public Collection<Specialization> getAllSpecializations() {
		return new LinkedList<>(this.specializations);
	}
	
	@Override
	public void deleteSpecialization(final Specialization specialization) {
		this.specializations.remove(specialization);
	}
	
	@Override
	public boolean existsType(final long typeId) {
		return this.cacheId.containsKey(typeId);
	}
	
	@Override
	public boolean existsType(final String typeName) {
		return this.cacheName.containsKey(this.prepareName(typeName));
	}
	
	@Override
	public boolean existsAssociation(final long associationId) {
		return this.associationCacheId.containsKey(associationId);
	}
	
	@Override
	public boolean existsSpecialization(final long specializationId) {
		// TODO Auto-generated method stub
		return false;
	}
	
}
