package de.fhdw.gaming.contest.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Spliterator;
import java.util.function.Consumer;

/**
 * A split iterator generating combinations over an input list.
 * <p>
 *
 * @param <T> The type of items in the list.
 * @see <a href=
 *      "https://www.baeldung.com/java-combinations-algorithm">https://www.baeldung.com/java-combinations-algorithm</a>
 */
final class CombinationSplitIterator<T> implements Spliterator<List<T>> {

    /**
     * The list of items to build combinations upon.
     */
    private final List<T> input;
    /**
     * The last computed combination of indices pointing to the input list.
     */
    private final int[] combination;
    /**
     * The size of the combination to generate.
     */
    private final int sublistSize;
    /**
     * If {@code true}, combinations with repetition are computed, else combinations without repetition.
     */
    private final boolean withRepetition;
    /**
     * The total number of combinations to be computed.
     */
    private final long totalNumberOfCombinations;

    /**
     * Creates a {@link CombinationSplitIterator}.
     *
     * @param input          The input list.
     * @param sublistSize    The k of "n over k", i.e. the size of the combinations to generate.
     * @param withRepetition If {@code true}, combinations with repetition are computed, else combinations without
     *                       repetition.
     */
    CombinationSplitIterator(final List<T> input, final int sublistSize, final boolean withRepetition) {
        this.input = input;
        this.sublistSize = sublistSize;
        this.withRepetition = withRepetition;
        this.combination = new int[sublistSize];
        for (int i = 0; i < sublistSize; ++i) {
            this.combination[i] = withRepetition ? 0 : i;
        }
        this.totalNumberOfCombinations = withRepetition
                ? CombinatoricsHelper.nOverK(this.input.size() + this.sublistSize - 1, this.sublistSize)
                : CombinatoricsHelper.nOverK(this.input.size(), this.sublistSize);
    }

    @Override
    public boolean tryAdvance(final Consumer<? super List<T>> action) {
        if (this.combination[this.sublistSize - 1] >= this.input.size()) {
            return false;
        }

        action.accept(this.toList());
        this.computeNextCombination();
        return true;
    }

    @Override
    public Spliterator<List<T>> trySplit() {
        return null;
    }

    @Override
    public long estimateSize() {
        return this.totalNumberOfCombinations;
    }

    @Override
    public int characteristics() {
        return Spliterator.DISTINCT | Spliterator.NONNULL | Spliterator.SIZED | Spliterator.ORDERED;
    }

    /**
     * Maps a combination to a suitable (sub-)list of items and returns it.
     */
    private List<T> toList() {
        final List<T> result = new ArrayList<>();
        for (int i = 0; i < this.sublistSize; ++i) {
            result.add(this.input.get(this.combination[i]));
        }
        return result;
    }

    /**
     * Computes the next combination.
     */
    private void computeNextCombination() {
        int index = this.sublistSize - 1;
        while (index != 0 && this.combination[index] == this.input.size() - this.sublistSize
                + (this.withRepetition ? this.sublistSize - 1 : index)) {
            --index;
        }
        ++this.combination[index];
        for (int i = index + 1; i < this.sublistSize; ++i) {
            this.combination[i] = this.combination[i - 1] + (this.withRepetition ? 0 : 1);
        }
    }
}
