package de.fhdw.gaming.contest.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Spliterator;
import java.util.function.Consumer;

/**
 * A split iterator generating permutations over an input list.
 * <p>
 *
 * @param <T> The type of items in the list.
 * @see <a href=
 *      "https://www.geeksforgeeks.org/print-all-permutations-of-a-string-with-duplicates-allowed-in-input-string/">
 *      https://www.geeksforgeeks.org/print-all-permutations-of-a-string-with-duplicates-allowed-in-input-string/</a>
 */
final class PermutationSplitIterator<T> implements Spliterator<List<T>> {

    /**
     * The last computed permutation of the input list.
     */
    private final List<T> permutation;
    /**
     * Compares elements of the input list.
     */
    private final Comparator<T> comparator;
    /**
     * An upper limit of the number of permutations to be computed.
     */
    private final long upperLimitOfNumberOfPermutations;

    /**
     * Creates a {@link PermutationSplitIterator}.
     *
     * @param input      The input list.
     * @param comparator The comparator to use.
     */
    PermutationSplitIterator(final List<T> input, final Comparator<T> comparator) {
        this.permutation = new ArrayList<>(input);
        this.comparator = comparator;
        this.upperLimitOfNumberOfPermutations = CombinatoricsHelper.factorial(input.size());

        this.permutation.sort(comparator);
    }

    /**
     * Creates a {@link PermutationSplitIterator} for input with {@link Comparable} items.
     *
     * @param <T>   The type of items in the list
     * @param input The input list.
     * @return The {@link PermutationSplitIterator}.
     */
    static <T extends Comparable<T>> PermutationSplitIterator<T> create(final List<T> input) {
        return new PermutationSplitIterator<>(input, Comparable::compareTo);
    }

    @Override
    public boolean tryAdvance(final Consumer<? super List<T>> action) {
        action.accept(new ArrayList<>(this.permutation));
        return this.computeNextPermutation();
    }

    @Override
    public Spliterator<List<T>> trySplit() {
        return null;
    }

    @Override
    public long estimateSize() {
        return this.upperLimitOfNumberOfPermutations;
    }

    @Override
    public int characteristics() {
        return Spliterator.DISTINCT | Spliterator.NONNULL | Spliterator.SIZED | Spliterator.ORDERED;
    }

    /**
     * Computes the next permutation.
     */
    private boolean computeNextPermutation() {
        final Integer left = this.getIndexOfLastItemBeingSmallerThanRightNeighbour();
        if (left == null) {
            return false;
        }

        final Integer right = this.getIndexOfCeilingOfItem(this.permutation.get(left), left + 1);
        assert right != null;

        Collections.swap(this.permutation, left, right);
        this.permutation.subList(left + 1, this.permutation.size()).sort(this.comparator);
        return true;
    }

    /**
     * Returns the index of the last item that is smaller than its right neighbour item.
     */
    private Integer getIndexOfLastItemBeingSmallerThanRightNeighbour() {
        for (int index = this.permutation.size() - 2; index >= 0; --index) {
            if (this.comparator.compare(this.permutation.get(index), this.permutation.get(index + 1)) < 0) {
                return index;
            }
        }
        return null;
    }

    /**
     * Returns the index of some minimum item to the right of {@code left} that is greater than {@code left}.
     *
     * @param left       The left item to be compared to.
     * @param startIndex The index of the first item right to {@code left}.
     */
    private Integer getIndexOfCeilingOfItem(final T left, final int startIndex) {
        Integer result = null;
        for (int index = startIndex; index < this.permutation.size(); ++index) {
            final T right = this.permutation.get(index);
            if (this.comparator.compare(left, right) < 0
                    && (result == null || this.comparator.compare(right, this.permutation.get(result)) < 0)) {
                result = index;
            }
        }
        return result;
    }
}
