package de.fhdw.gaming.contest.util;

import java.util.Comparator;
import java.util.Objects;

/**
 * Represents a comparable pair of {@link Comparable}s. Pairs are ordered by lexicographical order.
 *
 * @param <T> The type of the first element.
 * @param <U> The type of the second element.
 */
public final class ComparablePair<T, U> implements Comparable<ComparablePair<T, U>> {

    /**
     * The first element.
     */
    private final T first;
    /**
     * The comparator for the first element.
     */
    private final Comparator<T> compFirst;
    /**
     * The second element.
     */
    private final U second;
    /**
     * The comparator for the second element.
     */
    private final Comparator<U> compSecond;

    /**
     * Constructor.
     *
     * @param first      The first element. {@code null} is <em>not</em> allowed.
     * @param compFirst  The comparator for the first element.
     * @param second     The second element. {@code null} is <em>not</em> allowed.
     * @param compSecond The comparator for the second element.
     */
    private ComparablePair(final T first, final Comparator<T> compFirst, final U second,
            final Comparator<U> compSecond) {
        this.first = first;
        this.second = second;
        this.compFirst = compFirst;
        this.compSecond = compSecond;
    }

    /**
     * Creates a {@link ComparablePair} from two {@link Comparable}s.
     *
     * @param <T>    The type of the first element.
     * @param <U>    The type of the second element.
     * @param first  The first element. {@code null} is <em>not</em> allowed.
     * @param second The second element. {@code null} is <em>not</em> allowed.
     * @return The {@link ComparablePair}.
     */
    public static <T extends Comparable<T>, U extends Comparable<U>> ComparablePair<T, U> of(final T first,
            final U second) {
        return new ComparablePair<>(first, Comparable::compareTo, second, Comparable::compareTo);
    }

    /**
     * Creates a {@link ComparablePair} from a {@link Comparable} and another element together with its
     * {@link Comparator}.
     *
     * @param <T>        The type of the first element.
     * @param <U>        The type of the second element.
     * @param first      The first element. {@code null} is <em>not</em> allowed.
     * @param second     The second element. {@code null} is <em>not</em> allowed.
     * @param compSecond The comparator for the second element.
     * @return The {@link ComparablePair}.
     */
    public static <T extends Comparable<T>, U> ComparablePair<T, U> of(final T first, final U second,
            final Comparator<U> compSecond) {
        return new ComparablePair<>(first, Comparable::compareTo, second, compSecond);
    }

    /**
     * Creates a {@link ComparablePair} from an element together with its {@link Comparator} and a {@link Comparable}.
     *
     * @param <T>       The type of the first element.
     * @param <U>       The type of the second element.
     * @param first     The first element. {@code null} is <em>not</em> allowed.
     * @param compFirst The comparator for the first element.
     * @param second    The second element. {@code null} is <em>not</em> allowed.
     * @return The {@link ComparablePair}.
     */
    public static <T, U extends Comparable<U>> ComparablePair<T, U> of(final T first, final Comparator<T> compFirst,
            final U second) {
        return new ComparablePair<>(first, compFirst, second, Comparable::compareTo);
    }

    /**
     * Creates a {@link ComparablePair} from two elements and two comparators. This allows to use elements that are not
     * of type {@link Comparable}, or to use an ordering that differs from {@link Comparable#compareTo(Object)}.
     *
     * @param <T>        The type of the first element.
     * @param <U>        The type of the second element.
     * @param first      The first element. {@code null} is <em>not</em> allowed.
     * @param compFirst  The comparator for the first element.
     * @param second     The second element. {@code null} is <em>not</em> allowed.
     * @param compSecond The comparator for the second element.
     * @return The {@link ComparablePair}.
     */
    public static <T, U> ComparablePair<T, U> of(final T first, final Comparator<T> compFirst, final U second,
            final Comparator<U> compSecond) {
        return new ComparablePair<>(first, compFirst, second, compSecond);
    }

    /**
     * Returns the first element.
     */
    public T getFirst() {
        return this.first;
    }

    /**
     * Returns the second element.
     */
    public U getSecond() {
        return this.second;
    }

    @Override
    public int compareTo(final ComparablePair<T, U> other) {
        final int compareFirst = this.compFirst.compare(this.first, other.first);
        return compareFirst != 0 ? compareFirst : this.compSecond.compare(this.second, other.second);
    }

    /**
     * {@inheritDoc}
     * <p>
     * {@link Object#equals(Object)} is used for comparing the first and second elements. The comparators do not
     * influence the comparison.
     */
    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof ComparablePair<?, ?>) {
            final ComparablePair<?, ?> other = (ComparablePair<?, ?>) obj;
            return this.first.equals(other.first) && this.second.equals(other.second);
        }
        return false;
    }

    /**
     * {@inheritDoc}
     * <p>
     * The comparators do not influence the resulting hash code.
     */
    @Override
    public int hashCode() {
        return Objects.hash(this.first, this.second);
    }

    @Override
    public String toString() {
        return String.format("(%s, %s)", this.first, this.second);
    }
}
