/*
 * Copyright © 2020 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of gaming-contest.
 *
 * Gaming-contest is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Gaming-contest is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * gaming-contest. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.contest.ui;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.charset.Charset;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.BooleanFieldType;
import de.fhdw.gaming.core.ui.type.FieldType;
import de.fhdw.gaming.core.ui.type.IntegerFieldType;
import de.fhdw.gaming.core.ui.type.ObjectFieldType;
import de.fhdw.gaming.core.ui.type.StringFieldType;
import de.fhdw.gaming.core.ui.type.validator.Validator;

/**
 * An interactive {@link InputProvider} using streams.
 */
public final class InteractiveStreamInputProvider implements InputProvider {

    /**
     * The {@link BufferedReader} used for input.
     */
    private final BufferedReader in;
    /**
     * The {@link PrintStream} used for output.
     */
    private final PrintStream out;
    /**
     * The needed data.
     */
    private final Map<String, NeededDataEntry> neededData;
    /**
     * The predefined data.
     */
    private final Map<String, Object> providedData;

    /**
     * Represents a needed data entry.
     */
    private static final class NeededDataEntry {

        /**
         * The prompt.
         */
        private final String prompt;
        /**
         * The field type.
         */
        private final FieldType<?> fieldType;

        /**
         * Creates a {@link NeededDataEntry}.
         *
         * @param prompt    The prompt.
         * @param fieldType The field type.
         */
        NeededDataEntry(final String prompt, final FieldType<?> fieldType) {
            this.prompt = prompt;
            this.fieldType = fieldType;
        }

        /**
         * Returns the prompt.
         */
        String getPrompt() {
            return this.prompt;
        }

        /**
         * Returns the field type.
         */
        FieldType<?> getFieldType() {
            return this.fieldType;
        }
    }

    /**
     * Creates a {@link InteractiveStreamInputProvider}.
     *
     * @param in  The {@link InputStream} used for input.
     * @param out The {@link OutputStream} used for output.
     */
    public InteractiveStreamInputProvider(final InputStream in, final OutputStream out) {
        this.in = new BufferedReader(new InputStreamReader(in, Charset.defaultCharset()));
        this.out = new PrintStream(out, true, Charset.defaultCharset());
        this.neededData = new LinkedHashMap<>();
        this.providedData = new LinkedHashMap<>();
    }

    /**
     * Creates a {@link InteractiveStreamInputProvider}.
     *
     * @param in  The {@link BufferedReader} used for input.
     * @param out The {@link PrintStream} used for output.
     */
    private InteractiveStreamInputProvider(final BufferedReader in, final PrintStream out) {
        this.in = in;
        this.out = out;
        this.neededData = new LinkedHashMap<>();
        this.providedData = new LinkedHashMap<>();
    }

    @Override
    @SafeVarargs
    public final InteractiveStreamInputProvider needString(final String id, final String prompt,
            final Optional<String> defaultValue, final Validator<String>... validators) {

        this.neededData.put(id, new NeededDataEntry(prompt, new StringFieldType(defaultValue).validateBy(validators)));
        return this;
    }

    @Override
    public InteractiveStreamInputProvider fixedString(final String id, final String fixedValue) {
        this.providedData.put(id, fixedValue);
        return this;
    }

    @Override
    @SafeVarargs
    public final InteractiveStreamInputProvider needInteger(final String id, final String prompt,
            final Optional<Integer> defaultValue, final Validator<Integer>... validators) {

        this.neededData.put(id, new NeededDataEntry(prompt, new IntegerFieldType(defaultValue).validateBy(validators)));
        return this;
    }

    @Override
    public InteractiveStreamInputProvider fixedInteger(final String id, final Integer fixedValue) {
        this.providedData.put(id, fixedValue);
        return this;
    }

    @Override
    @SafeVarargs
    public final InteractiveStreamInputProvider needBoolean(final String id, final String prompt,
            final Optional<Boolean> defaultValue, final Validator<Boolean>... validators) {

        this.neededData.put(id, new NeededDataEntry(prompt, new BooleanFieldType(defaultValue).validateBy(validators)));
        return this;
    }

    @Override
    public InteractiveStreamInputProvider fixedBoolean(final String id, final Boolean fixedValue) {
        this.providedData.put(id, fixedValue);
        return this;
    }

    @Override
    public InteractiveStreamInputProvider needObject(final String id, final String prompt,
            final Optional<Object> defaultValue, final Set<?> objectSet) {

        this.neededData.put(id, new NeededDataEntry(prompt, new ObjectFieldType(defaultValue, objectSet)));
        return this;
    }

    @Override
    public InteractiveStreamInputProvider fixedObject(final String id, final Object fixedValue) {
        this.providedData.put(id, fixedValue);
        return this;
    }

    @Override
    public Map<String, Object> requestData(final String title) throws InputProviderException {
        this.out.println(title);
        this.out.println("-".repeat(title.length()));

        final Map<String, Object> result = new LinkedHashMap<>();
        for (final Map.Entry<String, NeededDataEntry> entry : this.neededData.entrySet()) {
            result.put(entry.getKey(), this.requestField(entry.getKey(), entry.getValue()));
        }

        this.out.println();
        return result;
    }

    /**
     * Request data for a field.
     *
     * @param id              The ID of the field.
     * @param neededDataEntry The {@link NeededDataEntry} associated with the field.
     * @return The object read and validated.
     * @throws InputProviderException if reading the field fails for some reason.
     */
    private Object requestField(final String id, final NeededDataEntry neededDataEntry) throws InputProviderException {

        while (true) {
            final FieldType<?> fieldType = neededDataEntry.getFieldType();
            final Optional<String> validationInfo = fieldType.getInfo();
            this.out.printf(
                    "%s%s%s: ",
                    neededDataEntry.getPrompt(),
                    validationInfo.map((final String s) -> " {" + s + "}").orElse(""),
                    fieldType.getDefaultValue().map((final Object o) -> " [" + o + "]").orElse(""));

            if (this.providedData.containsKey(id)) {
                final Object value = this.providedData.get(id);
                this.out.println(value);
                return value;
            }

            fieldType.preInput(id);

            final String value = readLine(this.in);
            try {
                if (value.isEmpty() && fieldType.getDefaultValue().isPresent()) {
                    return fieldType.getDefaultValue().get();
                } else {
                    return fieldType.postInput(id, value.strip());
                }
            } catch (final InputProviderException e) {
                this.out.println(e.getMessage());
            }
        }
    }

    /**
     * Reads a single line.
     *
     * @param reader The {@link BufferedReader} to use.
     * @return The line.
     * @throws InputProviderException if reading the line fails for some reason.
     */
    private static String readLine(final BufferedReader reader) throws InputProviderException {
        final String value;
        try {
            value = reader.readLine();
        } catch (final IOException e) {
            throw new InputProviderException("Failed to read input: " + e.getMessage(), e);
        }

        if (value == null) {
            throw new InputProviderException("End of input.");
        }
        return value;
    }

    @Override
    public InputProvider getNext(final Map<String, Object> lastDataSet) {
        return new InteractiveStreamInputProvider(this.in, this.out);
    }
}
