/*
 * Copyright © 2020-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of gaming-gui.
 *
 * Gaming-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Gaming-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * gaming-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.gui;

import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.BooleanFieldType;
import de.fhdw.gaming.core.ui.type.FieldType;
import de.fhdw.gaming.core.ui.type.FieldTypeException;
import de.fhdw.gaming.core.ui.type.FieldTypeVisitor;
import de.fhdw.gaming.core.ui.type.FieldTypeWithValidator;
import de.fhdw.gaming.core.ui.type.IntegerFieldType;
import de.fhdw.gaming.core.ui.type.ObjectFieldType;
import de.fhdw.gaming.core.ui.type.StringFieldType;
import javafx.beans.value.ObservableValue;
import javafx.beans.value.WritableObjectValue;
import javafx.scene.Node;
import javafx.scene.control.ComboBox;
import javafx.scene.control.RadioButton;
import javafx.scene.control.TextField;
import javafx.scene.control.ToggleGroup;
import javafx.scene.control.Tooltip;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Region;

/**
 * Creates one or more controls for a given field type.
 */
final class ControlCreator implements FieldTypeVisitor {

    /**
     * The ID of the field.
     */
    private final String id;
    /**
     * The handler processing validation events.
     */
    private final Consumer<? super String> validationHandler;
    /**
     * The provided value.
     */
    private final Optional<Object> providedValue;
    /**
     * Stores the values of the fields.
     */
    private final Map<String, WritableObjectValue<Object>> values;
    /**
     * The {@link Region} created.
     */
    private Region result;
    /**
     * The {@link Node} that can receive focus.
     */
    private Node focusTarget;
    /**
     * The original field type.
     */
    private FieldType<?> originalFieldType;

    /**
     * Creates a {@link ControlCreator}.
     *
     * @param id                The ID of the field.
     * @param validationHandler The handler processing validation events.
     * @param providedValue     The provided value.
     * @param values            Stores the values of the fields.
     */
    ControlCreator(final String id, final Consumer<? super String> validationHandler,
            final Optional<Object> providedValue, final Map<String, WritableObjectValue<Object>> values) {
        this.id = id;
        this.validationHandler = validationHandler;
        this.providedValue = providedValue;
        this.values = values;
    }

    /**
     * Returns the {@link Region} created.
     */
    Region getResult() {
        return this.result;
    }

    /**
     * Returns the {@link Node} that can receive focus. This can be different from the {@link Region} returned by
     * {@link #getResult()} as the latter can be a node that cannot receive focus (like a pane).
     */
    Node getFocusTarget() {
        return this.focusTarget;
    }

    @Override
    public void handle(final StringFieldType fieldType) {
        if (this.originalFieldType == null) {
            this.originalFieldType = fieldType;
        }

        final TextField textField = new TextField();

        if (this.providedValue.isPresent()) {
            final String fixedValue = (String) this.providedValue.get();
            this.values.get(this.id).set(fixedValue);
            textField.setText(fixedValue);
            textField.setDisable(true);
        } else {
            fieldType.getDefaultValue().ifPresentOrElse((final String value) -> {
                textField.setText(value);
                this.setFieldValue(value);
            }, () -> {
                textField.setText("");
                this.setFieldValue("");
            });

            textField.textProperty().addListener(
                    (final ObservableValue<? extends String> observable, final String oldValue,
                            final String newValue) -> this.setFieldValue(newValue));
            this.originalFieldType.getInfo().ifPresent((final String info) -> textField.setTooltip(new Tooltip(info)));
        }

        this.result = textField;
        this.focusTarget = textField;
    }

    @Override
    public void handle(final IntegerFieldType fieldType) {
        if (this.originalFieldType == null) {
            this.originalFieldType = fieldType;
        }

        final TextField textField = new TextField();

        if (this.providedValue.isPresent()) {
            final String fixedValue = (String) this.providedValue.get();
            this.values.get(this.id).set(fixedValue);
            textField.setText(fixedValue);
            textField.setDisable(true);
        } else {
            fieldType.getDefaultValue().ifPresentOrElse((final Integer value) -> {
                textField.setText(value.toString());
                this.setFieldValue(value.toString());
            }, () -> {
                textField.setText("");
                this.setFieldValue("");
            });

            textField.textProperty().addListener(
                    (final ObservableValue<? extends String> observable, final String oldValue,
                            final String newValue) -> this.setFieldValue(newValue));
            this.originalFieldType.getInfo().ifPresent((final String info) -> textField.setTooltip(new Tooltip(info)));
        }

        this.result = textField;
        this.focusTarget = textField;
    }

    @Override
    public void handle(final BooleanFieldType fieldType) {
        if (this.originalFieldType == null) {
            this.originalFieldType = fieldType;
        }

        final ToggleGroup toggleGroup = new ToggleGroup();
        final RadioButton yesButton = new RadioButton("yes");
        yesButton.setToggleGroup(toggleGroup);
        final RadioButton noButton = new RadioButton("no");
        noButton.setToggleGroup(toggleGroup);

        if (this.providedValue.isPresent()) {
            final boolean fixedValue = (Boolean) this.providedValue.get();
            if (fixedValue) {
                yesButton.setSelected(true);
                this.focusTarget = yesButton;
            } else {
                noButton.setSelected(true);
                this.focusTarget = noButton;
            }

            this.values.get(this.id).set(fixedValue);
            yesButton.setDisable(true);
            noButton.setDisable(true);
        } else {
            fieldType.getDefaultValue().ifPresentOrElse((final Boolean value) -> {
                if (value) {
                    yesButton.setSelected(true);
                    this.setFieldValue("true");
                    this.focusTarget = yesButton;
                } else {
                    noButton.setSelected(true);
                    this.setFieldValue("false");
                    this.focusTarget = noButton;
                }
            }, () -> {
                this.setFieldValue("");
                this.focusTarget = yesButton;
            });

            yesButton.selectedProperty().addListener(
                    (final ObservableValue<? extends Boolean> observable, final Boolean oldValue,
                            final Boolean newValue) -> this.setFieldValue("true"));
            noButton.selectedProperty().addListener(
                    (final ObservableValue<? extends Boolean> observable, final Boolean oldValue,
                            final Boolean newValue) -> this.setFieldValue("false"));
        }

        final HBox hBox = new HBox(10);
        hBox.getChildren().addAll(yesButton, noButton);
        this.originalFieldType.getInfo().ifPresent((final String info) -> {
            yesButton.setTooltip(new Tooltip(info));
            noButton.setTooltip(new Tooltip(info));
        });
        this.result = hBox;
    }

    @Override
    public void handle(final ObjectFieldType fieldType) {
        if (this.originalFieldType == null) {
            this.originalFieldType = fieldType;
        }

        final ComboBox<String> comboBox = new ComboBox<>();
        for (final Object value : fieldType.getAllowedValues()) {
            comboBox.getItems().add(value.toString());
        }

        if (this.providedValue.isPresent()) {
            final Object fixedValue = this.providedValue.get();
            this.values.get(this.id).set(fixedValue);
            comboBox.setValue(fixedValue.toString());
            comboBox.setDisable(true);
        } else {
            fieldType.getDefaultValue().ifPresentOrElse((final Object value) -> {
                comboBox.setValue(value.toString());
                this.setFieldValue(value.toString());
            }, () -> {
                final String value = fieldType.getAllowedValues().size() == 1
                        ? fieldType.getAllowedValues().iterator().next().toString()
                        : "";
                comboBox.setValue(value);
                this.setFieldValue(value);
            });

            comboBox.valueProperty().addListener(
                    (final ObservableValue<? extends String> observable, final String oldValue,
                            final String newValue) -> this.setFieldValue(newValue));
            this.originalFieldType.getInfo().ifPresent((final String info) -> comboBox.setTooltip(new Tooltip(info)));
        }

        this.result = comboBox;
        this.focusTarget = comboBox;
    }

    @Override
    public void handle(final FieldTypeWithValidator<?> fieldType) throws FieldTypeException {
        if (this.originalFieldType == null) {
            this.originalFieldType = fieldType;
        }

        fieldType.getDelegatee().accept(this);
    }

    /**
     * Sets the value of a field given its string representation.
     *
     * @param newValue The new field value.
     */
    private void setFieldValue(final String newValue) {
        try {
            final Object value = this.originalFieldType.postInput(this.id, newValue);
            this.values.get(this.id).set(value);
            this.validationHandler.accept("");
        } catch (final InputProviderException e) {
            this.values.get(this.id).set(null);
            this.validationHandler.accept(e.getMessage());
        }
    }
}
