/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-freizeit.
 *
 * Ipspiel21-freizeit is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-freizeit is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied  * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-freizeit. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.freizeit.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitGameBuilder;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitGameBuilderFactory;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitPlayerBuilder;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitStrategy;
import de.fhdw.gaming.ipspiel21.freizeit.domain.factory.FreizeitDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel21.freizeit.domain.factory.FreizeitStrategyFactory;
import de.fhdw.gaming.ipspiel21.freizeit.domain.factory.FreizeitStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel21.freizeit.moves.factory.FreizeitMoveFactory;
import de.fhdw.gaming.ipspiel21.freizeit.moves.impl.AbstractFreizeitMove;
import de.fhdw.gaming.ipspiel21.freizeit.moves.impl.FreizeitCinemaMove;
import de.fhdw.gaming.ipspiel21.freizeit.moves.impl.FreizeitDefaultMoveFactory;
import de.fhdw.gaming.ipspiel21.freizeit.moves.impl.FreizeitSoccerMove;

/**
 * Implements {@link GameBuilderFactory} by creating a Freizeit game builder.
 */
public final class FreizeitGameBuilderFactoryImpl implements FreizeitGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Freizeit strategies.
     */
    private final Set<FreizeitStrategy> strategies;

    /**
     * Creates a Freizeit game factory. Freizeit strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public FreizeitGameBuilderFactoryImpl() {
        this(new FreizeitDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Freizeit game factory.
     *
     * @param strategyFactoryProvider The {@link FreizeitStrategyFactoryProvider} for loading Freizeit strategies.
     */
    FreizeitGameBuilderFactoryImpl(final FreizeitStrategyFactoryProvider strategyFactoryProvider) {
        final FreizeitMoveFactory moveFactory = new FreizeitDefaultMoveFactory();

        final List<FreizeitStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final FreizeitStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Freizeit";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return FreizeitGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return FreizeitGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @SuppressWarnings("exports")
    @Override
    public FreizeitGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final FreizeitGameBuilder gameBuilder = new FreizeitGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(FreizeitGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(FreizeitGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayer1Data(firstPlayerInputProvider, "Player 1");
            final FreizeitPlayerBuilder firstPlayerBuilder = this
                    .initPlayerBuilder(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final FreizeitStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayerBuilder(firstPlayerBuilder, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayer2Data(secondPlayerInputProvider, "Player 2");
            final FreizeitPlayerBuilder secondPlayerBuilder = this
                    .initPlayerBuilder(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final FreizeitStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayerBuilder(secondPlayerBuilder, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Freizeit game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayer1Data(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(
                        GameBuilderFactory.PARAM_PLAYER_NAME,
                        "Name",
                        Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(
                        FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_CINEMA,
                        "Player's outcome on Cinema/Cinema",
                        Optional.of(2))
                .needInteger(
                        FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_SOCCER,
                        "Player's outcome on Cinema/Soccer",
                        Optional.of(0))
                .needInteger(
                        FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SOCCER_CINEMA,
                        "Player's outcome on Soccer/Cinema",
                        Optional.of(0))
                .needInteger(
                        FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SOCCER_SOCCER,
                        "Player's outcome on Soccer/Soccer",
                        Optional.of(1))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayer2Data(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(
                        GameBuilderFactory.PARAM_PLAYER_NAME,
                        "Name",
                        Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(
                        FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_CINEMA,
                        "Player's outcome on Cinema/Cinema",
                        Optional.of(1))
                .needInteger(
                        FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_SOCCER,
                        "Player's outcome on Cinema/Soccer",
                        Optional.of(0))
                .needInteger(
                        FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SOCCER_CINEMA,
                        "Player's outcome on Soccer/Cinema",
                        Optional.of(0))
                .needInteger(
                        FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SOCCER_SOCCER,
                        "Player's outcome on Soccer/Soccer",
                        Optional.of(2))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Initialises a Freizeit player builder.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return {@code playerBuilder}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private FreizeitPlayerBuilder initPlayerBuilder(final FreizeitPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<AbstractFreizeitMove, Map<AbstractFreizeitMove, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<AbstractFreizeitMove, Double> possibleOutcomesCinema = new LinkedHashMap<>();
        possibleOutcomesCinema.put(
                new FreizeitCinemaMove(),
                (double) (Integer) playerData.get(FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_CINEMA));
        possibleOutcomesCinema.put(
                new FreizeitSoccerMove(),
                (double) (Integer) playerData.get(FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_SOCCER));
        possibleOutcomes.put(new FreizeitCinemaMove(), possibleOutcomesCinema);

        final Map<AbstractFreizeitMove, Double> possibleOutcomesSoccer = new LinkedHashMap<>();
        possibleOutcomesSoccer.put(
                new FreizeitCinemaMove(),
                (double) (Integer) playerData.get(FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SOCCER_CINEMA));
        possibleOutcomesSoccer.put(
                new FreizeitSoccerMove(),
                (double) (Integer) playerData.get(FreizeitGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SOCCER_SOCCER));
        possibleOutcomes.put(new FreizeitSoccerMove(), possibleOutcomesSoccer);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes);
    }

    /**
     * Returns a Freizeit strategy.
     *
     * @param playerData The requested player data.
     * @return The Freizeit strategy.
     */
    private FreizeitStrategy getStrategy(final Map<String, Object> playerData) {
        return (FreizeitStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
