package de.fhdw.gaming.ipspiel21.searchtrees.domain.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.State;
import de.fhdw.gaming.ipspiel21.searchtrees.domain.Evaluation;
import de.fhdw.gaming.ipspiel21.searchtrees.domain.NextMoveGenerator;
import de.fhdw.gaming.ipspiel21.searchtrees.domain.SearchTreeAlgorithms;

/**
 * Implements {@link SearchTreeAlgorithms} by implementation of the Alpha-Beta algorithm.
 *
 * @param <P> the type of the player.
 * @param <S> the type of the state.
 * @param <M> the type of the move.
 */
public class AlphaBetaImpl<P extends Player, S extends State<P, S>, M extends Move<P, S>>
        implements SearchTreeAlgorithms<P, S, M> {

    @Override
    public Optional<M> getBestMove(final int depth, final P currentPlayer, final NextMoveGenerator<P, S, M> generator,
            final S state, final Evaluation<P, S> evaluator) throws GameException {
        final P otherPlayer = state.getPlayers().values().stream().filter((final P p) -> !p.equals(currentPlayer))
                .findAny().get();
        final Collection<MoveEvaluationPair<P, S, M>> moveBewertungen = new ArrayList<>();
        final Collection<M> allPossibleMoves = generator.getAllNextMoves(state, currentPlayer);
        for (final M move : allPossibleMoves) {
            final S stateCopy = state.deepCopy();
            move.applyTo(stateCopy, currentPlayer);
            stateCopy.nextTurn();
            moveBewertungen.add(
                    new MoveEvaluationPair<>(
                            move,
                            -this.alphaBetaScore(
                                    depth - 1,
                                    Double.NEGATIVE_INFINITY,
                                    Double.POSITIVE_INFINITY,
                                    otherPlayer,
                                    currentPlayer,
                                    generator,
                                    stateCopy,
                                    evaluator)));
        }
        return moveBewertungen.stream().max((p1, p2) -> p1.getEvaluation().compareTo(p2.getEvaluation()))
                .map(MoveEvaluationPair::getMove);
    }

    /**
     * The maximum Score of the maximizing player.
     *
     * @param depth         The depth of search tree.
     * @param alpha         The smallest score of the player.
     * @param beta          The beta biggest score of the player.
     * @param currentPlayer The player who has to do the next move.
     * @param otherPlayer   The other player who has done the last move.
     * @param generator     The generator that creates next moves.
     * @param state         The current state.
     * @param evaluator     The abstract evaluator.
     * @return The maximizing score that represents the best move of maximizing player.
     * @throws GameException
     */
    public Double alphaBetaScore(final int depth, final Double alpha, final Double beta, final P currentPlayer,
            final P otherPlayer, final NextMoveGenerator<P, S, M> generator, final S state,
            final Evaluation<P, S> evaluator) throws GameException {
        final Collection<M> allNextMoves = generator.getAllNextMoves(state, currentPlayer);
        if (depth == 0 || allNextMoves.isEmpty()) {
            return evaluator.evaluate(currentPlayer, state);
        }
        Double maxVal = alpha;
        final Iterator<M> it = allNextMoves.iterator();
        while (it.hasNext()) {
            final Move<P, S> currentMove = it.next();
            final S stateCopy = state.deepCopy();
            currentMove.applyTo(stateCopy, currentPlayer);
            stateCopy.nextTurn();
            final Double val = -this.alphaBetaScore(
                    depth - 1,
                    -beta,
                    -maxVal,
                    otherPlayer,
                    currentPlayer,
                    generator,
                    stateCopy,
                    evaluator);
            if (val > maxVal) {
                maxVal = val;
                if (maxVal >= beta) {
                    break;
                }
            }

        }
        return maxVal;
    }
}
