/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-dilemma.
 *
 * Ipspiel21-dil is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-dil is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-dilemma. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaGameBuilder;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaGameBuilderFactory;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaPlayerBuilder;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaStrategy;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.factory.DilemmaDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.factory.DilemmaStrategyFactory;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.factory.DilemmaStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.factory.DilemmaMoveFactory;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.AbstractDilemmaMove;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.DilemmaBeSilentMove;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.DilemmaConfessMove;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.DilemmaDefaultMoveFactory;

/**
 * Implements {@link GameBuilderFactory} by creating a dilemma game builder.
 */
public abstract class DilemmaGameBuilderFactoryImpl implements DilemmaGameBuilderFactory { // CustomGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;
    /**
     * All available dilemma strategies.
     */
    private final Set<DilemmaStrategy> strategies;
    /**
     * Strategy of first player.
     */
    private DilemmaStrategy actualStrategyPlayer1;

    /**
     * Strategy of second player.
     */
    private DilemmaStrategy actualStrategyPlayer2;

    /**
     * Creates a dilemma game factory. dilemma strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public DilemmaGameBuilderFactoryImpl() {
        this(new DilemmaDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a dilemma game factory.
     *
     * @param strategyFactoryProvider The {@link dilemmaStrategyFactoryProvider} for loading dilemma strategies.
     */
    DilemmaGameBuilderFactoryImpl(final DilemmaStrategyFactoryProvider strategyFactoryProvider) {
        final DilemmaMoveFactory moveFactory = new DilemmaDefaultMoveFactory();

        final List<DilemmaStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final DilemmaStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Dilemma";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return DilemmaGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return DilemmaGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public DilemmaGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final DilemmaGameBuilder gameBuilder = new DilemmaGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String, Object> gameData = inputProvider.needInteger(
                    GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                    "Maximum computation time per move in seconds",
                    Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MinValueValidator<>(DilemmaGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MaxValueValidator<>(DilemmaGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                    .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1");
            final DilemmaPlayerBuilder firstPlayerBuilder = this.initPlayerBuilder(gameBuilder.createPlayerBuilder(),
                    firstPlayerData);
            final DilemmaStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            this.actualStrategyPlayer1 = firstPlayerStrategy;
            gameBuilder.addPlayerBuilder(firstPlayerBuilder, this.actualStrategyPlayer1);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2");
            final DilemmaPlayerBuilder secondPlayerBuilder = this.initPlayerBuilder(gameBuilder.createPlayerBuilder(),
                    secondPlayerData);
            final DilemmaStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            this.actualStrategyPlayer2 = secondPlayerStrategy;
            gameBuilder.addPlayerBuilder(secondPlayerBuilder, this.actualStrategyPlayer2);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating dilemma game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns the Strategy of first player.
     *
     * @return
     */
    public DilemmaStrategy getActualStrategyPlayer1() {
        return this.actualStrategyPlayer1;
    }

    /**
     * Returns the Strategy of second player.
     *
     * @return
     */
    public DilemmaStrategy getActualStrategyPlayer2() {
        return this.actualStrategyPlayer2;
    }

    /**
     * Sets the strategy of first player.
     *
     * @param actualStrategyPlayer1
     */
    public void setActualStrategyPlayer1(final DilemmaStrategy actualStrategyPlayer1) {
        this.actualStrategyPlayer1 = actualStrategyPlayer1;
    }

    /**
     * Sets the strategy of second player.
     *
     * @param actualStrategyPlayer2
     */
    public void setActualStrategyPlayer2(final DilemmaStrategy actualStrategyPlayer2) {
        this.actualStrategyPlayer2 = actualStrategyPlayer2;
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely
     *
     *                                (e.g. if the user cancelled a dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(GameBuilderFactory.PARAM_PLAYER_NAME, "Name", Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(DilemmaGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_NO_NO,
                        "Player's outcome on Both Silent", Optional.of(-1))
                .needInteger(DilemmaGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_NO_YES,
                        "Player's outcome on SecondConfess", Optional.of(0))
                .needInteger(DilemmaGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_YES_NO,
                        "Player's outcome on FirstConfess", Optional.of(-10))
                .needInteger(DilemmaGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_YES_YES,
                        "Player's outcome on BothConfess", Optional.of(-8))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(),
                        this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Initializes a dilemma player builder.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return {@code playerBuilder}.
     * @throws InputProviderException if the operation has been aborted prematurely
     *
     *                                (e.g. if the user cancelled a dialog).
     */
    private DilemmaPlayerBuilder initPlayerBuilder(final DilemmaPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<AbstractDilemmaMove, Double> possibleOutcomesNo = new LinkedHashMap<>();
        possibleOutcomesNo.put(new DilemmaConfessMove(),
                (double) (Integer) playerData.get(DilemmaGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_NO_NO));
        possibleOutcomesNo.put(new DilemmaBeSilentMove(),
                (double) (Integer) playerData.get(DilemmaGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_NO_YES));
        possibleOutcomes.put(new DilemmaConfessMove(), possibleOutcomesNo);

        final Map<AbstractDilemmaMove, Double> possibleOutcomesYes = new LinkedHashMap<>();
        possibleOutcomesYes.put(new DilemmaConfessMove(),
                (double) (Integer) playerData.get(DilemmaGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_YES_NO));
        possibleOutcomesYes.put(new DilemmaBeSilentMove(),
                (double) (Integer) playerData.get(DilemmaGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_YES_YES));
        possibleOutcomes.put(new DilemmaBeSilentMove(), possibleOutcomesYes);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes);
    }

    /**
     * Returns a dilemma strategy.
     *
     * @param playerData The requested player data.
     * @return The dilemma strategy.
     */
    private DilemmaStrategy getStrategy(final Map<String, Object> playerData) {
        return (DilemmaStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }

    /**
     * Returns the decorated {@link InputProvider}.
     *
     * @param inputProvider The input provider to use (if necessary) in order to
     *                      determine mandatory or optional parameters for creating
     *                      the game. Mandatory parameters are the players' names
     *                      and strategies. All other parameters need to have
     *                      meaningful default values.
     *                      <p>
     *                      The order of the data sets request is as follows:
     *                      <ol>
     *                      <li>Parameters for the game as a whole</li>
     *                      <li>Parameters for player 1</li>
     *                      <li>Parameters for player 2</li>
     *                      <li>Parameters for player ...</li>
     *                      </ol>
     * @return An extended {@link InputProvider}.
     * @throws InputProviderException.
     */
    @Override
    public abstract InputProvider extendInputProvider(InputProvider inputProvider) throws InputProviderException;
}
