
package de.fhdw.gaming.ipspiel21.viergewinnt.gui.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntBoard;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntField;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntPlayer;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntPosition;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntState;
import de.fhdw.gaming.ipspiel21.viergewinnt.gui.VierGewinntBoardEventProvider;
import de.fhdw.gaming.ipspiel21.viergewinnt.gui.event.VierGewinntBoardEvent;
import de.fhdw.gaming.ipspiel21.viergewinnt.gui.event.VierGewinntMakeMoveBoardEvent;
import javafx.application.Platform;
import javafx.scene.Cursor;
import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

/**
 * Implements user interaction with a VierGewinnt board.
 */
final class VierGewinntBoardEventProviderImpl implements VierGewinntBoardEventProvider {

    /**
     * The associated {@link VierGewinntBoardView}.
     */
    private final VierGewinntBoardView boardView;

    /**
     * Constructor.
     *
     * @param boardView The associated {@link VierGewinntBoardView}.
     */
    VierGewinntBoardEventProviderImpl(final VierGewinntBoardView boardView) {
        this.boardView = boardView;
    }

    @Override
    public VierGewinntBoardEvent waitForEvent(final VierGewinntPlayer player, final VierGewinntState state) {
        final Map<VierGewinntPosition, VierGewinntFieldView> possibleFieldViews = new LinkedHashMap<>();
        for (final VierGewinntField entry : state.getBoard()
                .getAllPlayableFields()) {
            possibleFieldViews.put(entry.getPosition(), this.boardView.getFieldView(entry.getPosition()).orElseThrow());
        }

        final AtomicReference<VierGewinntBoardEvent> event = new AtomicReference<>();
        final Runnable cleanUp;
        if (possibleFieldViews.isEmpty()) {
            final List<VierGewinntFieldView> fieldViews = this.setupInactiveFields(state.getBoard());
            cleanUp = () -> this.cleanUpFields(fieldViews);
        } else {
            this.setupActiveFields(possibleFieldViews, event);
            cleanUp = () -> this.cleanUpFields(possibleFieldViews.values());
        }

        try {
            this.boardView.getUserInputSemaphore().acquire();
            return event.get();
        } catch (final InterruptedException e) {
            return null;
        } finally {
            Platform.runLater(cleanUp);
        }
    }

    /**
     * Sets up the fields when a move is possible (and hence the set of empty fields is not empty).
     *
     * @param emptyFieldViews The non-empty set of views for active fields.
     * @param event           The event to be set when the user selects an active field.
     */
    private void setupActiveFields(final Map<VierGewinntPosition, VierGewinntFieldView> emptyFieldViews,
            final AtomicReference<VierGewinntBoardEvent> event) {
        for (final Map.Entry<VierGewinntPosition, VierGewinntFieldView> entry : emptyFieldViews.entrySet()) {
            final VierGewinntPosition position = entry.getKey();
            final VierGewinntFieldView fieldView = entry.getValue();
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CROSSHAIR);
                fieldView.setOnMouseClicked((final MouseEvent mouseEvent) -> {
                    if (mouseEvent.getButton() == MouseButton.PRIMARY) {
                        event.set(new VierGewinntMakeMoveBoardEvent(position));
                        this.boardView.getUserInputSemaphore().release();
                    }
                });
            });
        }
    }

    /**
     * Sets up the fields when no move is possible (and hence the set of empty fields is empty).
     *
     * @param board The Vier gewinnt board.
     * @return A list of all field views on the board.
     */
    private List<VierGewinntFieldView> setupInactiveFields(final VierGewinntBoard board) {
        final List<VierGewinntField> fields = new ArrayList<>();
       
        board.getFields().forEach(
             (final VierGewinntField field) -> fields.add(field));

        final List<VierGewinntFieldView> fieldViews = new ArrayList<>();
        for (final VierGewinntField field : fields) {
            final VierGewinntFieldView fieldView = this.boardView.getFieldView(field.getPosition()).orElseThrow();
            fieldViews.add(fieldView);
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CLOSED_HAND);
                fieldView.setOnMouseClicked(null);
            });
        }

        return fieldViews;
    }

    /**
     * Cleans up after user interaction by resetting mouse cursors and removing event handlers.
     *
     * @param fieldViews The modified field views.
     */
    private void cleanUpFields(final Collection<? extends VierGewinntFieldView> fieldViews) {
        for (final VierGewinntFieldView fieldView : fieldViews) {
            fieldView.setCursor(Cursor.DEFAULT);
            fieldView.setOnMouseClicked(null);
        }
    }

    @Override
    public void cancelWaiting() {
        this.boardView.getUserInputSemaphore().release();
    }
}
