/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-Dilemma.
 *
 * Ipspiel21-Dil is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-Dil is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-Dilemma. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaPlayer;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaState;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.AbstractDilemmaMove;
import de.fhdw.gaming.ipspiel21.evolution.EvolutionPlayerImpl;

/**
 * Implements {@link DilemmaPlayer}.
 */
public final class DilemmaPlayerImpl extends EvolutionPlayerImpl implements DilemmaPlayer {

    /**
     * The associated game state.
     */
    private final DilemmaState gameState;
    /**
     * The name of this player.
     */
    private final String name;

    /**
     * The associated DilemmaMove.
     */
    private AbstractDilemmaMove move;

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>> possibleOutcomes;

    /**
     * Creates a Dilemma player.
     *
     * @param gameState        The associated game state.
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    DilemmaPlayerImpl(final DilemmaState gameState, final String name,
            final Map<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>> possibleOutcomes) {
        super();
        this.gameState = gameState;

        this.name = Objects.requireNonNull(name, "name");
        this.possibleOutcomes = Collections
                .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
    }

    @Override
    public String toString() {
        return String.format("DilemmaPlayer[name=%s, state=%s, outcome=%s, move=%s]", this.name, this.getState(),
                this.getOutcome(), this.move);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof DilemmaPlayerImpl) {
            final DilemmaPlayerImpl other = (DilemmaPlayerImpl) obj;
            return this.name.equals(other.name) && this.getState().equals(other.getState())
                    && this.getMove().equals(other.getMove()) && this.getOutcome().equals(other.getOutcome());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return this.name.hashCode();
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public Map<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public PlayerState getState() {
        return this.gameState.getPlayerState(this.name);
    }

    @Override
    public Optional<Double> getOutcome() {
        return this.gameState.getPlayerOutcome(this.name);
    }

    @Override
    public DilemmaPlayer deepCopy(final DilemmaState newGameState) {
        return new DilemmaPlayerImpl(newGameState, this.name, this.possibleOutcomes);
    }

    @Override
    public AbstractDilemmaMove getMove() {
        return move;
    }

    @Override
    public void setMove(final AbstractDilemmaMove move) {
        this.move = move;
    }
}
